/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */

/*
 * vdw.h
 * 
 * van der Waals interaction module 
 *
 * the real van der Waals potential is a C^1 function
 *
 *                  V(r)          r < r_s
 *      realV(r) =  V(r) * S(r)   r_s < r < r_c
 *                  0             r_c < r
 *
 * where V(r) is the real van der Waals interaction:
 *
 *                        [  sigma         sigma   ]
 *      V(r) = 4 \epsilon [ (-----)^12 - (-----)^6 ]
 *                        [    r             r     ]
 *
 * S(r) is the switching function:
 *
 *              (r_c^2 - r^2)^2 * (r_c^2 + 2*r^2 - 3*r_s^2)
 *      S(r) = ---------------------------------------------
 *                           (r_c^2 - r_s^2)^3
 * 
 * and r_s is the switch distance, r_c is the cutoff distance.
 * S(r) is a function of r^2, and 
 * 
 *      S(r_s) = 1,  S'(r_s) = 0
 *      S(r_c) = 0,  S'(r_c) = 0
 *
 * In implementation,
 *
 *              A_term       B_term
 *      V(r) = --------  -  --------
 *               r^12          r^6
 *
 * where A_term and B_term is stored in "vdwtable", provided from outside.
 *
 */

#ifndef VDW_H
#define VDW_H

#ifdef __cplusplus
extern "C" {
#endif

#include "mdtypes.h"
#include "linkcell.h"

  /*
   * offsets for vdwtable entry
   */
#undef   ATERM
#define  ATERM     0
#undef   BTERM
#define  BTERM     1
#undef   A_14
#define  A_14      2
#undef   B_14
#define  B_14      3


  /*
   * make cells of atom IDs for geometric hashing of atoms
   */
#define  NUM_NBRS     13
#define  MIN_ATOMIDS  10


  struct Fnonbond_Tag; 

  struct VDW_init_Tag 
  {
    struct Data_Tag *data;
    MD_Dvec *ppos;
    MD_Atom *patom;
    MD_Int **pexcllist;
    MD_Int **pscaled14;
    MD_Int natomprms;
    MD_Int natoms;
    MD_Dvec systemsize;
    MD_Double cutoff;
    MD_Int is_switching;
    MD_Double switch_dist;
  };

  struct VDW_Tag 
  {
    MD_Dvec *force;       /* own force array */
    MD_Dvec systemsize;   /* length of the box */
    MD_Double energy;     /* total energy from vdw interaction */
    MD_Int natoms;        /* # of atoms */
    const MD_Atom *patom; /* needed to retrive vdw parameter */
    MD_Int **pexcllist;  /* pointers to the exclusion list */
    MD_Int **pscaled14;  /* pointers to the scaled-14 list */
    MD_Int natomprms;  

    /*
     * table for van der Waals force constants
     *
     * there are (4 * natomprms^2) entries in table that store
     * A, B, A_14, B_14 for each pairwise interaction type
     *
     * for an interaction between atoms itype and jtype,
     * table entry is:  4 * (itype * natomprms + jtype)
     *
     * the table is built during rpol_force_init
     */
    MD_Double *vdwtable;
    const MD_Dvec *ppos; /* pointer to my position */
    /*
     * vdw force constants
     */
    MD_Double cutoff, cutoff2, inv_cutoff2;
    MD_Double switchdist, switchdist2;
    MD_Double sw_denom;
    MD_Double sw_denom_times_four;
    MD_Int is_switching;

    struct LinkCell_Tag *linkcell;
  };


  /*
   * functions, interface.
   */
  MD_Errcode vdw_init(struct VDW_Tag* vdw, struct VDW_init_Tag *init_data);
  MD_Errcode vdw_destroy(struct VDW_Tag *vdw);
  MD_Errcode vdw_compute(struct VDW_Tag *vdw);
  MD_Double vdw_get_energy(const struct VDW_Tag *vdw);
  const MD_Dvec*  vdw_get_force(const struct VDW_Tag *vdw);

#ifdef __cplusplus
}
#endif

#endif /* VDW_H */
