/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <stdlib.h>
#include "mdtypes.h"
#include "constant.h"
#include "utilities.h"
#include "shake.h"


void shake_init(struct Shake_Water *sw, 
		const MD_Double bondOH, const MD_Double angle_HOH, 
		const MD_Double massO, const MD_Double massH,
		const MD_Int natoms)
{
  assert(NULL != sw);
  sw->bondOH = bondOH;
  sw->angle_HOH = angle_HOH;
  sw->bondHH = sqrt(sw->bondOH * sw->bondOH * 2.0 - 
		    2.0 * sw->bondOH * sw->bondOH * cos(sw->angle_HOH));
  sw->massO = massO;
  sw->massH = massH;
  sw->natoms = natoms;
  sw->old_pos = malloc(sw->natoms * sizeof(MD_Dvec));
  assert(NULL != sw->old_pos);
  sw->errTol2 = 1e-14;
  sw->bond_errTol = 1e-8;  /* should be this accurate to avoid energy drift */
  printf("Use shake method to contraint water molecule:\n");
  printf("  bond error tolerance is: %g\n", sw->bond_errTol);

}


void shake_destroy(struct Shake_Water *sw) 
{
  if (NULL != sw->old_pos) free(sw->old_pos);
  memset(sw, 0, sizeof(struct Shake_Water));
}


void shake_prepare(struct Shake_Water *sw, const MD_Dvec *old_pos)
{
  memcpy(sw->old_pos, old_pos, sw->natoms * sizeof(MD_Dvec));
}


/* 
 * use shake method to do constraint dynamics in velocity verlet method.
 * input:  
 *        rold:  positon array (of a water molecule) at t
 *        rnew:  positon array at t + dt (not yet obeying constaint)
 * output:  
 *        rnew: modified positon at t + dt, satisfying constraint
 *        vel:  velocity is modified according to the force.
 * be careful about the periodic boundary condition.  
*/

/* something might be wrong in math.h for abs function */
MD_Errcode shake(struct Shake_Water *sw, const MD_Int iatom, 
		 MD_Dvec *rnew, MD_Dvec *vel,  const MD_Double dt)
{
  /* rigid water model parameters. d1 = d(O, H1) = d(O, H2),
   * d2 = d(H1, H2).  theta = angle(OH1, OH2). */
  const MD_Dvec *rold = sw->old_pos + iatom;
  const MD_Double errTol = sw->bond_errTol; /* bond length */
  const MD_Double errTol2 =  sw->errTol2;
  const MD_Double d1 = sw->bondOH;   /* angstron */ 
  const MD_Double theta = sw->angle_HOH;  /* 109.5 degree */
  const MD_Double phi = (Pi - theta) * 0.5; 
  const MD_Double d2 = sw->bondHH;
  const MD_Double d1sqr = d1 * d1;
  const MD_Double d2sqr = d2 * d2;
  const MD_Double d1d2cosphi = d1 * d2 * cos(phi);
  const MD_Double d1d1costheta = d1 * d1 * cos(theta);
  const MD_Double mo = sw->massO;
  const MD_Double mh = sw->massH;
  const MD_Double po = 2.0 * dt * dt / mo;
  const MD_Double ph = 2.0 * dt * dt / mh;
  const MD_Double poph = po+ph;
  const MD_Double posqr = po*po;
  const MD_Double phsqr = ph*ph;
  const MD_Double pophsqr = poph * poph;

  const MD_Dvec *r0 = rold;
  const MD_Dvec *r1 = rold + 1;
  const MD_Dvec *r2 = rold + 2;
  MD_Dvec *ro = rnew;
  MD_Dvec *rh1 = rnew + 1;
  MD_Dvec *rh2 = rnew + 2;
  MD_Dvec roh1, roh2, rh1h2;
  MD_Dvec r01, r02, r12;
  MD_Dvec *velo, *velh1, *velh2;
  MD_Dvec deltav;

  MD_Double c11, c12, c13, c21, c22, c23, c31, c32, c33;
  MD_Double lmd01 = 0.0, lmd02 = 0.0, lmd12 = 0.0;
  MD_Double lmd01o, lmd02o, lmd12o;
  MD_Double rhs1, rhs2, rhs3, inv_det;
  MD_Double err01, err02, err12;
  MD_Double dlmd01, dlmd02, dlmd12;
  MD_Double lmd01sqr, lmd02sqr, lmd12sqr, tmp1, tmp2;
  MD_Int iter;
  MD_Double doh1sqr, doh2sqr, dh1h2sqr;

#ifdef DEBUG_SHAKE
  printf("before:\n");
  printf(" O: (%f,%f, %f)\n", ro->x, ro->y, ro->z);
  printf("H1: (%f,%f, %f)\n", rh1->x, rh1->y, rh1->z);
  printf("H2: (%f,%f, %f)\n", rh2->x, rh2->y, rh2->z);
#endif

  MD_pvec_substract(ro, rh1, &roh1);
  MD_pvec_substract(ro, rh2, &roh2);
  MD_pvec_substract(rh1, rh2, &rh1h2);

  MD_pvec_substract(r0, r1, &r01);
  MD_pvec_substract(r0, r2, &r02);
  MD_pvec_substract(r1, r2, &r12);

  c11 =   poph * MD_vec_dot(roh1,  r01);
  c12 =     po * MD_vec_dot(roh1,  r02);
  c13 =   - ph * MD_vec_dot(roh1,  r12);
  c21 =     po * MD_vec_dot(roh2,  r01);
  c22 =   poph * MD_vec_dot(roh2,  r02);
  c23 =     ph * MD_vec_dot(roh2,  r12);
  c31 =   - ph * MD_vec_dot(rh1h2, r01);
  c32 =   + ph * MD_vec_dot(rh1h2, r02);
  c33 = 2.0*ph * MD_vec_dot(rh1h2, r12);

  inv_det = 1.0 /  (c11*c22*c33 + c21*c32*c13 + c31*c12*c23
		  - c13*c22*c31 - c23*c32*c11 - c33*c12*c21);

#ifdef DEBUG_SHAKE
  {
    MD_Double t1, t2, t3, t, t_inv;
    MD_Double d11, d12, d13, d21, d22, d23, d31, d32, d33;
    MD_Double cond;
    /* compute 1-norm of matrix {c_ij}. */
    t1 = fabs(c11) + fabs(c12) + fabs(c13);
    t2 = fabs(c21) + fabs(c22) + fabs(c23);
    t3 = fabs(c31) + fabs(c32) + fabs(c33);
    t = t1 > t2 ? t1:t2;
    t = t > t3 ? t:t3;
    /* compute {c_ij}^{-1} */
    d11 =  (c22*c33 - c23*c32) * inv_det;
    d12 = -(c21*c33 - c23*c31) * inv_det;
    d13 =  (c21*c32 - c22*c31) * inv_det;
    d21 = -(c12*c33 - c13*c32) * inv_det;
    d22 =  (c11*c33 - c13*c31) * inv_det;
    d23 = -(c11*c32 - c12*c31) * inv_det;
    d31 =  (c12*c23 - c22*c13) * inv_det;
    d32 = -(c11*c23 - c13*c21) * inv_det;
    d33 =  (c11*c22 - c12*c21) * inv_det;
    /* compute 1-norm of matrix {c_ij}^{-1} */
    t1 = fabs(d11) + fabs(d12) + fabs(d13);
    t2 = fabs(d21) + fabs(d22) + fabs(d23);
    t3 = fabs(d31) + fabs(d32) + fabs(d33);
    t_inv = t1 > t2 ? t1:t2;
    t_inv = t_inv > t3 ? t_inv:t3;
    cond = t * t_inv;
    printf("inv_det=%f\n", inv_det);
    printf("condition number is: %f\n", cond);
  }
#endif

  err01 = d1sqr - MD_vec_dot(roh1,  roh1);
  err02 = d1sqr - MD_vec_dot(roh2,  roh2);
  err12 = d2sqr - MD_vec_dot(rh1h2, rh1h2);
  iter = 0;
  do {   /* large iteration loop */
    lmd01o = lmd01;
    lmd02o = lmd02;
    lmd12o = lmd12;
    /* compute right hand side */
    lmd01sqr = lmd01*lmd01;
    lmd02sqr = lmd02*lmd02;
    lmd12sqr = lmd12*lmd12;
    tmp1 = lmd01 * lmd02 * d1d1costheta;
    tmp2 = lmd12 * d1d2cosphi;

    rhs1 = 0.5 * ( err01 - (pophsqr*lmd01sqr + posqr*lmd02sqr)*d1sqr
	         - phsqr*lmd12sqr*d2sqr )
         - po*poph*tmp1  - (poph*lmd01 - po*lmd02)*ph*tmp2;
    rhs2 = 0.5 * ( err02 - (posqr*lmd01sqr + pophsqr*lmd02sqr)*d1sqr
	         - phsqr*lmd12sqr*d2sqr )
	 - po*poph*tmp1 + (po*lmd01 - poph*lmd02)*ph*tmp2;
    rhs3 = 0.5 * ( err12 - phsqr * ((lmd01sqr + lmd02sqr)*d1sqr
	       		     + 4.0*lmd12sqr*d2sqr) )
	 + phsqr*(tmp1 - 2.0*(lmd01+lmd02)*tmp2);
    /* solve cij*lmdj = rhsi */
    lmd01 = (  rhs1*c22*c33 + rhs2*c32*c13 + rhs3*c12*c23
	     - c13*c22*rhs3 - c23*c32*rhs1 - c33*c12*rhs2) 
             * inv_det;
    lmd02 = (  c11*rhs2*c33 + c21*rhs3*c13 + c31*rhs1*c23
	     - c13*rhs2*c31 - c23*rhs3*c11 - c33*rhs1*c21) 
             * inv_det;
    lmd12 = (  c11*c22*rhs3 + c21*c32*rhs1 + c31*c12*rhs2
	     - rhs1*c22*c31 - rhs2*c32*c11 - rhs3*c12*c21) 
             * inv_det;
    dlmd01 = lmd01 - lmd01o;
    dlmd02 = lmd02 - lmd02o;
    dlmd12 = lmd12 - lmd12o;
    iter ++;
  } while (dlmd01*dlmd01+dlmd02*dlmd02+dlmd12*dlmd12 > errTol2 && iter < 100);

#ifdef DEBUG_SHAKE
  printf("iterate %d times in shake \n", iter);
  printf("lmd01=%f, lmd02=%f, lmd12=%f\n", lmd01, lmd02, lmd12);
#endif

  velo = vel;
  velh1 = vel+1;
  velh2 = vel+2;

  deltav.x = po * (lmd01 * r01.x + lmd02 * r02.x) / dt;
  deltav.y = po * (lmd01 * r01.y + lmd02 * r02.y) / dt;
  deltav.z = po * (lmd01 * r01.z + lmd02 * r02.z) / dt;
  MD_pvec_add(velo, &deltav, velo);
  MD_vec_mul_add(*ro, deltav, dt);

  deltav.x = ph * (-lmd01 * r01.x + lmd12 * r12.x) / dt;
  deltav.y = ph * (-lmd01 * r01.y + lmd12 * r12.y) / dt;
  deltav.z = ph * (-lmd01 * r01.z + lmd12 * r12.z) / dt;
  MD_pvec_add(velh1, &deltav, velh1);
  MD_vec_mul_add(*rh1, deltav, dt); 

  deltav.x = -ph * (lmd02 * r02.x + lmd12 * r12.x) / dt;
  deltav.y = -ph * (lmd02 * r02.y + lmd12 * r12.y) / dt;
  deltav.z = -ph * (lmd02 * r02.z + lmd12 * r12.z) / dt;
  MD_pvec_add(velh2, &deltav, velh2);
  MD_vec_mul_add(*rh2, deltav, dt); 

#ifdef DEBUG_SHAKE
  printf("after:\n");
  printf(" O: (%f,%f, %f)\n", ro->x, ro->y, ro->z);
  printf("H1: (%f,%f, %f)\n", rh1->x, rh1->y, rh1->z);
  printf("H2: (%f,%f, %f)\n", rh2->x, rh2->y, rh2->z);
#endif

  /* check the constraint: bond length. */
  MD_pvec_substract(ro, rh1, &roh1);
  MD_pvec_substract(ro, rh2, &roh2);
  MD_pvec_substract(rh1, rh2, &rh1h2);
  doh1sqr = MD_vec_dot(roh1, roh1);
  doh2sqr = MD_vec_dot(roh2, roh2);
  dh1h2sqr = MD_vec_dot(rh1h2, rh1h2);
  if (fabs(doh1sqr - d1sqr)/d1sqr > errTol ||
      fabs(doh2sqr - d1sqr)/d1sqr > errTol ||
      fabs(dh1h2sqr - d2sqr)/d2sqr > errTol) {
    fprintf(stderr, " O: (%f,%f, %f)\n", ro->x, ro->y, ro->z);
    fprintf(stderr, "H1: (%f,%f, %f)\n", rh1->x, rh1->y, rh1->z);
    fprintf(stderr, "H2: (%f,%f, %f)\n", rh2->x, rh2->y, rh2->z);
    fprintf(stderr, "*** shake fails: |O-H1|^2=%f, |O-H2|^2 = %f,"
	    "|H1-H2|^2 = %f\n", doh1sqr, doh2sqr, dh1h2sqr);
    return MD_FAIL;
  }

  return OK;
}



/* 
 * use shake method to do constraint dynamics in leapfrog method.
 * input:  
 *        rold:  positon array (of a water molecule) at t
 *        rnew:  positon array at t + dt, not yet satisfying constraint.
 * output:  
 *        rnew: modified positon at t + dt, with constaint satisfied.
 *        vel:  velocity is DEFINED as:  (rnew - rold) / dt
 *              vel = vel(t + dt/2)
 * be careful about the periodic boundary condition.  
 */

/* something might be wrong in math.h for abs function */
MD_Errcode shake_position(struct Shake_Water *sw, const MD_Int iatom, 
			  MD_Dvec *rnew, MD_Dvec *vel, const MD_Double dt)
{
  /* rigid water model parameters. d1 = d(O, H1) = d(O, H2),
   * d2 = d(H1, H2).  theta = angle(OH1, OH2). */
  const MD_Dvec *rold = sw->old_pos + iatom;
  const MD_Double errTol = sw->bond_errTol; /* bond length */
  const MD_Double errTol2 =  sw->errTol2;
  const MD_Double d1 = sw->bondOH;   /* angstron */ 
  const MD_Double theta = sw->angle_HOH;  /* 109.5 degree */
  const MD_Double phi = (Pi - theta) * 0.5; 
  const MD_Double d2 = 2.0 * d1 * cos(phi);
  const MD_Double d1sqr = d1 * d1;
  const MD_Double d2sqr = d2 * d2;
  const MD_Double d1d2cosphi = d1 * d2 * cos(phi);
  const MD_Double d1d1costheta = d1 * d1 * cos(theta);
  const MD_Double mo = sw->massO;
  const MD_Double mh = sw->massH;
  const MD_Double po = 2.0 * dt * dt / mo;
  const MD_Double ph = 2.0 * dt * dt / mh;
  const MD_Double poph = po+ph;
  const MD_Double posqr = po*po;
  const MD_Double phsqr = ph*ph;
  const MD_Double pophsqr = poph * poph;
  const MD_Double inv_dt = 1.0 / dt;
  const MD_Dvec *r0 = rold;
  const MD_Dvec *r1 = rold + 1;
  const MD_Dvec *r2 = rold + 2;
  MD_Dvec *ro = rnew;
  MD_Dvec *rh1 = rnew + 1;
  MD_Dvec *rh2 = rnew + 2;
  MD_Dvec roh1, roh2, rh1h2;
  MD_Dvec r01, r02, r12;
  MD_Dvec *velo, *velh1, *velh2;
  MD_Dvec deltav;

  MD_Double c11, c12, c13, c21, c22, c23, c31, c32, c33;
  MD_Double lmd01 = 0.0, lmd02 = 0.0, lmd12 = 0.0;
  MD_Double lmd01o, lmd02o, lmd12o;
  MD_Double rhs1, rhs2, rhs3, inv_det;
  MD_Double err01, err02, err12;
  MD_Double dlmd01, dlmd02, dlmd12;
  MD_Double lmd01sqr, lmd02sqr, lmd12sqr, tmp1, tmp2;
  MD_Int iter;
  MD_Double doh1sqr, doh2sqr, dh1h2sqr;

#ifdef DEBUG_SHAKE
  printf("before:\n");
  printf(" O: (%f,%f, %f)\n", ro->x, ro->y, ro->z);
  printf("H1: (%f,%f, %f)\n", rh1->x, rh1->y, rh1->z);
  printf("H2: (%f,%f, %f)\n", rh2->x, rh2->y, rh2->z);
#endif

  MD_pvec_substract(ro, rh1, &roh1);
  MD_pvec_substract(ro, rh2, &roh2);
  MD_pvec_substract(rh1, rh2, &rh1h2);

  MD_pvec_substract(r0, r1, &r01);
  MD_pvec_substract(r0, r2, &r02);
  MD_pvec_substract(r1, r2, &r12);

  c11 =   poph * MD_vec_dot(roh1,  r01);
  c12 =     po * MD_vec_dot(roh1,  r02);
  c13 =   - ph * MD_vec_dot(roh1,  r12);
  c21 =     po * MD_vec_dot(roh2,  r01);
  c22 =   poph * MD_vec_dot(roh2,  r02);
  c23 =     ph * MD_vec_dot(roh2,  r12);
  c31 =   - ph * MD_vec_dot(rh1h2, r01);
  c32 =   + ph * MD_vec_dot(rh1h2, r02);
  c33 = 2.0*ph * MD_vec_dot(rh1h2, r12);

  inv_det = 1.0 /  (c11*c22*c33 + c21*c32*c13 + c31*c12*c23
		  - c13*c22*c31 - c23*c32*c11 - c33*c12*c21);

#ifdef DEBUG_SHAKE
  {
    MD_Double t1, t2, t3, t, t_inv;
    MD_Double d11, d12, d13, d21, d22, d23, d31, d32, d33;
    MD_Double cond;
    /* compute 1-norm of matrix {c_ij}. */
    t1 = fabs(c11) + fabs(c12) + fabs(c13);
    t2 = fabs(c21) + fabs(c22) + fabs(c23);
    t3 = fabs(c31) + fabs(c32) + fabs(c33);
    t = t1 > t2 ? t1:t2;
    t = t > t3 ? t:t3;
    /* compute {c_ij}^{-1} */
    d11 =  (c22*c33 - c23*c32) * inv_det;
    d12 = -(c21*c33 - c23*c31) * inv_det;
    d13 =  (c21*c32 - c22*c31) * inv_det;
    d21 = -(c12*c33 - c13*c32) * inv_det;
    d22 =  (c11*c33 - c13*c31) * inv_det;
    d23 = -(c11*c32 - c12*c31) * inv_det;
    d31 =  (c12*c23 - c22*c13) * inv_det;
    d32 = -(c11*c23 - c13*c21) * inv_det;
    d33 =  (c11*c22 - c12*c21) * inv_det;
    /* compute 1-norm of matrix {c_ij}^{-1} */
    t1 = fabs(d11) + fabs(d12) + fabs(d13);
    t2 = fabs(d21) + fabs(d22) + fabs(d23);
    t3 = fabs(d31) + fabs(d32) + fabs(d33);
    t_inv = t1 > t2 ? t1:t2;
    t_inv = t_inv > t3 ? t_inv:t3;
    cond = t * t_inv;
    printf("inv_det=%f\n", inv_det);
    printf("condition number is: %f\n", cond);
  }
#endif

  err01 = d1sqr - MD_vec_dot(roh1,  roh1);
  err02 = d1sqr - MD_vec_dot(roh2,  roh2);
  err12 = d2sqr - MD_vec_dot(rh1h2, rh1h2);
  iter = 0;
  do {   /* large iteration loop */
    lmd01o = lmd01;
    lmd02o = lmd02;
    lmd12o = lmd12;
    /* compute right hand side */
    lmd01sqr = lmd01*lmd01;
    lmd02sqr = lmd02*lmd02;
    lmd12sqr = lmd12*lmd12;
    tmp1 = lmd01 * lmd02 * d1d1costheta;
    tmp2 = lmd12 * d1d2cosphi;

    rhs1 = 0.5 * ( err01 - (pophsqr*lmd01sqr + posqr*lmd02sqr)*d1sqr
	         - phsqr*lmd12sqr*d2sqr )
         - po*poph*tmp1  - (poph*lmd01 - po*lmd02)*ph*tmp2;
    rhs2 = 0.5 * ( err02 - (posqr*lmd01sqr + pophsqr*lmd02sqr)*d1sqr
	         - phsqr*lmd12sqr*d2sqr )
	 - po*poph*tmp1 + (po*lmd01 - poph*lmd02)*ph*tmp2;
    rhs3 = 0.5 * ( err12 - phsqr * ((lmd01sqr + lmd02sqr)*d1sqr
	       		     + 4.0*lmd12sqr*d2sqr) )
	 + phsqr*(tmp1 - 2.0*(lmd01+lmd02)*tmp2);
    /* solve cij*lmdj = rhsi */
    lmd01 = (  rhs1*c22*c33 + rhs2*c32*c13 + rhs3*c12*c23
	     - c13*c22*rhs3 - c23*c32*rhs1 - c33*c12*rhs2) 
             * inv_det;
    lmd02 = (  c11*rhs2*c33 + c21*rhs3*c13 + c31*rhs1*c23
	     - c13*rhs2*c31 - c23*rhs3*c11 - c33*rhs1*c21) 
             * inv_det;
    lmd12 = (  c11*c22*rhs3 + c21*c32*rhs1 + c31*c12*rhs2
	     - rhs1*c22*c31 - rhs2*c32*c11 - rhs3*c12*c21) 
             * inv_det;
    dlmd01 = lmd01 - lmd01o;
    dlmd02 = lmd02 - lmd02o;
    dlmd12 = lmd12 - lmd12o;
    iter ++;
  } while (dlmd01*dlmd01+dlmd02*dlmd02+dlmd12*dlmd12 > errTol2 && iter < 100);

#ifdef DEBUG_SHAKE
  printf("iterate %d times in shake \n", iter);
  printf("lmd01=%f, lmd02=%f, lmd12=%f\n", lmd01, lmd02, lmd12);
#endif

  velo = vel;
  velh1 = vel+1;
  velh2 = vel+2;

  deltav.x = po * (lmd01 * r01.x + lmd02 * r02.x) * inv_dt;
  deltav.y = po * (lmd01 * r01.y + lmd02 * r02.y) * inv_dt;
  deltav.z = po * (lmd01 * r01.z + lmd02 * r02.z) * inv_dt;
  MD_vec_mul_add(*ro, deltav, dt);
  velo->x = (ro->x - r0->x) * inv_dt;
  velo->y = (ro->y - r0->y) * inv_dt;
  velo->z = (ro->z - r0->z) * inv_dt;

  deltav.x = ph * (-lmd01 * r01.x + lmd12 * r12.x) * inv_dt;
  deltav.y = ph * (-lmd01 * r01.y + lmd12 * r12.y) * inv_dt;
  deltav.z = ph * (-lmd01 * r01.z + lmd12 * r12.z) * inv_dt;
  MD_vec_mul_add(*rh1, deltav, dt); 
  velh1->x = (rh1->x - r1->x) * inv_dt;
  velh1->y = (rh1->y - r1->y) * inv_dt;
  velh1->z = (rh1->z - r1->z) * inv_dt;

  deltav.x = -ph * (lmd02 * r02.x + lmd12 * r12.x) * inv_dt;
  deltav.y = -ph * (lmd02 * r02.y + lmd12 * r12.y) * inv_dt;
  deltav.z = -ph * (lmd02 * r02.z + lmd12 * r12.z) * inv_dt;
  MD_vec_mul_add(*rh2, deltav, dt); 
  velh2->x = (rh2->x - r2->x) * inv_dt;
  velh2->y = (rh2->y - r2->y) * inv_dt;
  velh2->z = (rh2->z - r2->z) * inv_dt;


#ifdef DEBUG_SHAKE
  printf("after:\n");
  printf(" O: (%f,%f, %f)\n", ro->x, ro->y, ro->z);
  printf("H1: (%f,%f, %f)\n", rh1->x, rh1->y, rh1->z);
  printf("H2: (%f,%f, %f)\n", rh2->x, rh2->y, rh2->z);
#endif

  /* check the constraint: bond length. */
  MD_pvec_substract(ro, rh1, &roh1);
  MD_pvec_substract(ro, rh2, &roh2);
  MD_pvec_substract(rh1, rh2, &rh1h2);
  doh1sqr = MD_vec_dot(roh1, roh1);
  doh2sqr = MD_vec_dot(roh2, roh2);
  dh1h2sqr = MD_vec_dot(rh1h2, rh1h2);
  if (fabs(doh1sqr - d1sqr)/d1sqr > errTol ||
      fabs(doh2sqr - d1sqr)/d1sqr > errTol ||
      fabs(dh1h2sqr - d2sqr)/d2sqr > errTol) {
    fprintf(stderr, " O: (%f,%f, %f)\n", ro->x, ro->y, ro->z);
    fprintf(stderr, "H1: (%f,%f, %f)\n", rh1->x, rh1->y, rh1->z);
    fprintf(stderr, "H2: (%f,%f, %f)\n", rh2->x, rh2->y, rh2->z);
    fprintf(stderr, "*** shake fails: |O-H1|^2=%f, |O-H2|^2 = %f,"
	    "|H1-H2|^2 = %f\n", doh1sqr, doh2sqr, dh1h2sqr);
    return MD_FAIL;
  }

  return OK;
}

