/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


/************************************************************************
 * 
 * pme.h 
 *
 * compute the electrostatic energy and force using Particle-Mesh-Ewald
 * (pme) method.
 * the module can handle induced dipole interactions.
 * 
 * the PME module follows the namd code structure and mdx implementation 
 * with slight modifications (simplications) for serial computation.
 * The simplications come from: (1) no need to pass message, so (2) no need
 * to resize array.
 *
 * The external interface is:
 *
 *
 * The internal interface is:
 *
 *          -- direct sum ----- pme_direct (normal, same as standard Ewald)
 *         |
 *  pme ---
 *         |
 *          - reciprocal sum -- pme_recip
 *  
 *
 ************************************************************************/

#ifndef PME_H
#define PME_H

#ifdef __cplusplus
extern "C" {
#endif
 
#include "mdtypes.h"
#include "dsolvers.h"


  struct PmeParams_Tag   /* initialization parameters */
  {  
    MD_Double PMETolerance;
    MD_Int PMEGridSizeX;  /* # of grid points along x direction */
    MD_Int PMEGridSizeY;
    MD_Int PMEGridSizeZ;
    MD_Int PMEInterpOrder;
    MD_Dvec cellBasisVector1;
    MD_Dvec cellBasisVector2;
    MD_Dvec cellBasisVector3;
    MD_Dvec cellOrigin;
    MD_Double cutoff;
    MD_Double ewaldcof;      /* computed, not assigned */
    MD_Int **pexcllist;      
    /* input, not changed by Pme module */
    MD_Atom* patom;          /* for obtainning charge array */
    MD_Double *polarizability;
    MD_Dvec *ppos;
    /* added for induced dipole computation */
    MD_Int natoms;
    MD_Int restart;          
    MD_Int has_induced_dipole;
    struct Dsolver_Parameters_Type dsolver_param;
  };


  struct Pme_Tag 
  {
    struct PmeParams_Tag savePmeParams;  
  
    MD_Double potential, qq_self_potential;
    MD_Dvec *force;      /* self maintained */
    MD_Double virial[9];
    MD_Double *charge;   /* self maintained */

    struct PmeDirect_Tag* dir;
    struct PmeRecip_Tag* rec; 
    struct Dsolver_Tag* dsolver; /* solve dipole equation */
  }; 


  MD_Errcode pme_init(struct Pme_Tag*, const struct PmeParams_Tag*);
  MD_Errcode pme_destroy(struct Pme_Tag*);
  MD_Errcode pme_compute(struct Pme_Tag*);

  MD_Errcode pme_setup_dsolver(struct Pme_Tag *pme,
			       struct Dsolver_Parameters_Type dsolver_param);

  /* functions used by dsolver */
  void pme_fill_diagonal(const struct Pme_Tag *pme, MD_Double *d);
  /* flag = 0, compute -G2d, result array length is 3*natoms 
   * flag = 1, compute -G1q - G2d */
  void pme_compute_pseudores(const struct Pme_Tag* pme, const MD_Double *d,
			     const MD_Int flag, MD_Double *pseudores);

  /* interface functions */
  MD_Double pme_get_energy(const struct Pme_Tag *);
  const MD_Dvec *pme_get_force(const struct Pme_Tag *);
  MD_Double *pme_get_dipole(const struct Pme_Tag *);
  MD_Errcode pme_dump_dipole(const struct Pme_Tag *, const char* filename);

#ifdef __cplusplus
}
#endif

#endif /* PME_H */
