/*
 * Copyright (C) 2006 by David Hardy.  All rights reserved.
 */

/*
 * output_dcd.c
 *
 * The idea here is to move the logic about DCD file writing
 * (as well as whether or not a DCD file is actually being written)
 * outside of the integration routines.
 *
 * The output_dcd_header() and output_dcd_done() routines can be
 * called, respectively, before and after the integration loop
 * without checking any other variables.
 *
 * The output_dcd_frame() should be called only if a frame is ready
 * to be written, i.e., when the test on the step number warrants it.
 * As above, writing a frame is attempted only if a DCD file is actually
 * being produced.
 *
 * If the user chooses to write a DCD file, then a PDB coordinate file
 * is also written with the initial coordinates, since no PDB file will
 * otherwise exist for VMD.
 */

#include <stdio.h>
#include "data.h"


/*
 * initialize DCD file and write header
 * also write a corresponding PDB coordinate file
 * since there won't already be one!
 * return MD_FAIL for an error, otherwise 0
 */
int output_dcd_header(struct Data_Tag *data, const enum RUN_TYPE run_type)
{
  const char *dcdfile = (data->dcdfreq == 0 ? NULL :
      (THERMALIZATION == run_type ? data->dcdthermalfile : data->dcdfile));

  if (dcdfile) {
    mdio_DcdHeader header;
    char filename[MAX_FILENAME_LEN]={0};
    int retval;

    /* determine dcd filename */
    retval = snprintf(filename, sizeof(filename), "%s.dcd", dcdfile);
    if (retval <= 0) {
      fprintf(stdout, "cannot compose filename string\n");
      return MD_FAIL;
    }
    else if (retval >= (int)sizeof(filename)) { 
      fprintf(stdout, "filename is too long for buffer\n");
      return MD_FAIL;
    }

    /* open dcd file and write header info */
    if ((data->dcd = mdio_createDcd()) == NULL) {
      fprintf(stderr, "mdio_createDcd() failed\n");
      return MD_FAIL;
    }
    if (MDIO_ERROR == mdio_writeBeginDcd(data->dcd, filename)) {
      fprintf(stderr, "mdio_writeBeginDcd() failed\n");
      return MD_FAIL;
    }
    header.timestep = data->timestep;
    header.natoms = data->natoms;
    header.firststep = data->firststepnum;
    header.framestepcnt = data->dcdfreq;
    header.cellstatus = MDIO_DCD_UNITCELL | MDIO_DCD_FIXEDCELL;
    if (MDIO_ERROR == mdio_writeHeaderDcd(data->dcd, &header)) {
      fprintf(stderr, "mdio_writeHeaderDcd() failed\n");
      return MD_FAIL;
    }

    /* set dcd cell info */
    data->dcdcell.vec1.x = data->systemsize.x;
    data->dcdcell.vec1.y = 0.0;
    data->dcdcell.vec1.z = 0.0;
    data->dcdcell.vec2.x = 0.0;
    data->dcdcell.vec2.y = data->systemsize.y;
    data->dcdcell.vec2.z = 0.0;
    data->dcdcell.vec3.x = 0.0;
    data->dcdcell.vec3.y = 0.0;
    data->dcdcell.vec3.z = data->systemsize.z;

    /* determine pdb filename (same base, different suffix) */
    retval = snprintf(filename, sizeof(filename), "%s.pdb", dcdfile);
    if (retval <= 0) {
      fprintf(stdout, "cannot compose filename string\n");
      return MD_FAIL;
    }
    else if (retval >= (int)sizeof(filename)) { 
      fprintf(stdout, "filename is too long for buffer\n");
      return MD_FAIL;
    }

    /* write pdb coordinate file */
    if (output_pdb(data, filename)) {
      fprintf(stdout, "failed to write pdb file\n");
      return MD_FAIL;
    }
  }

  return 0;
}


/*
 * write frame to DCD file
 * return MD_FAIL for an error, otherwise 0
 */
int output_dcd_frame(struct Data_Tag *data, MD_Int istep)
{
  if (data->dcd && MDIO_ERROR == mdio_writeFrameDcd(data->dcd, data->pos,
       	&(data->dcdcell), istep)) {
    fprintf(stderr, "mdio_writeFrameDcd() failed\n");
    return MD_FAIL;
  }
  return 0;
}


/*
 * close DCD file and destroy DCD file writer object
 * return MD_FAIL for an error, otherwise 0
 */
int output_dcd_done(struct Data_Tag *data)
{
  if (data->dcd) {
    if (MDIO_ERROR == mdio_writeEndDcd(data->dcd)) {
      fprintf(stderr, "mdio_writeEndDcd() failed\n");
      return MD_FAIL;
    }
    mdio_destroyDcd(data->dcd);
    data->dcd = NULL;
  }
  return 0;
}

