/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "myerfc.h"
#include "constant.h"
#include "helper.h"

double erfc(double x);

static MD_Double Hermite_interpolate(MD_Double x, MD_Double x0, MD_Double dx,
				     MD_Double f0, MD_Double f1,
				     MD_Double fp0, MD_Double fp1);

MD_Errcode erfc_init(struct Erfc_Tag *myerfc)
{
  register MD_Int i;
  register MD_Double x;

  myerfc->np = 1600;
  myerfc->xmax = 6.0;
  myerfc->dx = myerfc->xmax / (double) (myerfc->np);  
  myerfc->f  = my_calloc((size_t) (myerfc->np+1), sizeof(MD_Double), "erfc");
  myerfc->fp = my_calloc((size_t) (myerfc->np+1), sizeof(MD_Double), "erfc");

  for (i = 0; i <= myerfc->np; i++) {
    x = ((double) i) * myerfc->dx;
    myerfc->f[i] = erfc(x);
    myerfc->fp[i] = - exp(-x*x) * 2.0 / sqrt(Pi);
  }

  return 0;
}

MD_Errcode erfc_destroy(struct Erfc_Tag *myerfc)
{
  free(myerfc->f);
  free(myerfc->fp);
  return 0;
}



MD_Double erfc_eval(const struct Erfc_Tag *myerfc, const MD_Double x)
{
  MD_Int i;

  /*
  if (x < 0.0) { // do not need in the electrostatic computation.
    fprintf(stderr, "wrong value of x: %f < 0\n", x);
    return -1.0;
  } else if (x < myerfc->xmax) {
  */
  if (x < myerfc->xmax) {
    i = (MD_Int) (x / myerfc->dx);
    if (i < 0 || i >= myerfc->np) {
      fprintf(stderr, "wrong, x=%f, i=%d, np=%d\n", x,i, myerfc->np);
      return -1.0;
    }
    return Hermite_interpolate(x, ((MD_Double) i) * myerfc->dx, myerfc->dx,
			       myerfc->f[i], myerfc->f[i+1],
			       myerfc->fp[i], myerfc->fp[i+1]);
  } else if (x > 0.0) { /* numerically zero. */
    return 0.0;
  } 

  fprintf(stderr, "wrong value of x: %f < 0\n", x);
  return -1.0;
}



/* Hermite interpolation */
MD_Double Hermite_interpolate(MD_Double x, MD_Double x0, MD_Double dx,
			      MD_Double f0, MD_Double f1,
			      MD_Double fp0, MD_Double fp1)
{
  MD_Double d01 = (f1 - f0) / dx;
  MD_Double dd = x - x0;
  MD_Double a = (fp1 + fp0 - 2.0 * d01) / (dx*dx);
  MD_Double b = (3.0 * d01 - 2.0 * fp0 - fp1) / dx;

  return f0 + dd * (fp0 + dd * (b + dd * a));
}
