/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


/*
 * linkcell.c
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "linkcell.h"
#include "helper.h"
#include "utilities.h"

#undef BOX_INDEX
#define BOX_INDEX(x,y,z,nxbox,nybox)  ((x) + nxbox*((y) + nybox*(z)))

static void init_cells(struct LinkCell_Tag *linkcell);

static 
MD_Int is_less(MD_Int x,  MD_Int y,  MD_Int z,
		MD_Int xp, MD_Int yp, MD_Int zp,
		MD_Int nxcell, MD_Int nycell, MD_Int nzcell);
static 
MD_Int is_neibr(struct Cell_Tag *c1, struct Cell_Tag *c2,
		 MD_Dvec cellsize, MD_Dvec systemsize, MD_Double cutoff2);


MD_Errcode linkcell_init(struct LinkCell_Tag *linkcell, 
			 struct LinkCell_init_Tag *init_data)
{
  assert(NULL != linkcell);
  assert(NULL != init_data);

  linkcell->nxcell     = init_data->nxcell;
  linkcell->nycell     = init_data->nycell;
  linkcell->nzcell     = init_data->nzcell;
  linkcell->systemsize = init_data->systemsize;
  linkcell->min        = init_data->min;
  linkcell->cutoff     = init_data->cutoff;
  linkcell->natoms     = init_data->natoms;

  linkcell->cellsize.x = linkcell->systemsize.x / (MD_Double)linkcell->nxcell; 
  linkcell->cellsize.y = linkcell->systemsize.y / (MD_Double)linkcell->nycell; 
  linkcell->cellsize.z = linkcell->systemsize.z / (MD_Double)linkcell->nzcell; 

  linkcell->numcell = linkcell->nxcell * linkcell->nycell * linkcell->nzcell;
  linkcell->head = my_calloc((size_t)linkcell->numcell, sizeof(MD_Int), "head");
  linkcell->list = my_calloc((size_t)linkcell->natoms,  sizeof(MD_Int), "list");
  linkcell->cell = my_calloc((size_t)linkcell->numcell, 
			     sizeof(struct Cell_Tag), "cell");
  init_cells(linkcell);

  return OK;

}


MD_Errcode linkcell_destroy(struct LinkCell_Tag *linkcell)
{
  MD_Int i;

  for(i = 0; i < linkcell->numcell; i++) {
    free(linkcell->cell[i].nbr);
  }
  free(linkcell->cell);
  free(linkcell->head);
  free(linkcell->list);
  memset(linkcell, 0, sizeof(struct LinkCell_Tag));

  return OK;
}


MD_Errcode linkcell_hash_atoms(struct LinkCell_Tag *linkcell, 
			       const MD_Dvec* pos)
{
  MD_Int* head = linkcell->head;
  MD_Int* list = linkcell->list;
  const MD_Double inv_xcellsize = 1.0 / linkcell->cellsize.x;
  const MD_Double inv_ycellsize = 1.0 / linkcell->cellsize.y;
  const MD_Double inv_zcellsize = 1.0 / linkcell->cellsize.z;
  const MD_Dvec min = linkcell->min;
  const MD_Int nxcell = linkcell->nxcell;
  const MD_Int nycell = linkcell->nycell;
  const MD_Int natoms = linkcell->natoms;
  MD_Int numcell = linkcell->numcell;
  MD_Int i, kx, ky, kz, k;

  for (i = 0; i < numcell; i++) {
    head[i] = -1;
  }
  /* printf("xmin=%f, ymin=%f zmin=%f\n", xmin, ymin, zmin); */
  /* place each atom into its cell */
  for (i = 0;  i < natoms;  i++) {
    kx = (MD_Int)((pos[i].x - min.x) * inv_xcellsize);
    ky = (MD_Int)((pos[i].y - min.y) * inv_ycellsize);
    kz = (MD_Int)((pos[i].z - min.z) * inv_zcellsize);
    /* check the cell ID within range */
    ASSERT(0<=kx && kx<nxcell && 0<=ky && ky<nycell && 0<=kz && 
           kz<linkcell->nzcell);
    k = BOX_INDEX(kx, ky, kz, nxcell, nycell);
    ASSERT(0 <= k && k < numcell); 
    list[i] = head[k];
    head[k] = i;
  }

#ifdef DEBUG_HEAD_LIST
  for (k = 0; k < numcell; k++) {
    printf("cell %d: ", k);
    i = head[k];
    while (i >= 0) {
      printf("%d,", i);
      i = list[i];
    }
    printf("\n");
  }
  fflush(stdout);
#endif

  return OK;
}


/* init data members of each cell */
void init_cells(struct LinkCell_Tag *linkcell)
{
  const MD_Int nxcell = linkcell->nxcell;
  const MD_Int nycell = linkcell->nycell;
  const MD_Int nzcell = linkcell->nzcell;
  const MD_Dvec cellsize = linkcell->cellsize;
  const MD_Dvec systemsize = linkcell->systemsize;
  const MD_Double cutoff2 = linkcell->cutoff * linkcell->cutoff;
  const MD_Dvec min = linkcell->min;
  struct Cell_Tag* cell = linkcell->cell;
  MD_Int i,j,k;
  MD_Int ip, jp, kp;
  MD_Int ind, indp, numnbrs;
  MD_Int* nbr;

  for (k = 0;  k < nzcell;  k++) {
    for (j = 0;  j < nycell;  j++) {
      for (i = 0;  i < nxcell;  i++) {
        ind = BOX_INDEX(i, j, k, nxcell, nycell);
        cell[ind].nx = i;
        cell[ind].ny = j;
        cell[ind].nz = k;
        cell[ind].min.x = min.x + i * cellsize.x;
        cell[ind].min.y = min.y + j * cellsize.y;
        cell[ind].min.z = min.z + k * cellsize.z;
      }
    }
  }

  for (k = 0; k < nzcell; k++) {
    for (j = 0; j < nycell; j++) {
      for (i = 0; i < nxcell; i++) {
        ind = BOX_INDEX(i,j,k,nxcell,nycell);
	cell[ind].nbr = my_calloc((size_t)linkcell->numcell, sizeof(MD_Int),
				  "cell's neighborlist");
        nbr = cell[ind].nbr;
        numnbrs = 0;
        for (ip = 0; ip < nxcell; ip++) {
          for (jp = 0; jp < nycell; jp++) {
            for (kp = 0; kp < nzcell; kp++) {
	      indp = BOX_INDEX(ip,jp,kp,nxcell,nycell);
              /* so each cell has approximately same number of neibrs */
              if (is_less(i,j,k, ip,jp,kp, nxcell,nycell,nzcell) &&
                  is_neibr(cell+ind, cell+indp, cellsize, systemsize, cutoff2)) {
                nbr[numnbrs] = indp;
                numnbrs++;
              }
            }
          }
        }
        cell[ind].numnbrs = numnbrs;
	/*
	fprintf(stderr, "cell %d has %d neighbor cells\n", ind, numnbrs);
	*/
	cell[ind].nbr = realloc(cell[ind].nbr, 
				(size_t)numnbrs * sizeof(MD_Int));
	assert(NULL != cell[ind].nbr || 0 == cell[ind].numnbrs); 

      }
    }
  }

  /* naive check on neighbor cell index */
  for (i = 0; i < linkcell->numcell; i++) {
    nbr = cell[i].nbr;
    for (j = 0; j < cell[i].numnbrs; j++) {
      assert(0 <= nbr[j] && nbr[j] < linkcell->numcell);
    }
  }

}


MD_Int is_less(MD_Int x,  MD_Int y,  MD_Int z,
                MD_Int xp, MD_Int yp, MD_Int zp,
                MD_Int nxcell, MD_Int nycell, MD_Int nzcell)
{
  return 
    ( (zp>z && (zp-z)*2<=nzcell) || (z>zp && (z-zp)*2>nzcell))  ||
    (0==zp-z && 
     ((yp>y && (yp-y)*2<=nycell) || (y>yp && (y-yp)*2>nycell))) ||
    (0==zp-z && 0 == yp-y &&
     ((xp>x && (xp-x)*2<=nxcell) || (x>xp && (x-xp)*2>nxcell)));
}


/* this is robust, though a little stupid */
MD_Int is_neibr(struct Cell_Tag *c1, struct Cell_Tag *c2,
		 MD_Dvec cellsize, MD_Dvec systemsize, MD_Double cutoff2)
{
  /* use static to avoid performance drag */
  static MD_Dvec disp[8] = {
    {0.0, 0.0, 0.0}, {0.0, 0.0, 1.0}, {0.0, 1.0, 0.0}, {1.0, 0.0, 0.0},
    {0.0, 1.0, 1.0}, {1.0, 1.0, 0.0}, {1.0, 0.0, 1.0}, {1.0, 1.0, 1.0}};
  MD_Dvec vert1, vert2, diff;
  MD_Int i,j;

  for (i = 0; i < 8; i++) {
    vert1.x = c1->min.x + disp[i].x * cellsize.x;
    vert1.y = c1->min.y + disp[i].y * cellsize.y;
    vert1.z = c1->min.z + disp[i].z * cellsize.z;
    for (j = 0; j < 8; j++) {
      vert2.x = c2->min.x + disp[j].x * cellsize.x;
      vert2.y = c2->min.y + disp[j].y * cellsize.y;
      vert2.z = c2->min.z + disp[j].z * cellsize.z;
      MD_vec_substract(vert1, vert2, diff);
      SIMPLE_BOUND_VEC(diff, systemsize);
      if (MD_vec_dot(diff, diff) < cutoff2) {
        return 1;
      }
    }
  }
  return 0;
}
