/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include "linearfit.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <assert.h>

/* for large number of data points, and less round-off errors, it's better
 * to store data
 */


MD_Int linearfit_init(struct LinearFit_Tag *lf, const MD_Int maxsize)
{
  assert(maxsize > 0);
  lf->x = calloc((size_t) maxsize, sizeof(MD_Double));
  lf->y = calloc((size_t) maxsize, sizeof(MD_Double));
  if (NULL == lf->x || NULL == lf->y) {
    fprintf(stderr, "cannot allocate enough memory\n");
    return 1;
  }
  lf->nmax = maxsize;
  lf->ndata = 0;
  lf->computed = 0;

  return 0;
}

void linearfit_destroy(struct LinearFit_Tag* lf)
{
  free(lf->x);
  free(lf->y);
}

void linearfit_add_element(struct LinearFit_Tag *lf, const MD_Double xnew, 
			   const MD_Double ynew) 
{
  /*
  printf("add element, ndata=%d, nmax=%d\n", lf->ndata, lf->nmax);
  */
  if (lf->ndata - lf->nmax == 0) { /* double the size */
    MD_Double *tmpx, *tmpy;
    lf->nmax *= 2;
    tmpx = realloc(lf->x, lf->nmax * sizeof(MD_Double));
    tmpy = realloc(lf->y, lf->nmax * sizeof(MD_Double));
    assert(NULL != tmpx);
    assert(NULL != tmpy);
    lf->x = tmpx;
    lf->y = tmpy;
    printf ("memory expanded, nmax=%d\n", lf->nmax);
  }

  lf->x[lf->ndata] = xnew;
  lf->y[lf->ndata] = ynew;
  lf->ndata ++;
  lf->computed = 0;

}
  
void linearfit_compute(struct LinearFit_Tag *lf)
{
  const MD_Int ndata = lf->ndata;
  const MD_Double doublen = (MD_Double) lf->ndata;
  MD_Double avgx;
  MD_Double avgy;
  const MD_Double* x = lf->x;
  const MD_Double* y = lf->y;
  MD_Double sumX, sumY, sumX2, sumXY;
  MD_Double tmp, dx, dy, dxy, ex, ey;
  MD_Int i;

  if(1 >= lf->ndata ) {
    fprintf(stderr, "empty data set or only one data pair, cannot analyze\n");
    return;  
  }

  sumX = sumY = sumX2 = sumXY = 0.0;
  for(i = 0; i < ndata; i++) {
    sumX  += x[i];
    sumY  += y[i];
    sumX2 += x[i] * x[i];
    sumXY += x[i] * y[i];
  }
  avgx = sumX / doublen;
  avgy = sumY / doublen;

  tmp = 1.0 / (doublen*sumX2 - sumX*sumX);
  lf->k = (doublen*sumXY - sumX*sumY ) * tmp ;
  lf->b = (sumX2*sumY    - sumXY*sumX) * tmp ;
  dx = dy = dxy = 0.0;
  for (i = 0; i < ndata; i++) {
    ex  = x[i] - avgx;
    ey  = y[i] - avgy;
    dx += ex*ex;
    dy += ey*ey;
    dxy+= ex*ey;
  }
  lf->r = dxy / sqrt(dx*dy);

  lf->computed = 1;
}

void linearfit_print_stats(struct LinearFit_Tag *lf, const char *title)
{
  if (!lf->computed) linearfit_compute(lf);
  printf("%s, k=%g, b=%g, (0<|r|<1)r=%g\n", 
	 title, lf->k, lf->b, lf->r);
  /*
{
  MD_Int i;
  for (i = 0; i < lf->ndata; i++) {
    printf("%d,  %f, %f\n", i, lf->x[i], lf->y[i]);
  }
}
  */
}


MD_Double linearfit_getslope(struct LinearFit_Tag *lf) { 
  if (!lf->computed) linearfit_compute(lf); 
  return lf->k; 
}
MD_Double linearfit_getstart(struct LinearFit_Tag *lf) { 
  if (!lf->computed) linearfit_compute(lf); 
  return lf->b; 
}
MD_Double linearfit_getrelcoeff(struct LinearFit_Tag *lf) { 
  if (!lf->computed) linearfit_compute(lf); 
  return lf->r; 
}


void linearfit_dump_data(struct LinearFit_Tag *lf, const char *filename)
{
  const MD_Double *x = lf->x;
  const MD_Double *y = lf->y;
  const MD_Int ndata = lf->ndata;
  MD_Int i;
  FILE* fd = fopen(filename, "w");

  assert(NULL != fd);
  for(i=0; i<ndata; i++) {
    fprintf(fd, "%f\t%f\n", x[i], y[i]);
  } 
  if (ferror(fd)) {
    printf("error in writing file %s\n", filename);
  }
  fclose(fd);

}


void linearfit_clear(struct LinearFit_Tag *lf)
{
  lf->ndata = 0;
}

