/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */

/*
 * constant energy molecular dynamics simulation
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include <string.h>
#include "data.h"
#include "fnonbond.h"
#include "dsolvers.h"
#include "standEwald.h"
#include "force.h"
#include "utilities.h"
#include "helper.h"
#include "shake.h"
#include "linearfit.h"
#include "avgvar.h"
#include "data_collector.h"
#include "constEnergy.h"
#include "constant.h"
#include "unit.h"


MD_Errcode leapfrog_MD_run(struct Force_Tag *force, const MD_Int nsteps,
			   const enum RUN_TYPE run_type)
{
  struct Data_Tag *data = force->data;
  MD_Atom *atom = data->atom;
  MD_Dvec *vel = data->vel;
  MD_Dvec *pos = data->pos;
  MD_Dvec *f = force->f;
  MD_Double dt = data->timestep;  
  const MD_Int natoms = data->natoms;
  const MD_Int output_freq = 100;
  const MD_Int dump_period = 20000;  
  const MD_Int bkpoint_period = 100000; /* 200ps, about 1day run on linux */
  MD_Int istep, k;
  MD_Double tmp;
  MD_String filename;
  struct Data_Collector_Tag* data_collector = NULL;
  const MD_Int nmols = natoms / 3;
  struct Shake_Water *sw = NULL;

  if (0 == nsteps) return OK;

  /* sanity check */
  assert(force->model_id >= 0 && force->model_id < NMODELS); 
  if (POL1==force->model_id || POL3==force->model_id || 
      RPOL==force->model_id || SPC==force->model_id) {
    assert(nmols * 3 == natoms); /* water molecule systems */
  }

  printf("\n\n-------------------------------------------------------\n");
  printf("Constant energy simulation (leapfrog): to %s for %d steps ...\n", 
	 (THERMALIZATION == run_type ? "thermalize": "run"), nsteps);
  printf("  timestep = %f femto second \n", dt);
  printf("  output frequency = %d (output energy)\n", output_freq);
  printf("  dump period is %d (dump data) \n", dump_period);
  printf("  break point period is %d (dump image for restart)\n",
         bkpoint_period);
  printf("-------------------------------------------------------\n");


  if (PRODUCTION == run_type) {
    data_collector = my_malloc(sizeof(struct Data_Collector_Tag),
			       "data collector");
    data_collector_init(data_collector, data, force, ConstEnergy);
  }

  if (POL1==force->model_id || POL3==force->model_id || 
      RPOL==force->model_id || SPC==force->model_id) {
    sw = malloc(sizeof(struct Shake_Water));
    assert(NULL != sw);
    shake_init(sw, data->bond_len, data->bond_angle, O_MASS, H_MASS, 
	       natoms);
  } 

#if 0
  /* ad hoc, to make the total energy as desired value ============ */
  {
#include "unit.h"
    const MD_Double desired_tot_e = -1730.0 * KCAL_PER_MOL;
    MD_Double fac = (desired_tot_e - data->energy[TE]) 
                              / data->energy[KE] + 1.0;
    MD_Int i;

    assert(fac > 0.0);
    fac = sqrt(fac);
    printf("fac=%f\n", fac);
    for(i = 0; i < data->natoms; i++) {
      MD_vec_mul(vel[i], fac, vel[i]);
    }
  }
  compute_KETE(data); 
#endif  

  /*force_output(force);*/ 

  if (!conserve_linear_momentum(data)) return MD_FAIL;

  /*
  data_collector_update(data_collector, data->firststepnum);
  */

  /* 
   *start the MD run, leapfrog integrator
   */
  for (istep = data->firststepnum + 1; istep <= data->firststepnum + nsteps; 
       istep++) {     
    /* printf("step %d\n", istep); */
    /* fprintf(stderr, "half kick\n"); */
    data->energy[KE] = 0.5 *compute_KE(vel, atom, natoms);

    /* fprintf(stderr, "compute force\n"); */
    if (force_compute(force))  return MD_FAIL;
    /*force_output(force); */

    /* 1. kick */
    for (k = 0;  k < natoms;  k++) {
      tmp = dt / atom[k].m;
      MD_vec_mul_add(vel[k], f[k], tmp);
    }
    if (!conserve_linear_momentum(data)) return MD_FAIL;

    /* 2. drift and shake */
    /*  fprintf(stderr, "drift and shake\n"); */
    if (NULL != sw) {
      shake_prepare(sw, pos);
    }
    for (k = 0;  k < natoms;  k++) {
      MD_vec_mul_add(pos[k], vel[k], dt);
      /*printf("pos[%d]: %f, %f, %f\n", k, pos[k].x, pos[k].y, pos[k].z); */
    }

    if (NULL != sw) {
      /* fprintf(stderr, "shake to keep rigid water molecule\n"); */
      for (k = 0; k < nmols; k++) {
	if (shake(sw, 3*k, pos+3*k, vel+3*k, dt)) {
	  return MD_FAIL;
	}
      } 
    }

    if (!conserve_linear_momentum(data)) return MD_FAIL;
    
    data->energy[KE] += 0.5 * compute_KE(vel, atom, natoms);

    if (0 == (istep-1) % output_freq) {  /* output */
      data->energy[TE] = data->energy[KE] + data->energy[PE];
      data_print_energy(data, istep-1);

      if (NULL != data_collector) {
	data_collector_update(data_collector, istep-1);
      }

      if ((istep - 1 - data->firststepnum) % dump_period == 0 && istep > 1) {
	if (NULL != data_collector) data_collector_output(data_collector);
	if (sprintf(filename, "output/pos_%d.dat", istep) + 1 >
	    (MD_Int)sizeof(MD_String)) {
	  fprintf(stdout, "filename is too small\n");
	  return MD_FAIL;
	}
	if (bindump_vec_array(pos, natoms, filename)) {
	  fprintf(stderr, "cannot write file %s, disk full ?? \n", filename);
	}
      }

      if ((istep-1-data->firststepnum) % bkpoint_period == 0 && istep > 1) {
	/*printf("output position and velocity at step %d\n", istep);*/
	if (data_bindump_image(data, istep)) {
	  fprintf(stderr, "cannot dump system image \n"); /* soft fail */
	}
      }
    }

    /*fflush(NULL);*/
  } /* end of istep */

  if (NULL != data_collector) {
    data_collector_destroy(data_collector);
    free(data_collector);
  }

  if (NULL != sw) {
    shake_destroy(sw);
    free(sw);
  }

  return OK;
}

