/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */

#include <math.h>
#include <assert.h>
#include "lattice.h"

static void recalculate(struct Lattice_Tag *lat);


void lattice_set(struct Lattice_Tag *lat, MD_Dvec A, MD_Dvec B, MD_Dvec C, 
		 MD_Dvec Origin)
{
  lat->a1 = A;
  lat->a2 = B;
  lat->a3 = C;
  lat->corner.x = Origin.x - 0.5*A.x;  /* works only for othogonal lattice */
  lat->corner.y = Origin.y - 0.5*B.y;
  lat->corner.z = Origin.z - 0.5*C.z;
  lat->p1 = 1;  /* periodic in every direction */
  lat->p2 = 1;
  lat->p3 = 1;
  assert ( lattice_is_orthogonal(lat) ); 
  assert (lat->a1.x > 0 && lat->a2.y > 0 && lat->a3.z > 0);
  assert ( lattice_volume(lat) > 0.0 );
  recalculate(lat);
}


MD_Int lattice_is_orthogonal(const struct Lattice_Tag *lat)
{
  return !(lat->a1.y || lat->a1.z || lat->a2.x || lat->a2.z || lat->a3.x || 
	   lat->a3.y);
}


MD_Double lattice_volume(const struct Lattice_Tag *lat)
{
  return lat->a1.x * lat->a2.y * lat->a3.z;
}


/* calculate reciprocal lattice vectors */
void recalculate(struct Lattice_Tag *lat)
{
  lat->b1.y = lat->b1.z = 0.0; lat->b1.x = 1.0 / lat->a1.x;
  lat->b2.x = lat->b2.z = 0.0; lat->b2.y = 1.0 / lat->a2.y;
  lat->b3.x = lat->b3.y = 0.0; lat->b3.z = 1.0 / lat->a3.z;
}
