/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "utilities.h"
#include "helper.h"


void outputv(const MD_Double *v, const MD_Int n, const char *title)
{
  int i;
  printf("%s\n", title);
  for (i = 0; i < n; i++) {
    printf("%d %20.14f\n", i, v[i]);
  }
  fflush(stdout);
  return;
}


void output_vec_array(const MD_Dvec *v, const MD_Int n, const char *title)
{
  int i;
  printf("%s\n", title);
  for (i = 0; i < n; i++) {
    printf("%6d %20.14f %20.14f %20.14f\n", i, v[i].x, v[i].y, v[i].z);
  }
  fflush(stdout);
  return;
}


/*
 * compute dot product of vector v1 and v2 of length n
 */
MD_Double DOT(const MD_Double *v1, const MD_Double *v2, const MD_Int n) 
{
  register MD_Double sum = 0.0;
  register MD_Int i;
  for (i = 0; i < n; i++) sum += v1[i] * v2[i];
  return sum;
}


MD_Double array_sum(const MD_Double *v, const MD_Int n)
{
  register MD_Double sum = 0.0;
  register MD_Int i;
  for (i = 0; i < n; i++) sum += v[i];
  return sum;
}


MD_Errcode bindump_vec_array(const MD_Dvec *v, const MD_Int n, 
			     const char* filename)
{
  FILE *fd = fopen(filename, "wb");
  size_t noutput;

  if (NULL == fd) {
    fprintf(stderr, "cannot open file %s to output vector.\n", 
	    filename);
    return MD_FAIL;
  }

  noutput = fwrite(v, sizeof(MD_Dvec), (size_t)n, fd);
  if (noutput < (size_t)n || ferror(fd)) {
    perror("error in writing file");
    return MD_FAIL;
  }

  if (fclose(fd)) {
    perror("canno close file");
    return MD_FAIL;
  }

  return OK;
}


MD_Errcode binread_vec_array(MD_Dvec *v, const MD_Int n, 
			     const char* filename)
{
  MD_Int nreadelem;
  FILE *fd = fopen(filename, "rb");

  if (NULL == fd) {
    fprintf(stderr, "cannot open file %s to read vector.\n", 
	    filename);
    return MD_FAIL;
  }

  nreadelem = 0;
  while (nreadelem  < n) {
    nreadelem += fread(v+nreadelem, sizeof(MD_Dvec), (size_t)(n-nreadelem), 
		       fd);
  }

  if (ferror(fd)) {
    perror("in reading file");
    fclose(fd);
    return MD_FAIL;
  } else {
    fclose(fd);
    return OK;
  }
}

MD_Errcode binread_array(MD_Double *v, const MD_Int n, 
			 const char* filename)
{
  MD_Int nreadelem;
  FILE *fd = fopen(filename, "rb");

  if (NULL == fd) {
    fprintf(stderr, "cannot open file %s to read vector.\n", 
	    filename);
    return MD_FAIL;
  }

  nreadelem = 0;
  while (nreadelem  < n) {
    nreadelem += fread(v+nreadelem, sizeof(MD_Double), (size_t)(n-nreadelem), 
		       fd);
  }

  if (ferror(fd)) {
    perror("in reading file");
    fclose(fd);
    return MD_FAIL;
  } else {
    fclose(fd);
    return OK;
  }
}




MD_Errcode bindump_array(const MD_Double *v, const MD_Int n, 
			     const char* filename)
{
  FILE *fd = fopen(filename, "wb");
  size_t noutput;

  if (NULL == fd) {
    perror("cannot open file");
    return MD_FAIL;
  }

  noutput = fwrite(v, sizeof(MD_Double), (size_t)n, fd);
  if (noutput < (size_t)n || ferror(fd)) {
    perror("error in writing file");
    return MD_FAIL;
  }

  fclose(fd);
  return OK;
}


MD_Int my_strcasecmp(register const char *cs, 
		     register const char *ct)
{
  register int ks, kt;
  do {
    ks=tolower((int)(*cs++));
    kt=tolower((int)(*ct++));
  } while (ks == kt && ks != 0);

  return (ks - kt);
}


void* my_calloc(size_t nmemb, size_t size, const char* name) 
{
  void *ptr = calloc(nmemb, size);
  if (NULL == ptr) {
    fprintf(stderr, "failed to allocate memeory for %s\n", name);
    exit(MD_ERR_MEMALLOC);
  }
  return ptr;
}

void* my_malloc(size_t total_size, const char* name) 
{
  void *ptr = malloc(total_size);
  if (NULL == ptr) {
    fprintf(stderr, "failed to allocate memeory for %s\n", name);
    exit(MD_ERR_MEMALLOC);
  }
  return ptr;
}


/* use gauss elimnation method to solve the linear system 
 * assume the size is less than 20. */
MD_Int GaussElmination(MD_Int n, MD_Double *a, MD_Double *b)
{
  MD_Int i,j,k;
  MD_Double tmp[50];
  MD_Double pivot;

  assert(n < 50);

  for(i = 0; i < n; i++) {
    /* pivot */
    MD_Int imax = i; 
    MD_Double rmax = fabs(a[i*n+i]);
    for(j=i+1; j< n; j++) {
      if (rmax < fabs(a[j*n+i])) {
    	imax = j;
	    rmax = fabs(a[j*n+i]);
      }
    }
    if (0 != i - imax) {  
      MD_Double t = b[i];  b[i] = b[imax];  b[imax] = t;
      memcpy(tmp, a+i*n+i, (n-i)*sizeof(MD_Double));
      memcpy(a+i*n+i, a+imax*n+i, (n-i)*sizeof(MD_Double));
      memcpy(a+imax*n+i, tmp, (n-i)*sizeof(MD_Double));
    }

    /* eliminate */
    pivot = a[i*n+i];
    /* a[i*n+i] = 1.0; */
    for(j=i+1; j < n; j++)  a[i*n+j] /= pivot;
    b[i] /= pivot;
    for(j=i+1; j < n; j++) {
      MD_Double t = a[j*n+i];
      for(k = i+1; k < n; k++) a[j*n+k] -= t * a[i*n+k];
      /*a[j*n+i] = 0.0;  // deletable  */
      b[j] -= t * b[i];
    } 
  }

  /* back substitute */
  for(i=n-2; i >=0; i--) {
    for(j = i+1; j < n; j++) {
      b[i] -= a[i*n+j] * b[j];
    }
  }

  return OK;
}

