/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */

/*
 * force.c
 *
 * High level force routines, build exclusion lists and vdw param matrix.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "force.h"
#include "utilities.h"
#include "helper.h"

/*
 * initialize the force object
 *
 * assumes that force is already allocated and memory cleared
 * data must already be initialized
 *
 * allocate and initialize nonbonded exclusion lists
 *
 * allocate and initialize van der Waals parameters table
 */
MD_Errcode force_init(struct Force_Tag *force, 
		      struct Force_init_Tag* params)
{
  force->data = params->data;   /* set back pointer, cannot not use it */
  force->restart = params->restart;

  force->model_id = force->data->model_id;
  force->natoms = force->data->natoms;
  force->systemsize = force->data->systemsize;

  printf("systemsize: %f, %f, %f\n", force->systemsize.x, force->systemsize.y,
	 force->systemsize.z);

  force->f = my_calloc((size_t)force->natoms, sizeof(MD_Dvec), "force->f");

  /* allocate my own position array */
  force->wrapped_pos = my_calloc((size_t)force->natoms, sizeof(MD_Dvec), 
				  "wrapped position" );

  /*
  force->fbond = calloc( (size_t)1, sizeof(struct Fbond_Tag) );
  if (NULL == force->fbond) {
    fprintf(stderr, "cannot allocate memory for bond force \n");
    return MD_FAIL;
  }
  if (fbond_init(force->fbond, force)) {
    fprintf(stderr, "cannot init bond force structure \n");
    return MD_FAIL;
  }
  */

  force->fnonbond = my_calloc( (size_t)1, sizeof(struct Fnonbond_Tag),
			       "fnonbond" );

  {
    struct Fnonbond_init_Tag nbparams;
    nbparams.force         = force;
    nbparams.data          = force->data;
    nbparams.ewaldmethod   = params->emodule;
    nbparams.dsolver_param = params->dsolver_param;
    if (fnonbond_init(force->fnonbond, &nbparams)) {
      fprintf(stderr, "cannot init nonbond force structure \n");
      return MD_FAIL;
    }
  }
  return OK;
}



/*
 * destroy the force object
 *
 * free all memory allocated and clear memory used by force object
 */
MD_Errcode force_destroy(struct Force_Tag *force)
{
  if (force->f) free(force->f);
  if (force->wrapped_pos) free(force->wrapped_pos);
  /*
  if (fbond_destroy(force->fbond)) return MD_FAIL;
  free(force->fbond);
  */
  if (fnonbond_destroy(force->fnonbond)) return MD_FAIL;
  free(force->fnonbond);
  /* prevent seg fault from double free */
  memset(force, 0, sizeof(struct Force_Tag)); 

  return OK;
}


/*
 * compute the force and the potential
 */
MD_Errcode force_compute(struct Force_Tag *force)
{
  struct Data_Tag* data = force->data;
  MD_Double *energy = data->energy;
  MD_Dvec *wrapped_pos = force->wrapped_pos;
  MD_Dvec systemsize = force->systemsize;
  MD_Dvec *f = force->f;
  const MD_Dvec *vdwf = NULL; 
  const MD_Dvec *elef = NULL;
  const MD_Int natoms = force->natoms;
  MD_Int i;

  switch (force->model_id) {
  case POL1: case POL3: case RPOL: case SPC:
    vdwf = vdw_get_force(force->fnonbond->vdw);
    if (ES_StandardEwald == force->fnonbond->ewaldmethod) {
      elef = stdEw_get_force((struct standEwald_Tag *)
			     force->fnonbond->electro);
    } else if (ES_SPME == force->fnonbond->ewaldmethod) {
      elef = ((struct Pme_Tag *)force->fnonbond->electro)->force;
    } else {
      fprintf(stderr, "wrong ewaldmethod: %d\n", force->fnonbond->ewaldmethod);
      return MD_FAIL;
    }
    break;
  case ARGON:
    vdwf = vdw_get_force(force->fnonbond->vdw);
    break;
  default:
    fprintf(stderr, "wrong model_id: %d\n", force->model_id);
    return MD_FAIL;
  }

#ifdef DIPOLE_POLY
  elef = ((struct DPoly_Tag*)force->fnonbond->electro)->pme->force;
#endif

  /* wrap the position */
  memcpy(wrapped_pos, data->pos, natoms * sizeof(MD_Dvec));
  for (i=0; i<natoms; i++) BOUND_VEC(wrapped_pos[i], systemsize);

  /*
  if (fbond_compute(force->fbond)) {
    fprintf(stderr, "error in computing bonded forces\n");
    return MD_FAIL;
  }
  */
  if (fnonbond_compute(force->fnonbond)) {
    if (bindump_pos_vel(data, "emergency_pos_vel", 0)) {
      fprintf(stderr, "cannot dump position velocity \n");
    }
    fprintf(stderr, "failed to compute nonbonded force\n");
    return MD_FAIL;
  }

  if (NULL != vdwf && NULL != elef) {
    for (i=0; i<natoms; i++) MD_vec_add(vdwf[i], elef[i], f[i]);
    energy[PE_VDW] = vdw_get_energy(force->fnonbond->vdw);
#ifndef DIPOLE_POLY
    if (ES_StandardEwald == force->fnonbond->ewaldmethod) {
      energy[PE_ELEC] = stdEw_get_energy((struct standEwald_Tag *)
					   force->fnonbond->electro);
    } else if (ES_SPME == force->fnonbond->ewaldmethod) {
      energy[PE_ELEC] = ((struct Pme_Tag *)force->fnonbond->electro)->potential;
    }
#else 
    energy[PE_ELEC] = ((struct DPoly_Tag*)force->fnonbond->electro)->pme->potential;
#endif
  } else if (NULL != vdwf) {
    memcpy(f, vdwf, natoms*sizeof(MD_Dvec));
    energy[PE_VDW] = vdw_get_energy(force->fnonbond->vdw);
  } else if (NULL != elef) { 
    memcpy(f, elef, natoms*sizeof(MD_Dvec));
    energy[PE_ELEC] = vdw_get_energy(force->fnonbond->vdw);
  } else {
    ;  /* no other force components, empty */
  } 

  /* compute the total potential energy */
  energy[PE] = energy[PE_BOND] + energy[PE_ANGLE]
             + energy[PE_DIHED] + energy[PE_IMPR] 
             + energy[PE_ELEC] + energy[PE_VDW];

# ifdef GET_ELECTRO_DEF
    {
      static MD_Int counter = 0;
      if (counter % 100 == 0) {
	MD_String fname;
	FILE *fd;
	if (sprintf(fname, "output/dipole_energy_force_%d.dat", counter) + 1 >
	    (MD_Int)sizeof(MD_String)) {
	  fprintf(stdout, "filename is too long\n");
	  return MD_FAIL;
	}
	fd = fopen(fname, "wb");
	assert(NULL != fd);
	fwrite(data->pdipole, sizeof(MD_Double), 3*natoms, fd);
	fwrite(&(energy[PE_ELEC]), sizeof(MD_Double), 1, fd);
	fwrite(elef, sizeof(MD_Double), (size_t)(3*natoms), fd);
	fclose(fd);  

	data_bindump_image(data, counter);
      }
      counter++;
    }
# endif

  return OK;
}


void find_nearest_neibr(const struct Force_Tag *force) 
{
  const MD_Dvec *pos = force->data->pos;
  const MD_Dvec *vel = force->data->vel;
  const MD_Int natoms = force->data->natoms;
  MD_Int **excllist = force->fnonbond->excllist;
  const MD_Int *excl;
  const MD_Dvec systemsize = force->systemsize;
  MD_Dvec diff;
  MD_Double r2, r2min;
  MD_Int i,j, imin, jmin;

  r2min = MD_vec_dot(systemsize, systemsize);
  imin = jmin = -1;
  for (i = 0; i < natoms; i++) {  /* since the function is not used often */
    for (j = i+1; j < natoms; j++) {  /* I am satisfied with O(N^2) algo. */
      for (excl = excllist[i];  *excl < j;  excl++) ;
      if (j == *excl) continue;
      MD_vec_substract(pos[j], pos[i], diff);
      BOUND_VEC(diff, systemsize);
      r2 = MD_vec_dot(diff, diff);
      if (r2 < r2min) {
	r2min = r2;
	imin = i;
	jmin = j;
      }
    }
  }

  printf("nearest neibr is: (%d, %d) with distance %f\n", 
	 imin, jmin, sqrt(r2min));
  printf("atom %d: %f, %f, %f, vel: %f, %f, %f\n", imin, pos[imin].x,
	 pos[imin].y, pos[imin].z, vel[imin].x, vel[imin].y, vel[imin].z);
  printf("atom %d: %f, %f, %f, vel: %f, %f, %f\n", jmin, pos[jmin].x,
	 pos[jmin].y, pos[jmin].z, vel[jmin].x, vel[jmin].y, vel[jmin].z);

}

/* output exclusion lists*/
void output_exclusion(const struct Force_Tag *force)
{
  MD_Int *exclx;
  const MD_Int natoms = force->data->natoms;
  MD_Int i, j;

  for (i = 0;  i < natoms;  i++) {
    exclx = force->fnonbond->excllist[i];
    j = *exclx;
    printf("exclusion list for atom %d:\t", i);
    while(j < INT_MAX) {
      printf("%d, ", j);
      j = *(++exclx);
    }
    printf("\n");
  }
}


/* output force value*/
void force_output(const struct Force_Tag *force)
{
  MD_Int i;
  const MD_Int natoms = force->data->natoms;
  const MD_Dvec *f = force->f;

  printf("force \n");
  for (i = 0; i < natoms; i++) {
    printf("%d  %20.15f %20.15f %20.15f \n", i, f[i].x, f[i].y, f[i].z);
  }
  printf("\n");
}


MD_Double force_compute_volume(const struct Force_Tag *force)
{
  return force->systemsize.x * force->systemsize.y * force->systemsize.z;
}


MD_Errcode force_dump(const struct Force_Tag *force, const char* file)
{
  if (force->fnonbond) return fnonbond_dump(force->fnonbond, file);
  return OK;
}
