/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include <stdlib.h>
#include <stdio.h>
#include "electro.h"
#include "standEwald.h"
#include "pme.h"
#include "dipole_poly.h"
#include "dpoly.h"
#include "utilities.h"
#include "helper.h"



MD_Errcode electro_init(struct Electro_Tag* electro, void* parameters, 
			const enum ElectroStatic_Solver_Tag method)
{
  electro->method = method;

  switch (electro->method) {
  case ES_NotUse:
    electro->electro_ptr = NULL;
    printf("no electrostatic interaction\n");
    break;
  case ES_StandardEwald:
    electro->electro_ptr = (void*)my_calloc(1, sizeof (struct standEwald_Tag), 
				     "standEwald");
    return stdEw_init( (struct standEwald_Tag*) electro->electro_ptr, 
		       (struct standEwald_init_Tag*) parameters );
    break;
  case ES_SPME:
    electro->electro_ptr = (void*)my_calloc(1, sizeof (struct Pme_Tag), "PME");
    return pme_init( (struct Pme_Tag*) electro->electro_ptr,
		     (struct PmeParams_Tag*) parameters );
    break;
  case DIPOLE_POLY_StandEwald:
    electro->electro_ptr = (void*)my_calloc(1, sizeof (struct Dipole_Poly_Tag), 
				     "dp_standEwald");
    return dipole_poly_init( (struct Dipole_Poly_Tag*) electro->electro_ptr,
			     (struct dipole_poly_init*) parameters );    
    break;
  case DIPOLE_POLY_PME:
    electro->electro_ptr = (void*)my_calloc(1, sizeof (struct DPoly_Tag), "dp_PME");
    return dpoly_init( (struct DPoly_Tag*) electro->electro_ptr,
		       (struct PmeParams_Tag*) parameters, 0);    
    break;
  default:
    printf("%d is not a valid method\n", method);
    return MD_FAIL;
  };

  /* should not be here */
  return MD_FAIL;
}


MD_Errcode electro_destroy(struct Electro_Tag* electro)
{
  switch(electro->method) {
  case ES_NotUse:
    return OK;
  case ES_StandardEwald:
    return stdEw_destroy( (struct standEwald_Tag*) electro->electro_ptr);
  case ES_SPME:
    return pme_destroy( (struct Pme_Tag*) electro->electro_ptr);
  case DIPOLE_POLY_StandEwald:
    return dipole_poly_destroy((struct Dipole_Poly_Tag*)electro->electro_ptr);
  case DIPOLE_POLY_PME:
    return dpoly_destroy( (struct DPoly_Tag *) electro->electro_ptr);
  default:
    printf("should not reach here\n");
    return OK;
  };

  /* should not be here */
  return MD_FAIL;
}


MD_Errcode electro_compute(struct Electro_Tag* electro)
{
  switch(electro->method) {
  case ES_NotUse:
    return OK;
  case ES_StandardEwald:
    return stdEw_compute( (struct standEwald_Tag*) electro->electro_ptr);
  case ES_SPME:
    return pme_compute( (struct Pme_Tag*) electro->electro_ptr);
  case DIPOLE_POLY_StandEwald:
    return dipole_poly_compute((struct Dipole_Poly_Tag*)electro->electro_ptr);
  case DIPOLE_POLY_PME:
    return dpoly_compute( (struct DPoly_Tag *) electro->electro_ptr);
  default:
    printf("should not reach here\n");
    return MD_FAIL;
  };

  /* should not be here */
  return MD_FAIL;
}


MD_Double electro_get_energy(const struct Electro_Tag* electro)
{
  switch(electro->method) {
  case ES_NotUse:
    printf("should not be here\n");
    return 0.0;
  case ES_StandardEwald:
    return stdEw_get_energy((struct standEwald_Tag*) electro->electro_ptr);
  case ES_SPME:
    return pme_get_energy( (struct Pme_Tag *) electro->electro_ptr);
  case DIPOLE_POLY_StandEwald:
    return stdEw_get_energy( ((struct Dipole_Poly_Tag*)electro->electro_ptr)
			     ->se ); /* wrong */
  case DIPOLE_POLY_PME:
    return dpoly_get_energy( (struct DPoly_Tag *) electro->electro_ptr );
  default:
    printf("should not be here\n");
    return 0.0;
  };

  /* should not be here */
  return MD_FAIL;

}


const MD_Dvec* electro_get_force(const struct Electro_Tag* electro)
{
  switch(electro->method) {
  case ES_NotUse:    return NULL;
  case ES_StandardEwald:
    return stdEw_get_force( (struct standEwald_Tag*) electro->electro_ptr );
  case ES_SPME:
    return pme_get_force( (struct Pme_Tag *) electro->electro_ptr );
  case DIPOLE_POLY_StandEwald:
    return stdEw_get_force( ((struct Dipole_Poly_Tag*) electro->electro_ptr)->se );
  case DIPOLE_POLY_PME:
    return pme_get_force( ((struct DPoly_Tag *) electro->electro_ptr) ->pme );
  default:
    printf("wrong electrostatic method\n");
    return NULL;
  };

  /* should not be here */
  return NULL;
}


const MD_Double* electro_get_dipole(const struct Electro_Tag* electro)
{
  switch(electro->method) {
  case ES_NotUse:
    printf("should not get here\n");
    return NULL;
  case ES_StandardEwald:
    return stdEw_get_dipole( (struct standEwald_Tag*) electro->electro_ptr );
  case ES_SPME:
    return pme_get_dipole( (struct Pme_Tag *) electro->electro_ptr );
  case DIPOLE_POLY_StandEwald:
    return dipole_poly_get_dipole( (struct Dipole_Poly_Tag*) 
				   electro->electro_ptr );
  case DIPOLE_POLY_PME:
    return dpoly_get_dipole( ((struct DPoly_Tag *) electro->electro_ptr) );
  default:
    printf("invalid electrostatic solver\n");
    return NULL;
  };

  /* should not be here */
  return NULL;
}


void electro_fill_diagonal(const struct Electro_Tag* electro, MD_Double *d)
{
  switch(electro->method) {
  case ES_NotUse:
    break;
  case ES_StandardEwald:
    stdEw_fill_diagonal( (struct standEwald_Tag*) electro->electro_ptr, d);
    break;
  case ES_SPME:
    pme_fill_diagonal( (struct Pme_Tag *) electro->electro_ptr, d );
    break;
  case DIPOLE_POLY_StandEwald: /* break through */
  case DIPOLE_POLY_PME: /* break through */
  default:
    printf("should not be here\n");
    break;
  };

  /* should not be here */
  return;
}


MD_Errcode electro_dump_dipole(const struct Electro_Tag* electro, 
			       const char *filename)
{
  switch(electro->method) {
  case ES_NotUse:
    break;
  case ES_StandardEwald:
    return stdEw_dump_dipole((const struct standEwald_Tag *)electro, filename);
    break;
  case ES_SPME:
    return pme_dump_dipole((const struct Pme_Tag *)electro, filename);
    break;
  case DIPOLE_POLY_StandEwald: /* break through */
  case DIPOLE_POLY_PME: /* break through */
  default:
    printf("should not be here\n");
    break;
  };

  /* should not be here */
  return OK;
}
