/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 *
 * Modified by David Hardy to incorporate global config params.
 */

/*
 * SPC water model specification, reference:
 *
 * H.J.C. Berendsen, J.P.M. Postma, W.F.van Gunsteren, J. Hermans
 * Intermolecular Forces, Edited by Bernard Pullman, page 331
 *
 * Contains init, destroy, and validate routines for simen 'Data'
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "data.h"
#include "force.h"
#include "helper.h"
#include "random.h"
#include "unit.h"
#include "constant.h"

/*
#include "helper.h"
#include "utilities.h"
*/

/*
 * SPC water model, units are NOT in internal unit yet 
 */
static MD_Double H_CHARGE = 0.41;    /* unit: electron charge */
static MD_Double O_CHARGE = -0.82; 
static MD_Double BOND_LEN = 1.0;     /* unit: angstrom */
static MD_Double BOND_ANGLE = ((109.0 + 28.0/60.0) * Pi / 180.0); 
     /* 109 degree 28' */
static MD_Double VDW_A = 3.7122;    /* unit: angstrom * (kJoule/mol)^{1/6} */
static MD_Double VDW_B = 3.428;     /* unit: angstrom * (kJoule/mol)^{1/12} */
static const char *Oname = "O";
static const char *Hname = "H";

/*
 * Exclusion strings
 */
static const char *EXCL_STRING[] = {
  "none",
  "1-2",
  "1-3",
  "1-4",
  "scaled1-4"
};

/* 
 * local functions 
 */

 /* initialize data structures, return the amount of allocated 
 * memory in bytes */
static MD_Errcode spc_data_init_value(struct Data_Tag * data);
static MD_Int init_atomprm(MD_Atom_Param **atomprm, const MD_Int natomprms);
static MD_Int init_atom(MD_Atom **patom, const MD_Int Nmols);
static MD_Int init_position(MD_Dvec **pos, 
			    const MD_Double bond_len, 
			    const MD_Double bond_angle,
			    const MD_Double Len, 
			    const MD_Int Nmol_Line, 
			    const MD_Int Natoms);
static MD_Int init_velocity(MD_Dvec **pvel, const MD_Int Natoms, 
			    const MD_Int Nmols, const MD_Double Temperature);
static MD_Int init_excllist(MD_Excl **pexcl, const MD_Int Nexcl, 
			    const MD_Int Nmols);


MD_Errcode spc_data_init(struct Data_Tag *data)
{
  assert(NULL != data);
  data->model_id = SPC;
  
  printf("SPC water model\n");

  /* allocate memory for "force" */
  data->force=my_calloc((size_t)1, sizeof(struct Force_Tag), "force module");

  printf("  spc_data_init allocates %d bytes for Force\n",
         (MD_Int) sizeof(struct Force_Tag));

  /* initialize simulation parameters */
  if (-1 == data->temperature) data->temperature = DEFAULT_SPC_TEMPERATURE;
  if (-1 == data->timestep) data->timestep = DEFAULT_SPC_TIMESTEP;
  if (-1 == data->switchdist_vdw) data->switchdist_vdw = DEFAULT_SPC_SWITCHDIST;
  if (-1 == data->cutoff_vdw) data->cutoff_vdw = DEFAULT_SPC_CUTOFF;
  if (-1 == data->dielectric) data->dielectric = DEFAULT_DIELECTRIC;
  if (-1 == data->scaling14) data->scaling14 = DEFAULT_SCALING14;

  data->is_switching = 1;

  return spc_data_init_value(data);
}



/* 
 * note that all values are 0 initially 
 * assume called after spc_data_init
 */
MD_Errcode spc_data_init_value(struct Data_Tag * data)
{
  const MD_Int Nmol_Line = data->nmolPerLine;  /* # of molecules per line */
  /* total number of water molecules in the system */
  const MD_Int Nmols = Nmol_Line * Nmol_Line * Nmol_Line; 
  const MD_Int Natoms = Nmols * 3; /* total number of atoms */
  const MD_Double givenDensity = 1.0;  /* in g/cm^3 */
  const MD_Double H2O_mass = H_MASS * 2.0 + O_MASS; /* unit: amu */
  /* Density: # of H2O molecules per angstrom^3  */
  const MD_Double Density = givenDensity * 1e-3 / (1e-6*METER*METER*METER)
                            / (ATOMIC_MASS * H2O_mass) ; 

  const MD_Double Temperature = data->temperature;  /* Kelvin , not used yet */

  /* system size is Len x Len x Len */
  const MD_Double Len = (MD_Double)pow(Nmols / Density, 1.0 / 3.0);   /* angstrom */

  MD_Int alloc_size = 0;

  printf("  density = %f g/cm^3\n", givenDensity);

  data->systemsize.x = data->systemsize.y = data->systemsize.z = Len;

  /* 
   * init atom parameter array for vdw computation 
   */
  data->natomprms = 2;  /* only two types of atoms: oxygen and hydrogen */
  alloc_size += init_atomprm(&(data->atomprm), data->natomprms);

  /* 
   * init atom charge, mass, type, name (data->atom)
   */
  data->natoms = Natoms;
  alloc_size += init_atom(&(data->atom), Nmols);
  printf("  system has %d atoms.\n", data->natoms);

  /* SPC water model spcific */
  data->bond_len = BOND_LEN;
  data->bond_angle = BOND_ANGLE; 

  /* 
   * init position and velocity, the overall linear momentum is 0
   */
  if (data->fromRestartFile) {
    MD_Int size;
    printf("Restart, read position and velocity from file \"%s\"\n",
        data->fromRestartFile);
    size = binread_pos_vel(&(data->pos), &(data->vel),
                        Natoms, data->fromRestartFile, &(data->firststepnum));
    {  /* unapply PBC */
      MD_Int i, nmols = Natoms / 3;
      MD_Dvec systemsize;
      systemsize.x = systemsize.y = systemsize.z = Len;
      for (i = 0; i < nmols; i++) { /* fix Oxygen atom */
        MD_Dvec *p0, *p1, *p2, diff;
        MD_Double ratio;
        p0 = data->pos + 3*i;
        p1 = data->pos + 3*i+1;
        p2 = data->pos + 3*i+2;
        MD_pvec_substract(p0, p1, (&diff));
        ratio = diff.x / systemsize.x;  p1->x += ANINT(ratio) * systemsize.x;
        ratio = diff.y / systemsize.y;  p1->y += ANINT(ratio) * systemsize.y;
        ratio = diff.z / systemsize.z;  p1->z += ANINT(ratio) * systemsize.z;
        MD_pvec_substract(p0, p2, (&diff));
        ratio = diff.x / systemsize.x;  p2->x += ANINT(ratio) * systemsize.x;
        ratio = diff.y / systemsize.y;  p2->y += ANINT(ratio) * systemsize.y;
        ratio = diff.z / systemsize.z;  p2->z += ANINT(ratio) * systemsize.z;
      }
    }
    if (size < 0) {
      return MD_FAIL;
    } else {
      alloc_size += size;
    }
  } else {
    printf("to initilize position velocity (not reading from files) \n");
    alloc_size += init_position(&(data->pos), data->bond_len, 
				data->bond_angle, Len, Nmol_Line, Natoms);
    alloc_size += init_velocity(&(data->vel), Natoms, Nmols, Temperature);
  } 


  /* 
   * init explicit direct exclusion list
   * in each molecule, O-H1, O-H2, H1-H2 are exclusion pairs.
   */
  data->nexcls = Nmols * 3;
  alloc_size += init_excllist(&(data->excl), data->nexcls, Nmols);

#if 0
  data->cutoff_vdw = 8.5;  /* from THE paper */
  data->switchdist_vdw = 6.0;
#endif

  if (spc_data_validate_simparam(data)) {
    fprintf(stderr,"*** invalid parameter \n");
    return MD_FAIL;
  }

  printf("  data_init_value allocate memory: %d bytes\n", alloc_size);

  /* 3 constraint per molecule, and the total linear momentum is fixed */
  data->degree_freedom = (3-1) * data->natoms - 3;

  return OK; 
}



MD_Int init_atomprm(MD_Atom_Param **atomprm, const MD_Int natomprms) 
{
  /* from the paper */

  *atomprm = my_calloc((size_t)natomprms, sizeof(MD_Atom_Param), "atomrpm");
  (*atomprm)[0].emin = -pow(VDW_A, 12.0)/(4.0*pow(VDW_B, 12.0)) * 
                       KJOULE_PER_MOL;
  (*atomprm)[0].rmin =  pow(2.0, 1.0/6.0) * VDW_B * VDW_B / VDW_A;  
  strncpy((*atomprm)[0].type, Oname, sizeof(MD_Short_String)-1);
  (*atomprm)[1].emin = 0.0;   
  (*atomprm)[1].rmin = (*atomprm)[0].rmin;  /* does not matter */
  strncpy((*atomprm)[1].type, Hname, sizeof(MD_Short_String)-1); 

  return natomprms * sizeof(MD_Atom_Param);
}


MD_Int init_atom(MD_Atom **patom, const MD_Int Nmols)
{
  MD_Atom *atom=NULL;
  const MD_Int natoms = Nmols * 3;
  MD_Int i;

  atom = *patom = my_calloc((size_t)natoms, sizeof(MD_Atom), "atom");

  for (i = 0; i < Nmols; i++) {
    atom[3*i].m = O_MASS;  /* amu */
    atom[3*i].q = O_CHARGE;   /* electron charge */
    atom[3*i].prm = 0;    /* paramter type */
    strncpy(atom[3*i].name, Oname, sizeof(MD_Short_String) - 1);
    strncpy(atom[3*i].type, Oname, sizeof(MD_Short_String) - 1); /* same as Atom_Param's type */
    atom[3*i+1].m = H_MASS;
    atom[3*i+1].q = H_CHARGE;    /* electron charge */
    atom[3*i+1].prm = 1;       /* paramter type  */
    strncpy(atom[3*i+1].name, Hname, sizeof(MD_Short_String) - 1); 
    strncpy(atom[3*i+1].type, Hname, sizeof(MD_Short_String) - 1);  /* same as Atom_Param's type */ 
    atom[3*i+2].m = H_MASS;
    atom[3*i+2].q = H_CHARGE; 
    atom[3*i+2].prm = 1; 
    strncpy(atom[3*i+2].name, Hname, sizeof(MD_Short_String) - 1);  
    strncpy(atom[3*i+2].type, Hname, sizeof(MD_Short_String) - 1);  /* same as Atom_Param's type */
  }

  return natoms * sizeof(MD_Atom);
}

 
MD_Int init_position(MD_Dvec **ppos, 
		     const MD_Double bond_len, 
		     const MD_Double bond_angle,
		     const MD_Double Len, 
		     const MD_Int Nmol_Line, 
		     const MD_Int Natoms)
{
  const MD_Double Half_angle = 0.5 * bond_angle;
  const MD_Double Neg_half_len = - 0.5 * Len;
  const MD_Double Dl = Len / Nmol_Line;

  MD_Dvec *pos=NULL, *oxygen=NULL, *h1=NULL, *h2=NULL;
  MD_Double ty, tz;
  MD_Int ix, iy, iz;

  pos = *ppos = my_calloc((size_t)Natoms, sizeof(MD_Dvec), "position");

  oxygen = pos;
  for (iz = 0; iz < Nmol_Line; iz++) {
    tz = Neg_half_len + (iz + 0.5) * Dl;
    for (iy = 0; iy < Nmol_Line; iy++) {
      ty = Neg_half_len + (iy + 0.5) * Dl;
      for (ix = 0; ix < Nmol_Line; ix++) {
	oxygen->x = Neg_half_len + (ix + 0.5) * Dl;
	oxygen->y = ty;
	oxygen->z = tz;
	h1 = oxygen + 1;
	h2 = oxygen + 2;
	h1->x = h2->x = oxygen->x - bond_len * cos(Half_angle);
	h1->y = oxygen->y - bond_len * sin(Half_angle);
	h2->y = oxygen->y + bond_len * sin(Half_angle);
	h1->z = h2->z = oxygen->z;   /* same z plane */
	oxygen += 3;  /* next oxygen atom */
      }
    }
  }

  return Natoms * sizeof(MD_Dvec);
}


MD_Int init_velocity(MD_Dvec **pvel, const MD_Int Natoms, 
		     const MD_Int Nmols, const MD_Double Temperature)
{
  const MD_Double H2O_mass = H_MASS * 2.0 + O_MASS;
  const MD_Double sv = sqrt(KELVIN * Temperature / H2O_mass);
  MD_Dvec sum_v={0.0,0.0,0.0}; /* sum of linear momentum */
  MD_Dvec *vel=NULL;
  MD_Double tmp;
  MD_Int i;

  if (Nmols * 3 != Natoms) {
    fprintf(stderr,"*** %s: line %d, wrong parameters:\n", __FILE__, __LINE__);
    fprintf(stderr,"   Nmols=%d, Natoms=%d\n", Nmols, Natoms);
  }

  vel = *pvel = my_calloc((size_t)Natoms, sizeof(MD_Dvec), "velocity");
  for (i = 0; i < Nmols; i++) {  
    sum_v.x += (vel[i*3].x = sv * rannyu_normal());
    sum_v.y += (vel[i*3].y = sv * rannyu_normal());
    sum_v.z += (vel[i*3].z = sv * rannyu_normal());
  }  
  sum_v.x /= (MD_Double) Nmols; 
  sum_v.y /= (MD_Double) Nmols; 
  sum_v.z /= (MD_Double )Nmols; 
  for (i = 0; i < Nmols; i++) { 
    tmp = vel[i*3].x - sum_v.x;
    vel[i*3].x = vel[i*3+1].x = vel[i*3+2].x = tmp;
    tmp = vel[i*3].y - sum_v.y;
    vel[i*3].y = vel[i*3+1].y = vel[i*3+2].y = tmp;
    tmp = vel[i*3].z - sum_v.z;
    vel[i*3].z = vel[i*3+1].z = vel[i*3+2].z = tmp;
  } 

  return Natoms * sizeof(MD_Dvec);
}


MD_Int init_excllist(MD_Excl **pexcl, const MD_Int Nexcl, const MD_Int Nmols)
{
  MD_Excl *excl=NULL;
  MD_Int i;
  const MD_Int ex_in_mol = 3;

  if (Nmols * ex_in_mol != Nexcl) {
    fprintf(stderr,"*** %s: line %d, wrong parameter:  ", __FILE__, __LINE__);
    fprintf(stderr,"Nmols=%d, Nexcl=%d\n", Nmols, Nexcl);
  }

  excl = *pexcl = my_calloc((size_t)Nexcl, sizeof(MD_Excl), "exclusion list");
  for (i = 0; i < Nmols; i++) {
    excl[ex_in_mol*i  ].atom[0] = 3*i;   /* O-H1 */
    excl[ex_in_mol*i  ].atom[1] = 3*i+1; 
    excl[ex_in_mol*i+1].atom[0] = 3*i;   /* O-H2 */
    excl[ex_in_mol*i+1].atom[1] = 3*i+2; 
    excl[ex_in_mol*i+2].atom[0] = 3*i+1; /* H1-H2 */
    excl[ex_in_mol*i+2].atom[1] = 3*i+2; 
  }
  return sizeof(MD_Excl) * Nexcl;
}



/*
 * destroy Data.
 */
void spc_data_destroy(struct Data_Tag *data)
{
  fprintf(stderr, "to free data->force\n");
  if (data->force) {
    if (force_destroy(data->force)) {
      fprintf(stderr, "failed to destroy force\n");
      return;
    }
  }
  free(data->force);
  fprintf(stderr, "to free memory allocated for data\n");
  if (data->atomprm) free(data->atomprm);
  if (data->bondprm) free(data->bondprm);
  if (data->angleprm) free(data->angleprm);
  if (data->dihedprm) free(data->dihedprm);
  if (data->imprprm) free(data->imprprm);
  if (data->nbfixprm) free(data->nbfixprm);
  if (data->atom) free(data->atom);
  if (data->bond) free(data->bond);
  if (data->angle) free(data->angle);
  if (data->dihed) free(data->dihed);
  if (data->impr) free(data->impr);
  if (data->excl) free(data->excl);
  if (data->pos) free(data->pos);
  if (data->vel) free(data->vel);
  if (data->polarizability) free(data->polarizability);

  return;
}


/*
 * validate the simulation parameters
 */
MD_Errcode spc_data_validate_simparam(struct Data_Tag *data)
{
  MD_Int k;

  if (data->dielectric < 1.0) {
    printf("spc_data_validate_simparam: "
           "dielectric constant should be >= 1");
    return MD_FAIL;
  }
  if (data->scaling14 > 1.0 || data->scaling14 < 0.0) {
    printf("spc_data_validate_simparam: "
           "1-4 scaling should be between 0 and 1 inclusive");
    return MD_FAIL;
  }
  if (data->switchdist_vdw <= 0.0) {
    printf("spc_data_validate_simparam: "
           "switchdist should be > 0");
    return MD_FAIL;
  }
  if (data->cutoff_vdw <= 0.0) {
    printf("spc_data_validate_simparam: "
           "cutoff should be > 0");
    return MD_FAIL;
  }
  if (data->switchdist_vdw > data->cutoff_vdw) {
    printf("spc_data_validate_simparam: "
           "switchdist should be <= cutoff");
    return MD_FAIL;
  }

  /* set exclusion policy */
  data->excl_policy = -1;
  for (k = EXCL_NONE;  k < NEXCLS;  k++) {
    if (my_strcasecmp(data->exclude, EXCL_STRING[k]) == 0) {
      data->excl_policy = k;
    }
  }
  if (data->excl_policy < 0) {
    printf("argon_data_validate_simparam: unrecognized exclusion policy.");
    return MD_FAIL;
  }

  return OK;
}



