/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */

/*
 * data_common.c
 *
 * Contains init, destroy, and validate routines for simen 'Data'
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "data.h"
#include "force.h"
#include "unit.h"
#include "helper.h"
#include "utilities.h"

#ifdef DIPOLE_POLY
#include "dipole_poly.h"
#endif

Model_Data_Type data_routines[NMODELS] = {
  {argon_data_init,argon_data_destroy,argon_data_validate_simparam,"Argon"},
  {spc_data_init,  spc_data_destroy,  spc_data_validate_simparam,  "SPC"},
  {rpol_data_init, rpol_data_destroy, rpol_data_validate_simparam, "RPOL"},
  {pol3_data_init, pol3_data_destroy, pol3_data_validate_simparam, "POL3"},
  {pol1_data_init, pol1_data_destroy, pol1_data_validate_simparam, "POL1"},
};


void output_data(const struct Data_Tag *data)
{
  MD_Int i;

  printf("There are %d atoms\n", data->natoms);
  printf("atom type   mass    charge\n");
  for (i = 0; i < data->natoms; i++) {
    printf("%4d %s %9.5f %9.5f \n", i, data->atom[i].type,
	   data->atom[i].m, data->atom[i].q);
  }
  printf("atom              position                     velocity\n");
  for (i = 0; i < data->natoms; i++) {
    printf("%4d  (%9.4f,%9.4f,%9.4f) (%9.4f,%9.4f,%9.4f)\n", i,
	   data->pos[i].x, data->pos[i].y, data->pos[i].z,
	   data->vel[i].x, data->vel[i].y, data->vel[i].z);
  }
  printf("atom        polarizability\n");
  for (i = 0; i < data->natoms; i++) {
    printf("%4d  (%9.4f,%9.4f,%9.4f)\n", i,
	   data->polarizability[3*i], data->polarizability[3*i+1], 
	   data->polarizability[3*i+2]);
  } 
  printf("exclusion pairs\n");
  if (NULL == data->excl) {
    printf("nothing\n");
  } else {
    for (i = 0; i < data->nexcls; i+=6) {
      printf("(%4d,%4d) ", data->excl[i  ].atom[0], data->excl[i  ].atom[1]);
      if (i+1 < data->nexcls)
	printf("(%4d,%4d) ", data->excl[i+1].atom[0], data->excl[i+1].atom[1]);
      if (i+2 < data->nexcls)
	printf("(%4d,%4d) ", data->excl[i+2].atom[0], data->excl[i+2].atom[1]);
      if (i+3 < data->nexcls)
	printf("(%4d,%4d) ", data->excl[i+3].atom[0], data->excl[i+3].atom[1]);
      if (i+4 < data->nexcls)
	printf("(%4d,%4d) ", data->excl[i+4].atom[0], data->excl[i+4].atom[1]);
      if (i+5 < data->nexcls)
	printf("(%4d,%4d) ", data->excl[i+5].atom[0], data->excl[i+5].atom[1]);
      printf("\n");
    }
    printf("\n");
  }
  return;
}


void compute_KETE(struct Data_Tag *data)
{
  MD_Atom *atom = data->atom;
  MD_Double *energy = data->energy;
  MD_Dvec *vel = data->vel; 
  MD_Double ke = 0.0;
  MD_Int natoms = data->natoms;
  MD_Int i;

  for(i = 0; i < natoms; i++) {
    ke += MD_vec_dot(vel[i], vel[i]) * atom[i].m;
  }

  energy[KE] = ke * 0.5;
  energy[TE] = energy[KE] + energy[PE];

  return;
}


MD_Double compute_KE(const MD_Dvec *vel, const MD_Atom *atom, 
		     const MD_Int natoms)
{
  MD_Double ke = 0.0;
  MD_Int i;

  for(i = 0; i < natoms; i++) {
    ke += MD_vec_dot(vel[i], vel[i]) * atom[i].m;
  }

  return ke*0.5;
}




MD_Int conserve_linear_momentum(struct Data_Tag *data)
{
  MD_Dvec *vel = data->vel;
  const MD_Int natoms = data->natoms;
  const MD_Atom *atom = data->atom;
  MD_Dvec lm_sum = {0.0, 0.0, 0.0}; /* sum of linear momentum */
  MD_Double mass = -1.0;
  MD_Int i;

  for(i = 0; i < natoms; i++) {
    mass = atom[i].m;
    MD_vec_mul_add(lm_sum, vel[i], mass)
  }

  if (MD_vecLen(lm_sum) > 1e-6) {
    fprintf(stderr, "linear momentum is not conserved: \n");
    fprintf(stderr, "    sum of linear momentum: (%g, %g, %g)\n",
	    lm_sum.x, lm_sum.y, lm_sum.z);
    return 0;
  }

  return 1;
}


/*  store system cofiguration, used for resuming MD. */
void dump_pos_vel(const struct Data_Tag *data, const char *filename, 
		  const MD_Int nstep)
{
  const MD_Dvec *pos = data->pos;
  const MD_Dvec *vel = data->vel;
  const MD_Int natoms = data->natoms;
  MD_Int i;
  FILE *fd = fopen(filename, "w");

  if (NULL == fd) {
    printf("cannot open file %s to output postion and velocity.\n", filename);
    return;
  }

  printf("dump position and velocity information to file %s\n", filename);
  fflush(NULL);

  fprintf(fd, "%d\t%d\n", nstep, natoms); 
  fprintf(fd, "position\n"); 
  for (i = 0; i < natoms; i++) {
#ifdef DEBUG_POS_VEL
    printf("position %d\n", i); fflush(NULL);
#endif
    fprintf(fd, "%5d %20.15f %20.15f %20.15f\n", 
	    i, pos[i].x, pos[i].y, pos[i].z);
  }
  fprintf(fd, "velocity\n");
  for (i = 0; i < natoms; i++) {
#ifdef DEBUG_POS_VEL
    printf("velocity %d\n", i); fflush(NULL);
#endif
    fprintf(fd, "%5d %20.15f %20.15f %20.15f\n", 
	    i, vel[i].x, vel[i].y, vel[i].z);
  }
  fclose(fd);
}

MD_Errcode read_pos_vel(MD_Dvec **ppos,  
			MD_Dvec **pvel, const MD_Int natoms,
			MD_Char *filename, MD_Int *pnstep)
{
  MD_Dvec *pos=NULL, *vel=NULL;
  MD_Int i, j;
  FILE *fd = fopen(filename, "r");
  char name[9]; /* length of "position", or "velocity" */
  const char *format = (sizeof(MD_Double) == sizeof(double) ? 
			"%d %lf %lf %lf": "%d %f %f %f");

  /* allocate space */
  pos = *ppos = my_calloc((size_t)natoms, sizeof(MD_Dvec), "position");

  vel = *pvel = my_calloc((size_t)natoms, sizeof(MD_Dvec), "velocity");


  fscanf(fd, "%d %d", pnstep, &i);
  if (0 != i - natoms) {
    fprintf(stderr,"*** natoms=%d, while input file has %d atoms\n", 
	    natoms, i);
    fclose(fd);
    return -1;   /* not approapriate value yet */
  }

#ifdef DEBUG_INIT_DATA
  printf("position\n");
#endif
  fscanf(fd, "%s", name); /* read in string "position" */
  for (i = 0; i < natoms; i++) {
    if (4 != fscanf(fd, format, &j, &pos[i].x, &pos[i].y, 
		    &pos[i].z)) {
      fclose(fd);
      return MD_FAIL; 
    }; 
    if (0 != j-i) { 
      fclose(fd);
      fprintf(stderr,"*** wrong input file expect %d, get %d\n", i,j); 
      return MD_FAIL; 
    } 
#ifdef DEBUG_INIT_DATA
    printf("%d %f %f %f\n", j, pos[j].x, pos[j].y, pos[j].z);
#endif
  }

#ifdef DEBUG_INIT_DATA
  printf("velocity\n");
#endif
  fscanf(fd, "%s", name);  /* read in string "velocity" */
  for (i = 0; i < natoms; i++) {
    if (4 != fscanf(fd, format, &j, &vel[i].x, &vel[i].y, 
		    &vel[i].z)) {
      fclose(fd);
      return MD_FAIL;
    };
    if (0 != j-i) { 
      fclose(fd);
      fprintf(stderr,"*** wrong input file expect %d, get %d\n", i,j); 
      return MD_FAIL; 
    } 
#ifdef DEBUG_INIT_DATA
    printf("%d %f %f %f\n", j, vel[j].x, vel[j].y, vel[j].z);
#endif
  }

  fclose(fd);

  return 3 * natoms * sizeof(MD_Dvec);
}



/*
#define DEBUG_POS_VEL
*/

MD_Errcode bindump_pos_vel(const struct Data_Tag *data, const char *filename, 
			   const MD_Int nstep)
{
  const MD_Dvec *pos = data->pos;
  const MD_Dvec *vel = data->vel;
  const MD_Int natoms = data->natoms;
  FILE *fd = fopen(filename, "wb");
  size_t noutput;

  if (NULL == fd) {
    fprintf(stderr, "cannot open file %s to output postion and velocity.\n", 
	    filename);
    return MD_FAIL;
  }

#ifdef DEBUG_POS_VEL
  printf("dump position and velocity information to file %s\n", filename);
#endif

  noutput = fwrite(&nstep,  sizeof(nstep),  (size_t) 1, fd);  
  noutput+= fwrite(&natoms, sizeof(natoms), (size_t) 1, fd); 
  /* this implementation depends on the fact that there is no patch
   * inside the MD_Dvec structure, and the size of MD_Dvec is right
   * so that there is no memory holes between consecutive elements 
   * of MD_Dvec array */
  noutput+= fwrite(pos, sizeof(MD_Dvec), (size_t) natoms, fd);
  noutput+= fwrite(vel, sizeof(MD_Dvec), (size_t) natoms, fd);
  if ((size_t)(2 + natoms*2) != noutput || ferror(fd) ) {
    perror("error writing file");
    return MD_FAIL;
  }

  if (fclose(fd)) {
    fprintf(stderr, "cannot close file %s\n", filename);
    return MD_FAIL;
  }

  return OK;
}


MD_Errcode binread_pos_vel(MD_Dvec **ppos, 
			   MD_Dvec **pvel, const MD_Int natoms,
			   const MD_Char *filename, MD_Int *pnstep)
{
  MD_Dvec *pos=NULL, *vel=NULL;
  MD_Int i;
  FILE *fd = fopen(filename, "rb");
  size_t ninput;

  /* allocate space */
  pos = *ppos = my_calloc((size_t)natoms, sizeof(MD_Dvec), "pos");
  vel = *pvel = my_calloc((size_t)natoms, sizeof(MD_Dvec), "velocity");

  ninput = fread(pnstep, sizeof(MD_Int), (size_t) 1, fd);
  ninput+= fread(&i, sizeof(MD_Int), (size_t) 1, fd);
  if (0 != i - natoms) {
    fprintf(stderr,"*** natoms=%d, while input file has %d atoms\n", 
	    natoms, i);
    fclose(fd);
    return -1;   
  }


  ninput+= fread(pos, sizeof(MD_Dvec), (size_t)natoms, fd);
  ninput+= fread(vel, sizeof(MD_Dvec), (size_t)natoms, fd);

  if ((size_t)(2+2*natoms) != ninput || ferror(fd)) {
    perror("error in reading in pos_vel file");
    return -1;
  }

  if (fclose(fd)) {
    perror("failed to close file\n");
    return -1;
  }

  return 3 * natoms * sizeof(MD_Dvec);
}


void data_print_energy(struct Data_Tag *data, const MD_Int istep)
{
  static const MD_Double inv_Kcal_per_mol = 1.0 / KCAL_PER_MOL;
  static MD_Int firstime = 1;

  if (firstime) { /* print header */
    firstime = 0;
    printf("---- all energy unit is kcal/mol\n");
    if (POL1==data->model_id || POL3==data->model_id || 
	RPOL==data->model_id || SPC==data->model_id) {
      printf("    step          vdw               elec              potential"
             "          kinE            total_E \n");
    } else if (data->model_id - ARGON == 0) {
      printf("    step        vdw energy      kinetic energy  total_E\n");
    }
  }

  if (POL1==data->model_id || POL3==data->model_id || 
      RPOL == data->model_id || SPC == data->model_id) {
    printf("== %8d %15.12f %15.12f %15.12f %15.12f %15.12f\n",
           istep,
           data->energy[PE_VDW] * inv_Kcal_per_mol,
           data->energy[PE_ELEC] * inv_Kcal_per_mol,
           data->energy[PE] * inv_Kcal_per_mol,
           data->energy[KE] * inv_Kcal_per_mol,
           data->energy[TE] * inv_Kcal_per_mol);
  } else if (ARGON == data->model_id) {
    printf("== %8d %15.12f %15.12f %15.12f \n",
           istep,
           data->energy[PE_VDW] * inv_Kcal_per_mol,
           data->energy[KE] * inv_Kcal_per_mol,
           data->energy[TE] * inv_Kcal_per_mol);
  };

}



MD_Errcode data_bindump_image(struct Data_Tag *data, 
			      const MD_Int istep)
{
  char filename[MAX_FILENAME_LEN]={0};
  int retval;

  /* save pos and vel arrays */
  retval = snprintf(filename, sizeof(filename), "%s_%d.dat",
      data->restartname, istep);
  if (retval <= 0) {
    fprintf(stdout, "cannot compose filename string\n");
    return MD_FAIL;
  }
  else if (retval >= (int)sizeof(filename)) { 
    fprintf(stdout, "filename is too long for buffer\n");
    return MD_FAIL;
  }

  fprintf(stdout, "saving restart file \"%s\"\n", filename);
  if (bindump_pos_vel(data, filename, istep)) {
    fprintf(stdout, "cannot output %s\n", filename);
  }

  /* save dipole array */
  retval = snprintf(filename, sizeof(filename), "%s_%d.dat",
      data->forcedumpname, istep);
  if (retval <= 0) {
    fprintf(stdout, "cannot compose filename string\n");
    return MD_FAIL;
  }
  else if (retval >= (int)sizeof(filename)) { 
    fprintf(stdout, "filename is too long for buffer\n");
    return MD_FAIL;
  }

  fprintf(stdout, "saving dipoles file \"%s\"\n", filename);
  if (force_dump(data->force, filename)) {
    fprintf(stdout, "failed to dump dipoles\n");
    return MD_FAIL;
  }

  if (0 == data->restartsave && 0 != data->prevsavestep) {

    /* remove previous restart */
    retval = snprintf(filename, sizeof(filename), "%s_%d.dat",
        data->restartname, data->prevsavestep);
    if (retval <= 0) {
      fprintf(stdout, "cannot compose filename string\n");
      return MD_FAIL;
    }
    else if (retval >= (int)sizeof(filename)) { 
      fprintf(stdout, "filename is too long for buffer\n");
      return MD_FAIL;
    }

    fprintf(stdout, "removing previous restart file \"%s\"\n", filename);
    if (remove(filename)) {
      fprintf(stdout, "unable to remove previous restart file \"%s\"\n",
          filename);
    }

    /* remove previous dipole dump */
    retval = snprintf(filename, sizeof(filename), "%s_%d.dat",
        data->forcedumpname, data->prevsavestep);
    if (retval <= 0) {
      fprintf(stdout, "cannot compose filename string\n");
      return MD_FAIL;
    }
    else if (retval >= (int)sizeof(filename)) { 
      fprintf(stdout, "filename is too long for buffer\n");
      return MD_FAIL;
    }

    fprintf(stdout, "removing previous dipoles file \"%s\"\n", filename);
    if (remove(filename)) {
      fprintf(stdout, "unable to remove previous dipole file \"%s\"\n",
          filename);
    }
  }

  /* save current step number */
  data->prevsavestep = istep;

  return OK;
}



/* pressure = (p^T M^{-1}p - r^T f) / (dV) */
MD_Double compute_pressure(const MD_Dvec *force, 
			   const MD_Dvec *wrapped_pos, 
			   const MD_Dvec *vel,
			   const MD_Atom *atom,
			   const MD_Double volume, 
			   const MD_Int natoms)
{
  MD_Int i;
  MD_Double pressure = 0.0;

  for (i = 0; i < natoms; i++) {
    pressure += MD_vec_dot(vel[i], vel[i]) * atom[i].m 
      + MD_vec_dot(wrapped_pos[i], force[i]);
  }

  return pressure/(DIMENSION * volume);
}


void compute_mol_sys_dipole(const MD_Atom *atom,
			    const MD_Dvec realpos[],
			    const MD_Double idipole[], /* induced dipole */
			    const MD_Int natoms,
			    MD_Double *mol_perm_dipole,
			    MD_Double *mol_induc_dipole, 
			    MD_Double *mol_total_dipole,
			    MD_Dvec *system_total_dipole) 
{
  const MD_Int nmols = natoms / 3;  /* water */
  const MD_Double inv_nmols = 1.0 / (MD_Double) nmols;
  const MD_Double inv_DEBYE = 1.0 / DEBYE;
  MD_Dvec mpd, mid, mtd; /* molecule (permanent | induced | total) dipole */ 
  MD_Dvec sys_totd;
  MD_Double mol_permd, mol_induced, mol_totd;
  MD_Int imol, io, ih1, ih2;

  mol_permd = 0.0;
  mol_induced = 0.0;
  mol_totd = 0.0;
  mid.x = mid.y = mid.z = 0.0; /* must in case there is no dipole */
  sys_totd.x = sys_totd.y = sys_totd.z = 0.0;
  for (imol = 0; imol < nmols; imol++) {
    io = imol*3; ih1 = io+1; ih2 = io+2;
    mpd.x = atom[io ].q * realpos[io ].x 
          + atom[ih1].q * realpos[ih1].x
          + atom[ih2].q * realpos[ih2].x;
    mpd.y = atom[io ].q * realpos[io ].y
          + atom[ih1].q * realpos[ih1].y
          + atom[ih2].q * realpos[ih2].y;
    mpd.z = atom[io ].q * realpos[io ].z 
          + atom[ih1].q * realpos[ih1].z
          + atom[ih2].q * realpos[ih2].z;
    mol_permd += sqrt(MD_vec_dot(mpd, mpd));
    if (NULL != idipole) {
      io = imol*9; ih1 = io+3; ih2 = io+6;
      mid.x = idipole[io+X] + idipole[ih1+X] + idipole[ih2+X];
      mid.y = idipole[io+Y] + idipole[ih1+Y] + idipole[ih2+Y];
      mid.z = idipole[io+Z] + idipole[ih1+Z] + idipole[ih2+Z];
      mol_induced += sqrt(MD_vec_dot(mid, mid));
    }
    MD_vec_add(mpd, mid, mtd);
    MD_vec_add(sys_totd, mtd, sys_totd);
    mol_totd += sqrt(MD_vec_dot(mtd, mtd));
  }
  mol_permd   *= inv_nmols;
  mol_induced *= inv_nmols;
  mol_totd    *= inv_nmols;

  *mol_perm_dipole  = mol_permd * inv_DEBYE;
  *mol_induc_dipole = mol_induced * inv_DEBYE;
  *mol_total_dipole = mol_totd * inv_DEBYE;
  MD_vec_mul(sys_totd, inv_DEBYE, (*system_total_dipole));

}
