/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include <assert.h>
#include <stdlib.h>
#include "data_collector.h"
#include "dipole_poly.h"
#include "unit.h"

static const char *avgvar_names[N_mda_av] = 
{
  "kineticE_per_dof(Kelvin)", 
  "potentialE_per_mol(Kcal/Mol)",
  "total_dipole_sqr(Debye^2)", 
  "total_dipole_x(Debye)", 
  "total_dipole_y(Debye)", 
  "total_dipole_z(Debye)",
  "molecule_permanent_dipole(Debye)",
  "molecule_induced_dipole(Debye)",
  "molecule_total_dipole(Debye)",
  /* "pv/nkT (ideal gas = 1)", *//* comma is important */
};


static const char *linearfit_names[N_mda_lf] = {"energy"};

static void argon_init(struct Data_Collector_Tag *dc);
static void spc_init(struct Data_Collector_Tag *dc);
static void rpol_init(struct Data_Collector_Tag *dc);
typedef void (*init_function_type) (struct Data_Collector_Tag *dc);

static void argon_update(struct Data_Collector_Tag *dc, MD_Int istep);
static void spc_update(struct Data_Collector_Tag *dc, MD_Int istep);
static void rpol_update(struct Data_Collector_Tag *dc, MD_Int istep);
typedef void (*update_function_type) (struct Data_Collector_Tag *dc, 
				      MD_Int istep);

/*static*/ 
struct model_functions_tag {
    init_function_type init_function;
  update_function_type update_function;
};

/* the order should be consistent with enum Model_Types_Tag */
static struct model_functions_tag model_function[NMODELS] = {
  {argon_init, argon_update}, /* ARGON */
  {spc_init, spc_update},     /* SPC */
  {rpol_init, rpol_update},   /* RPOL */
  {rpol_init, rpol_update},   /* POL3, same function as RPOl */
  {rpol_init, rpol_update},   /* POL1, same function as RPOl */
};


void data_collector_init(struct Data_Collector_Tag *dc,  
		      struct Data_Tag *data, struct Force_Tag *force,
		      enum MD_TYPE md_type)
{
  assert(NULL != dc);
  dc->pdata = data;
  dc->pforce = force;
  dc->model_id = data->model_id;
  dc->md_type = md_type;
  assert(data->model_id - force->model_id == 0);
  assert(dc->model_id < NMODELS);
  assert(dc->md_type < N_MD_TYPES);

  /* model specific initilizations */
  model_function[dc->model_id].init_function(dc);
}


void data_collector_destroy(struct Data_Collector_Tag* dc)
{
  MD_Int i;

  dc->pdata=NULL;
  dc->pforce=NULL;
  
  for (i = 0; i < dc->n_avgvar; i++) {
    avgvar_destroy(dc->avgvar[i]);
  }
  free(dc->avgvar);

  for (i = 0; i < dc->n_lf; i++) {
    linearfit_destroy(dc->lf[i]);
  }
  free(dc->lf);
}


void data_collector_assign_pmd(struct Data_Collector_Tag *dc, void *pmd)
{
  dc->p_md = pmd;
}


void data_collector_update(struct Data_Collector_Tag *dc, MD_Int istep)
{
  model_function[dc->model_id].update_function(dc, istep);
}


void data_collector_output(struct Data_Collector_Tag *dc)
{
  MD_Int i;

  for (i = 0; i < dc->n_avgvar; i++) {
    avgvar_output(dc->avgvar[i]);
  }

  /* some extra output */
  if (RPOL == dc->model_id)  {
    struct AvgVar_Tag **avs = dc->avgvar;
    const MD_Int * const av_index = dc->av_index;
    MD_Double mx = avgvar_get_avg(avs[av_index[AV_Mx]]);
    MD_Double my = avgvar_get_avg(avs[av_index[AV_My]]);
    MD_Double mz = avgvar_get_avg(avs[av_index[AV_Mz]]);
    printf("<M>^2/<M^2>= %f\n",  (mx*mx + my*my + mz*mz) / 
	   avgvar_get_avg(avs[av_index[AV_Msqr]]));
    /* avgvar_dump_data(dc->avgvar[AV_Msqr], stdout); */
  }

  for (i = 0; i < dc->n_lf; i++) {
    linearfit_print_stats(dc->lf[i], linearfit_names[i]);
  }

}



/* ----------------- argon model ------------------------------------- */

static void argon_init(struct Data_Collector_Tag *dc)
{
  const MD_Int init_size = 1000;
  /* avgvar_ptr is the inverse of av_index map, avgvar_ptr[i] = component
     index of "enum data_collector_avgvar" type */
  MD_Int avgvar_ptr[N_mda_av]; 
  MD_Int i;

  i = 0;
  dc->av_index[AV_kineticE] = i;
  avgvar_ptr[i++] = AV_kineticE;
  dc->av_index[AV_potentialE_per_mol] = i;
  avgvar_ptr[i++] = AV_potentialE_per_mol;
  dc->n_avgvar = i;

  dc->avgvar = malloc(dc->n_avgvar * sizeof(struct AvgVar_Tag *));
  assert(NULL != dc->avgvar);
  for (i = 0; i < dc->n_avgvar; i++) {
    dc->avgvar[i] = malloc(sizeof(struct AvgVar_Tag));
    assert(NULL != dc->avgvar[i]);
    avgvar_init(dc->avgvar[i], init_size, avgvar_names[avgvar_ptr[i]]);
  }

  dc->n_lf = N_mda_lf;
  dc->lf = malloc(dc->n_lf * sizeof(struct LinearFit_Tag *));
  assert(NULL != dc->lf);
  for (i = 0; i < dc->n_lf; i++) {
    dc->lf[i] = malloc(sizeof(struct LinearFit_Tag));
    assert(NULL != dc->lf[i]);
    if (linearfit_init(dc->lf[i], init_size)) {
      fprintf(stderr, "cannot init linearfit module\n");
      exit(FAILURE);
    }
  }  
}

void argon_update(struct Data_Collector_Tag *dc, MD_Int istep)
{
  struct AvgVar_Tag **avs = dc->avgvar;
  struct LinearFit_Tag **lfs = dc->lf;
  const MD_Int * const av_index = dc->av_index;
  const struct Data_Tag *data = dc->pdata;
  MD_Double inv_nmols = 3.0 / (MD_Double) data->natoms;

  avgvar_add(avs[av_index[AV_kineticE]], data->energy[KE] / KELVIN / 
	     (MD_Double) data->degree_freedom);
  avgvar_add(avs[av_index[AV_potentialE_per_mol]], data->energy[PE] /
	     KCAL_PER_MOL * inv_nmols); 

  if (ConstEnergy == dc->md_type) {
    linearfit_add_element(lfs[LF_energy], (MD_Double) istep, 
			  data->energy[TE] / KCAL_PER_MOL);
  } else if (ConstTemp == dc->md_type) {
    linearfit_add_element(lfs[LF_energy], (MD_Double) istep, 
			  data->energy[KE] / KELVIN);
  } else {
    fprintf(stderr, "wrong md_type !!!\n");
  }

}


/* ----------------- SPC water model ------------------------------------- */

static void spc_init(struct Data_Collector_Tag *dc)
{
  const MD_Int init_size = 1000;
  /* avgvar_ptr is the inverse of av_index map, avgvar_ptr[i] = component
     index of "enum data_collector_avgvar" type */
  MD_Int avgvar_ptr[N_mda_av]; 
  MD_Int i;

  i = 0;
  dc->av_index[AV_kineticE] = i;
  avgvar_ptr[i++] = AV_kineticE;
  dc->av_index[AV_potentialE_per_mol] = i;
  avgvar_ptr[i++] = AV_potentialE_per_mol;
  dc->av_index[AV_Msqr] = i;
  avgvar_ptr[i++] = AV_Msqr;
  /*
  if (ConstTemp == dc->md_type) {
    dc->av_index[AV_pv_over_nkT] = i;
    avgvar_ptr[i++] = AV_pv_over_nkT;
  }
  */
  dc->n_avgvar = i;

  dc->avgvar = malloc(dc->n_avgvar * sizeof(struct AvgVar_Tag *));
  assert(NULL != dc->avgvar);
  for (i = 0; i < dc->n_avgvar; i++) {
    dc->avgvar[i] = malloc(sizeof(struct AvgVar_Tag));
    assert(NULL != dc->avgvar[i]);
    avgvar_init(dc->avgvar[i], init_size, avgvar_names[avgvar_ptr[i]]);
  }

  dc->n_lf = N_mda_lf;
  dc->lf = malloc(dc->n_lf * sizeof(struct LinearFit_Tag *));
  assert(NULL != dc->lf);
  for (i = 0; i < dc->n_lf; i++) {
    dc->lf[i] = malloc(sizeof(struct LinearFit_Tag));
    assert(NULL != dc->lf[i]);
    if (linearfit_init(dc->lf[i], init_size)) {
      fprintf(stderr, "cannot init linearfit module\n");
      exit(FAILURE);
    }
  }  
}


void spc_update(struct Data_Collector_Tag *dc, MD_Int istep)
{
  struct AvgVar_Tag **avs = dc->avgvar;
  struct LinearFit_Tag **lfs = dc->lf;
  const MD_Int * const av_index = dc->av_index;
  const struct Data_Tag *data = dc->pdata;
  MD_Double inv_nmols = 3.0 / (MD_Double) data->natoms;

  MD_Double mol_perm_dipole, mol_induced_dipole, mol_total_dipole;
  MD_Dvec system_total_dipole;

  avgvar_add(avs[av_index[AV_kineticE]], data->energy[KE] / KELVIN / 
	     (MD_Double) data->degree_freedom);
  avgvar_add(avs[av_index[AV_potentialE_per_mol]],  data->energy[PE] /
	     KCAL_PER_MOL * inv_nmols); 
  
  compute_mol_sys_dipole(data->atom, data->pos, data->pdipole, data->natoms,
			 &mol_perm_dipole, &mol_induced_dipole, 
			 &mol_total_dipole, &system_total_dipole);

  avgvar_add(avs[av_index[AV_Msqr]], MD_vec_dot(system_total_dipole, 
						system_total_dipole));
#if 0
  if (ConstTemp == dc->md_type) {  /* !!!! serious bug !!!! */
    avgvar_add(avs[av_index[AV_pv_over_nkT]], 
	       vgb_get_pv_over_nkT(dc->p_md));	 
  }
#endif

  if (ConstEnergy == dc->md_type) {
    linearfit_add_element(lfs[LF_energy], (MD_Double) istep, 
			  data->energy[TE] / KCAL_PER_MOL);
  } else if (ConstTemp == dc->md_type) {
    linearfit_add_element(lfs[LF_energy], (MD_Double) istep, 
			  data->energy[KE] / KELVIN);
  } else {
    fprintf(stderr, "wrong md_type !!!\n");
  }

}

/* ----------------- RPOL water model ----------------------------------- */

static void rpol_init(struct Data_Collector_Tag *dc)
{
  const MD_Int init_size = 1000;
  /* avgvar_ptr is the inverse of av_index map, avgvar_ptr[i] = component
     index of "enum dc_avgvar" type */
  MD_Int avgvar_ptr[N_mda_av]; 
  MD_Int i;

  switch(dc->md_type) {
  case (ConstEnergy):   /* skip the last one */
    dc->n_avgvar = N_mda_av; 
    break;
  case (ConstTemp):
    dc->n_avgvar = N_mda_av;
    break;
  default:
    fprintf(stderr, "wrong md_type\n");
  }
  for (i = 0; i < dc->n_avgvar; i++) {
    avgvar_ptr[i] = i;
    dc->av_index[i] = i;
  }
  dc->avgvar = malloc(dc->n_avgvar * sizeof(struct AvgVar_Tag *));
  assert(NULL != dc->avgvar);
  for (i = 0; i < dc->n_avgvar; i++) {
    dc->avgvar[i] = malloc(sizeof(struct AvgVar_Tag));
    assert(NULL != dc->avgvar[i]);
    avgvar_init(dc->avgvar[i], init_size, avgvar_names[avgvar_ptr[i]]);
  }

  dc->n_lf = N_mda_lf;
  dc->lf = malloc(dc->n_lf * sizeof(struct LinearFit_Tag *));
  assert(NULL != dc->lf);
  for (i = 0; i < dc->n_lf; i++) {
    dc->lf[i] = malloc(sizeof(struct LinearFit_Tag));
    assert(NULL != dc->lf[i]);
    if (linearfit_init(dc->lf[i], init_size)) {
      fprintf(stderr, "cannot init linearfit module\n");
      exit(FAILURE);
    }
  }  
}


void rpol_update(struct Data_Collector_Tag* dc, MD_Int istep)
{
  struct AvgVar_Tag **avs = dc->avgvar;
  struct LinearFit_Tag **lfs = dc->lf;
  const MD_Int * const av_index = dc->av_index;
  const struct Data_Tag *data = dc->pdata;
  MD_Double inv_nmols = 3.0 / (MD_Double) data->natoms;

  MD_Double mol_perm_dipole, mol_induced_dipole, mol_total_dipole;
  MD_Dvec system_total_dipole;

  avgvar_add(avs[av_index[AV_kineticE]], data->energy[KE] / KELVIN / 
	     (MD_Double) data->degree_freedom);
  avgvar_add(avs[av_index[AV_potentialE_per_mol]],  data->energy[PE] /
	     KCAL_PER_MOL * inv_nmols);

  compute_mol_sys_dipole(data->atom, data->pos, data->pdipole, data->natoms,
			 &mol_perm_dipole, &mol_induced_dipole, 
			 &mol_total_dipole, &system_total_dipole);

  avgvar_add(avs[av_index[AV_Msqr]], MD_vec_dot(system_total_dipole, 
						system_total_dipole));
  avgvar_add(avs[av_index[AV_Mx]], system_total_dipole.x);
  avgvar_add(avs[av_index[AV_My]], system_total_dipole.y);
  avgvar_add(avs[av_index[AV_Mz]], system_total_dipole.z);
  avgvar_add(avs[av_index[AV_mol_perm_dipole]], mol_perm_dipole);
  avgvar_add(avs[av_index[AV_mol_ind_dipole]],  mol_induced_dipole);
  avgvar_add(avs[av_index[AV_mol_tot_dipole]],  mol_total_dipole);

  if (ConstEnergy == dc->md_type) {
    linearfit_add_element(lfs[LF_energy], (MD_Double) istep, 
			  data->energy[TE] / KCAL_PER_MOL);
  } else if (ConstTemp == dc->md_type) {
    linearfit_add_element(lfs[LF_energy], (MD_Double) istep, 
			  data->energy[KE] / KELVIN);
  } else {
    fprintf(stderr, "wrong md_type !!!\n");
  }

}


