/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "avgvar.h"

/* note that no static variables (shared value) should be used, 
 * since this module is used by several objects. They should not 
 * share any value.
 */

static void avgvar_compute(struct AvgVar_Tag *av);

void avgvar_init(struct AvgVar_Tag *av, const MD_Int size,  const char* title) 
{ 
  av->nelements = 0;
  av->array_size = size;
  av->data = malloc((size_t) size * sizeof(MD_Double));
  assert(NULL != av->data);
  av->average = 0.0;
  av->variance = 0.0;
  av->computed = 0;
  strncpy(av->title, title, (size_t)MD_STRING_SIZE);
  av->title[MD_STRING_SIZE-1] = '\0'; /* make sure the string ends */
}


void avgvar_destroy(struct AvgVar_Tag *av) {
  free(av->data);
}


void avgvar_add(struct AvgVar_Tag *av, MD_Double element)
{
  if (av->array_size  - av->nelements == 0) { /* enlarge data array */
    MD_Double *tmp;
    av->array_size *= 2;
    tmp = realloc(av->data, av->array_size * sizeof(MD_Double));
    assert(NULL != tmp);
    av->data = tmp;
  }
    
  av->data[av->nelements] = element;
  av->nelements++;
  av->computed = 0;
}


void avgvar_compute(struct AvgVar_Tag *av)
{
  MD_Double sum = 0.0, sqrsum = 0.0;
  MD_Double delta;
  MD_Int i;

  if (1 >= av->nelements) {
    printf("not eneough data\n");
    return;
  }

  sum = 0.0;
  for (i = 0; i < av->nelements; i++) {
    sum += av->data[i];
  }
  av->average = sum / (MD_Double) av->nelements;
  sqrsum = 0.0;
  for (i = 0; i < av->nelements; i++) {
    delta = av->data[i] - av->average;
    sqrsum += delta * delta;
  }
  av->variance = sqrt(sqrsum / (av->nelements - 1)); /* sqrsum > 0 for sure */
  av->computed = 1;
}


void avgvar_output(struct AvgVar_Tag *av)
{
  if (!av->computed) avgvar_compute(av);
  printf("%s:  %f (+-) %f\n", av->title, av->average, av->variance);
}


MD_Double avgvar_get_avg(struct AvgVar_Tag *av)
{
  if (!av->computed) avgvar_compute(av);
  return av->average;
}


MD_Double avgvar_get_var(struct AvgVar_Tag *av)
{
  if (!av->computed) avgvar_compute(av);
  return av->variance;
}

void avgvar_dump_data(struct AvgVar_Tag *av, FILE* fid)
{
  MD_Int i;
  fprintf(fid, "%s\n", av->title);
  for (i = 0; i < av->nelements; i++) {
    fprintf(fid, "%d\t%f\n", i, av->data[i]);
  }
}
