/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "auto_corr.h"
#include "utilities.h"
#include "helper.h"

/* we do not keep all vectors within the time window we're considering
 * since this costs a lot of memory. Instead, for each new vector
 * we obtained, we use it only for updating <v(0)v(t)>.
 */

MD_Int autocorr_init(struct Auto_Corr_Tag* ac,
			const MD_String name,
			const MD_Int timelen, 
			const MD_Int veclen, 
			const MD_Double dt) 
{
  assert(NULL != ac);
  strncpy(ac->name, name, sizeof(ac->name)-1);
  ac->name[sizeof(ac->name)-1] = '\0';
  ac->timelen = timelen;
  ac->veclen = veclen;
  ac->dt = dt;

  ac->current_tim = 0;
  ac->v0 = my_calloc((size_t)veclen, sizeof(MD_Dvec), "v0");
  ac->corr = my_calloc((size_t)ac->timelen, sizeof(MD_Double), "corr");
  ac->counter = my_calloc((size_t)ac->timelen, sizeof(MD_Int), "counter");

  return OK;
}


void autocorr_destroy(struct Auto_Corr_Tag* ac)
{
  free(ac->v0);
  free(ac->corr);
  free(ac->counter);
}


void autocorr_update(struct Auto_Corr_Tag* ac, const MD_Dvec* v)
{
  MD_Dvec* v0 = ac->v0;
  MD_Double vv = 0.0;
  MD_Int i;

  if (0 == ac->current_tim) 
    memcpy(v0, v, ac->veclen * sizeof(MD_Dvec));
  for (i = 0; i < ac->veclen; i++) {
    vv  += MD_vec_dot(v0[i], v[i]); 
  }
  vv  /= (MD_Double) (ac->veclen * sizeof(MD_Dvec)/sizeof(MD_Double));
  ac->corr[ac->current_tim] += vv;
  ac->counter[ac->current_tim] ++;

  ac->current_tim = (ac->current_tim + 1) % ac->timelen;
}


void autocorr_output(struct Auto_Corr_Tag* ac, char *filename)
{
  FILE *fid = fopen(filename, "w");
  MD_Double cr, cr0;
  MD_Int i;

  assert(NULL != fid);
  if (0 == ac->counter[0]) {
    printf("no data is ready yet to compute autocorrelation \n");
    return;
  } 
  cr0 = ac->corr[0] / (MD_Double) ac->counter[0];
  fprintf(fid, "%f %f\n", 0.0, 1.0);   /* normalization */
  for (i = 1; i < ac->timelen; i++) {
    cr = (0 == ac->counter[i]) ? 0:
      ac->corr[i] / (MD_Double) ac->counter[i];
    fprintf(fid, "%f %f\n", (MD_Double)i*ac->dt, cr/cr0);
  }

  if (ferror(fid)) {
    perror("cannot output autocorrelation result into file \n");
  }
  
  fclose(fid);
  return;
}


void autocorr_print(struct Auto_Corr_Tag* ac)
{
  MD_Double cr, cr0;
  MD_Int i;

  if (0 == ac->counter[0]) {
    printf("no data is ready yet to compute autocorrelation\n");
    return;
  } 
  printf("Autocorrelation function for %s\n", ac->name);
  cr0 = ac->corr[0] / (MD_Double) ac->counter[0];
  printf("%f %f\n", 0.0, 1.0);   /* normalization */
  for (i = 1; i < ac->timelen; i++) {
    cr = (0 == ac->counter[i]) ? 0:
      ac->corr[i] / (MD_Double) ac->counter[i];
    printf("%f %f\n", (MD_Double)i*ac->dt, cr/cr0);
  }

  return;
}
