/*
 * Copyright (C) 2004-2006 by Wei Wang.  All rights reserved.
 */


#include "PmeBase.h"
#include "helper.h"
#include "utilities.h"
#include <assert.h>
#include <stdio.h>
#include <math.h>
 
/* 
 * input: frac[3], 0 <= frac[0,1,2] < 1, frac[i] = u[i] - floor(u[i])
 *        order >= 3
 * output: M[3*order], dM[3*order]
 *          Mx =  M,  My =  Mx + order,  Mz =  My + order
 *         dMx = dM, dMy = dMx + order, dMz = dMy + order
 *         for 0 <= i < order,  n = order, Mn: B-spline of order n
 *          M1[i] = Mn (n - 1 - i + frac[0]);
 *          M2[i] = Mn (n - 1 - i + frac[1]);
 *          M3[i] = Mn (n - 1 - i + frac[2]);
 *         dM1[i] = Mn'(n - 1 - i + frac[0]); (derivative)
 *         dM2[i] = Mn'(n - 1 - i + frac[1]);
 *         dM3[i] = Mn'(n - 1 - i + frac[2]);
 *
 * the B-spline functions are defined as:
 *
 *                   - 1  (0<=x<1)
 *          M1(x) = |      
 *                   - 0  otherwise
 *
 *                     x                     n - x
 *  Mn(x) = M_n(x) = ------- M_(n-1) (x)  + ------- M_(n-1) (x-1)   (n > 1)
 *                   n - 1                   n - 1
 * 
 *  Mn'(x) = M_(n-1) (x) - M_(n-1) (x-1)
 *
 *  Mn''(x) = M_(n-2) (x) - 2*M_(n-2) (x-1) + M_(n-2) (x-2)
 *
 *
 * Internally, however, the start of the array is offset by -1, and the
 * index runs from 1 to n (fortran style), so 
 *               M[i] = Mn(n-i+f) for i = 1,2,...,n
 * the derivative is computed after M_(n-1) is ready:
 *              dM[i] = Mn'(n-i+f) = M_(n-1)(n-i+f) - M_(n-1)(n-i-1+f)
 *                    = M[i-1] - M[i]    (M[x] is about function M_(n-1))
 * the second derivative is computed after M_(n-2) is ready:
 *      ddM[i] = Mn''(n-i+f)
 *             = M_(n-2) (n-i+f) - 2*M_(n-2) (n-i-1+f) + M_(n-2) (n-i-2+f)
 *             = M[i-2] - 2*M[i-1] + M[i] 
 *
 * given the vaules of "M_(n-1)[i]" as
 *   M[i] = M_(n-1) (n-1-i+f) for i = 1..(n-1)
 * the induction formula gives "M_n[i]" as
 *   M[i] = M_n (n-i+f)   <--- (M[x] is the new M)
 *             1
 *        = -------{ (n-i+f) * M_(n-1) (n-i+f) + (i-f) * M_(n-1) (n-i-1+f) }
 *           (n-1)
 *             
 *             1
 *        = -------{ (n-i+f)*M[i-1] + (i-f)*M[i] }  <--- (M[x] are old Ms)
 *           (n-1)
 */

#ifdef BSPLINE_INDUCTION
#undef BSPLINE_INDUCTION
#endif

#define BSPLINE_INDUCTION(n, div, x, y, z, x1, y1, z1, Mx, My, Mz, i) {	\
  div=1.0/(MD_Double) (n-1); \
  Mx[n] = x*div*Mx[n-1]; \
  My[n] = y*div*My[n-1]; \
  Mz[n] = z*div*Mz[n-1]; \
  for (i=n-1; i>=2; i--) { \
    Mx[i] = ((n-i+x)*Mx[i-1] + (i-x)*Mx[i]) * div; \
    My[i] = ((n-i+y)*My[i-1] + (i-y)*My[i]) * div; \
    Mz[i] = ((n-i+z)*Mz[i-1] + (i-z)*Mz[i]) * div; \
  } \
  Mx[1] *= x1*div; \
  My[1] *= y1*div; \
  Mz[1] *= z1*div; \
  }


void compute_bspline1(const MD_Double *frac, MD_Double *M, MD_Double *dM, 
		      const MD_Int order) 
{
  MD_Int i, n;
  MD_Double x,y,z,x1,y1,z1,div;
  MD_Double *Mx, *My, *Mz, *dMx, *dMy, *dMz;

   Mx=  M-1;   My=  Mx+order;   Mz=  My+order;
  dMx= dM-1;  dMy= dMx+order;  dMz= dMy+order;
  x=frac[0];  y=frac[1];  z=frac[2];
  x1=1.0-x; y1=1.0-y; z1=1.0-z;

  ASSERT(order >= 4);

  /* M_3 */
  Mx[1]=0.5*x1*x1;
  Mx[2]=x1*x + 0.5;
  Mx[3]=0.5*x*x;
  My[1]=0.5*y1*y1;
  My[2]=y1*y + 0.5;
  My[3]=0.5*y*y;
  Mz[1]=0.5*z1*z1;
  Mz[2]=z1*z + 0.5;
  Mz[3]=0.5*z*z;
 
  /* M_{n-1} */
  for (n=4; n<=order-1; n++) {
    BSPLINE_INDUCTION(n, div, x, y, z, x1, y1, z1, Mx, My, Mz, i);
  }

  /* M'_{n} */
  dMx[1]=-Mx[1]; dMy[1]=-My[1]; dMz[1]=-Mz[1];
  Mx[order] = My[order] = Mz[order] = 0.0; /* M[order] is not assigned yet */
  for (i=2; i <= order; i++) {
    dMx[i] = Mx[i-1] - Mx[i];
    dMy[i] = My[i-1] - My[i];
    dMz[i] = Mz[i-1] - Mz[i];
  }

  /* M_n */
  BSPLINE_INDUCTION(order, div, x, y, z, x1, y1, z1, Mx, My, Mz, i);

#ifdef DEBUG_PME /* check */
  { 
    MD_Double sx=array_sum(Mx+1, order);
    MD_Double sy=array_sum(My+1, order);
    MD_Double sz=array_sum(Mz+1, order);
    MD_Double dx=array_sum(dMx+1, order);
    MD_Double dy=array_sum(dMy+1, order);
    MD_Double dz=array_sum(dMz+1, order);
    MD_Double eps = 1e-15;
    if (fabs(sx-1.0) > eps || fabs(sy-1.0)>eps || fabs(sz-1.0)>eps ||
	fabs(dx) > eps || fabs(dy)>eps || fabs(dz)>eps) {
      printf("expected: sx=sy=sz=1, dx=dy=dz=0, ex=ey=ez=0\n");
      printf("sx=%f, sy=%f, sz=%f\n", sx, sy, sz);
      printf("dx=%g, dy=%g, dz=%g\n", dx, dy, dz);
    }
  }
#endif
  return;
}


void compute_bspline2(const MD_Double *frac, MD_Double *M, MD_Double *dM, 
		      MD_Double *ddM, const MD_Int order) 
{
  MD_Int i, n;
  MD_Double x,y,z,x1,y1,z1,div;
  MD_Double *Mx, *My, *Mz, *dMx, *dMy, *dMz, *ddMx, *ddMy, *ddMz;

    Mx=  M-1;   My=  Mx+order;   Mz=  My+order;
   dMx= dM-1;  dMy= dMx+order;  dMz= dMy+order;
  ddMx=ddM-1; ddMy=ddMx+order; ddMz=ddMy+order;
  x=frac[0]; y=frac[1]; z=frac[2];
  x1=1.0-x;  y1=1.0-y;  z1=1.0-z;

  ASSERT(order >= 4);

  /* M_2 */
  Mx[1] = x1;  Mx[2] = x;
  My[1] = y1;  My[2] = y;
  Mz[1] = z1;  Mz[2] = z;

  /* M_{order-2}, recursive  */
  for (n=3; n<=order-2; n++) {
    BSPLINE_INDUCTION(n, div, x, y, z, x1, y1, z1, Mx, My, Mz, i);
  }

  /* do the second derivative */
  ddMx[1]=Mx[1];  
  ddMy[1]=My[1];  
  ddMz[1]=Mz[1];
  ddMx[2]=Mx[2]-2.0*Mx[1]; 
  ddMy[2]=My[2]-2.0*My[1]; 
  ddMz[2]=Mz[2]-2.0*Mz[1]; 
  Mx[order] = Mx[order-1] = 0.0; /* for the second and first derivative */
  My[order] = My[order-1] = 0.0; /* they are no assigned yet */
  Mz[order] = Mz[order-1] = 0.0;
  for (i=3; i<=order; i++) {
    ddMx[i] = Mx[i] - 2.0*Mx[i-1] + Mx[i-2];
    ddMy[i] = My[i] - 2.0*My[i-1] + My[i-2];
    ddMz[i] = Mz[i] - 2.0*Mz[i-1] + Mz[i-2];
  }

  /* M_{order-1} */
  n = order - 1;
  BSPLINE_INDUCTION(n, div, x, y, z, x1, y1, z1, Mx, My, Mz, i);

  /* do the first derivatives  */
  dMx[1]=-Mx[1]; dMy[1]=-My[1]; dMz[1]=-Mz[1];
  for (i=2; i <= order; i++) {
    dMx[i] = Mx[i-1] - Mx[i];
    dMy[i] = My[i-1] - My[i];
    dMz[i] = Mz[i-1] - Mz[i];
  }

  /* M_{order} */
  BSPLINE_INDUCTION(order, div, x, y, z, x1, y1, z1, Mx, My, Mz, i);

  
#ifdef DEBUG_PME /* check */
  { 
    MD_Double sx=array_sum(Mx+1, order);
    MD_Double sy=array_sum(My+1, order);
    MD_Double sz=array_sum(Mz+1, order);
    MD_Double dx=array_sum(dMx+1, order);
    MD_Double dy=array_sum(dMy+1, order);
    MD_Double dz=array_sum(dMz+1, order);
    MD_Double ex=array_sum(ddMx+1, order);
    MD_Double ey=array_sum(ddMy+1, order);
    MD_Double ez=array_sum(ddMz+1, order);
    MD_Double eps = 1e-15;
    if (fabs(sx-1.0) > eps || fabs(sy-1.0)>eps || fabs(sz-1.0)>eps ||
	fabs(dx) > eps || fabs(dy)>eps || fabs(dz)>eps ||
	fabs(ex) > eps || fabs(ey)>eps || fabs(ez)>eps) {
      printf("expected: sx=sy=sz=1, dx=dy=dz=0, ex=ey=ez=0\n");
      printf("sx=%f, sy=%f, sz=%f\n", sx, sy, sz);
      printf("dx=%g, dy=%g, dz=%g\n", dx, dy, dz);
      printf("ex=%g, ey=%g, ez=%g\n", ex, ey, ez);
    }
  }
#endif

  return;
}
