/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * engine.h
 */

#ifndef ENGINE_H
#define ENGINE_H

#include "mdapi/mdengine.h"
#include "force/force.h"
#include "mgrid/mgrid.h"
#include "pmetest/pmetest.h"
#include "step/step.h"
#include "timer/timer.h"
#include "deven/deven.h"
#include "deven/param.h"
#include "deven/result.h"

#ifdef __cplusplus
extern "C" {
#endif


  typedef struct Engine_tag {

    /* engine data arrays */
    MD_Engdata *atomprm;             /* atom force field params */
    MD_Engdata *bondprm;             /* bond force field params */
    MD_Engdata *angleprm;            /* angle force field params */
    MD_Engdata *dihedprm;            /* dihedral force field params */
    MD_Engdata *imprprm;             /* improper force field params */
    MD_Engdata *nbfixprm;            /* fix nonbonded force field params */
    MD_Engdata *atom;                /* atom topology */
    MD_Engdata *bond;                /* bond topology */
    MD_Engdata *angle;               /* angle topology */
    MD_Engdata *dihed;               /* dihedral topology */
    MD_Engdata *impr;                /* improper topology */
    MD_Engdata *excl;                /* excluded nonbonded pairs */
    MD_Engdata *pos;                 /* position array */
    MD_Engdata *vel;                 /* velocity array */
    MD_Engdata *force_engdata;       /* provides force array for "f_result" */
    MD_Engdata *result_engdata;      /* uses "result" as static buffer */
    MD_Engdata *param_engdata;       /* uses "param" as static buffer */
    MD_Engdata *timestep_engdata;    /* uses "timestep" as static buffer */
    MD_Engdata *init_temp_engdata;   /* uses "init_temp" as static buffer */
    MD_Engdata *com_motion_engdata;  /* uses "com_motion" as static buffer */
    MD_Engdata *seed_engdata;        /* uses "seed" as static buffer */
    MD_Engdata *numdegreesfreedom;   /* uses "ndegfreedom" as static buffer */
  /*** data arrays for mgrid ***/
    MD_Engdata *poswrap;             /* positions offset by wrapping */
    MD_Engdata *f_elec;              /* provides f_elec for mgrid_system */
    MD_Engdata *charge;              /* provides charge for mgrid_system */
#if 0
  /*** for debugging electrostatics ***/
    MD_Engdata *force_elec;
#endif
  /*** data arrays for computing interaction energiess (arrays of double) ***/
    MD_Engdata *u_bond;              /* spring bond interaction energies */
    MD_Engdata *u_angle;             /* angle interaction energies */
    MD_Engdata *u_dihed;             /* dihedral interaction energies */
    MD_Engdata *u_impr;              /* improper interaction energies */
    MD_Engdata *u_elec;              /* electrostatic interaction energies */
    MD_Engdata *u_vdw;               /* van der Waals interaction energies */
    MD_Engdata *u_bound;             /* boundary interaction energies */
    MD_Engdata *pot_elec;            /* electrostatic potentials */

  /*** data array for fixed atoms ***/
    MD_Engdata *fixedatomslist;      /* list of atom indices */

  /*** data array for harmonic constraints ***/
    MD_Engdata *constraintlist;      /* list of DevenConstraint */

    /* declared engine-defined types */
    Result result;                   /* simulation results (e.g. energy) */
    Param param;                     /* sim params that impact force */

    /* force data */
    Force *force;
    ForceParam f_param;
    ForceDomain f_domain;
    ForceResult f_result;
    int32 is_domain_periodic;

    /* mgrid data */
    int32 ismgrid;
    Mgrid mgrid;
    MgridParam mgrid_param;
    MgridSystem mgrid_system;

    /* pmetest data */
    int32 ispmetest;
    Pmetest *pmetest;
    PmetestParams pt_param;
    PmetestSystem pt_system;

    /* step data */
    Step *step;
    StepParam step_param;
    int32 is_step_initialized;

    /* fixed atoms data */
    int32 nfixedatoms;               /* number of fixed atoms */
    int32 *fixedatom;                /* array of fixed atom indices */

    /* harmonic constraints */
    int32 nconstraints;
    DevenConstraint *constraint;
    int32 is_select_components;
    int32 is_select_xcoord;
    int32 is_select_ycoord;
    int32 is_select_zcoord;

    /* additional sim params set by front end */
    double timestep;                 /* timestep in femtoseconds */
    double init_temp;                /* initial temperature */
    char com_motion[4];              /* allow center of mass motion? */
    uint32 seed;                     /* for random number generator */

    /* additional results reported to front end */
    int32 ndegfreedom;               /* number of degrees of freedom */

    /* internal variables */
    int32 natoms;                    /* number of atoms in system */
    int32 forceopts;                 /* for post-processing total force */
    double inv_natoms;               /* 1/N */
    double inv_totalmass;            /* inverse of sum of mass */
    double *scaled_mass;             /* each mass divided by sum of mass */
    MD_Dvec *init_pos;               /* retain initial atomic positions */

    int32 last_stepnum;              /* track stepnum between reductions() */
    int32 cntsteps_outputMomenta;    /* count steps between momenta output */
    int32 cntsteps_outputPressure;   /* count steps between pressure output */
    int32 cntsteps_outputTiming;     /* count steps between timing output */
    Timer *timer;                    /* wall-clock timer */

    /* engine-define error numbers */
    int32 err_param;
    int32 err_unsupported;
    int32 err_force_init;
    int32 err_force;

  } Engine;


  enum {
    ENGINE_FIX_LINMO = 1,
    /* conserve linear momentum using standard averaging of net force
     * (this causes energy drift!) */
    ENGINE_CONS_LINMO = 2
    /* conserve linear momentum using weighted averaging of net force
     * (no energy drift!) */
  };


  /* prototypes for run, update, and force routines */
  int32 deven_run(MD_Front *front, int32 numsteps, int32 flags);
  int32 deven_update(MD_Front *front);
  int32 deven_force(void *vfront, double *pe, MD_Dvec *f, const MD_Dvec *pos);

  /* callbacks */
  int deven_output_from_step(void *vfront, const char *msg);
  int deven_results_from_step(void *vfront, StepSystem *sys);


#ifdef __cplusplus
}
#endif

#endif /* ENGINE_H */
