/*
 * Copyright (C) 2000 by John Stone.  All rights reserved.
 *
 * table.c - simple hash table ADT
 * 
 * Uses null terminated strings as the keys for the table.
 * Stores an integer value with the string key.  It would
 * be easy to change to use void * values instead of int.
 * Maybe rewrite as a C++ template??
 *
 * Donated by John Stone.
 *
 * Modified by David Hardy.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "adt/table.h"
#include "debug/debug.h"

#undef   HASH_LIMIT
#define  HASH_LIMIT  0.5


/*
 *  Local types
 */
typedef struct hash_node_t {
  int data;                 /* data in hash node */
  const char *key;            /* key for hash lookup */
  struct hash_node_t *next;   /* next node in hash chain */
} hash_node_t;


/*
 *  hash() - Hash function returns a hash number for a given key.
 *
 *  tptr: Pointer to a hash table
 *  key: The key to create a hash number for
 */
static int hash(adt_Table *tptr, const char *key) {
  int i=0;
  int hashvalue;
 
  while (*key != '\0')
    i=(i<<3)+(*key++ - '0');
 
  hashvalue = (((i*1103515249)>>tptr->downshift) & tptr->mask);
  if (hashvalue < 0) {
    hashvalue = 0;
  }

  return hashvalue;
}


/*
 *  rebuild_table() - Create new hash table when old one fills up.
 *
 *  tptr: Pointer to a hash table
 */
static int rebuild_table(adt_Table *tptr) {
  hash_node_t **old_bucket, *old_hash, *tmp;
  int old_size, old_entries, old_downshift, old_mask, h, i;

  old_bucket=tptr->bucket;
  old_size=tptr->size;
  old_entries=tptr->entries;
  old_downshift=tptr->downshift;
  old_mask=tptr->mask;

  /* create a new table and rehash old buckets */
  if (adt_initializeTable(tptr, old_size<<1)) {
    tptr->bucket = old_bucket;
    tptr->size = old_size;
    tptr->entries = old_entries;
    tptr->downshift = old_downshift;
    tptr->entries = old_entries;
    return ADT_ERROR;
  }
  for (i=0; i<old_size; i++) {
    old_hash=old_bucket[i];
    while(old_hash) {
      tmp=old_hash;
      old_hash=old_hash->next;
      h=hash(tptr, tmp->key);
      tmp->next=tptr->bucket[h];
      tptr->bucket[h]=tmp;
      tptr->entries++;
    } /* while */
  } /* for */

  /* free memory used by old table */
  free(old_bucket);

  return 0;
}


adt_Table *adt_createTable(int buckets) {
  adt_Table *p;

  p = (adt_Table *) malloc(sizeof(adt_Table));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (adt_initializeTable(p, buckets)) {
    free(p);
    return NULL;
  }
  return p;
}


void adt_destroyTable(adt_Table *p) {
  ASSERT(p != NULL);
  adt_cleanupTable(p);
  free(p);
}


/*
 *   - Initialize a new hash table.
 *
 *  tptr: Pointer to the hash table to initialize
 *  buckets: The number of initial buckets to create
 */
int adt_initializeTable(adt_Table *tptr, int buckets) {

  /* make sure we allocate something */
  if (buckets==0)
    buckets=16;

  /* initialize the table */
  tptr->entries=0;
  tptr->size=2;
  tptr->mask=1;
  tptr->downshift=29;

  /* ensure buckets is a power of 2 */
  while (tptr->size<buckets) {
    tptr->size<<=1;
    tptr->mask=(tptr->mask<<1)+1;
    tptr->downshift--;
  } /* while */

  /* allocate memory for table */
  tptr->bucket=(hash_node_t **) calloc(tptr->size, sizeof(hash_node_t *));
  if (tptr->bucket == NULL) {
    tptr->size = 0;
    adt_cleanupTable(tptr);
    return ADT_ERROR;
  }

  return 0;
}


/*
 *   - Lookup an entry in the hash table and return a
 *  pointer to it or ADT_ERROR if it wasn't found.
 *
 *  tptr: Pointer to the hash table
 *  key: The key to lookup
 */
int adt_lookupTable(adt_Table *tptr, const char *key) {
  int h;
  hash_node_t *node;

  /* find the entry in the hash table */
  h=hash(tptr, key);
  for (node=tptr->bucket[h]; node!=NULL; node=node->next) {
    if (strcmp(node->key, key)==0) {
      /* found entry, return data */
      return node->data;
    }
  }

  /* otherwise we failed */
  return ADT_ERROR;
}


/*
 *   - Update int data for this key if the key is already
 *  in this table.  Return updated data or ADT_ERROR if key isn't in table.
 *
 *  tptr: Pointer to the hash table
 *  key: The key to lookup
 *  data: The new data value for this key
 */
int adt_updateTable(adt_Table *tptr, const char *key, int data) {
  int h, olddata;
  hash_node_t *node;

  /* find the entry in the hash table */
  h = hash(tptr, key);
  for (node = tptr->bucket[h];  node != NULL;  node = node->next) {
    if (strcmp(node->key, key)==0) {
      /* found entry, update and return old data */
      olddata = node->data;
      node->data = data;
      return olddata;
    }
  }

  /* otherwise we failed */
  return ADT_ERROR;
}


/*
 *   - Insert an entry into the hash table and return
 *  the data value.  If the entry already exists return a pointer to it,
 *  otherwise return ADT_ERROR.
 *
 *  tptr: A pointer to the hash table
 *  key: The key to insert into the hash table
 *  data: A pointer to the data to insert into the hash table
 */
int adt_insertTable(adt_Table *tptr, const char *key, int data) {
  hash_node_t *node;
  int h;


  /* find the entry in the hash table */
  h=hash(tptr, key);
  for (node=tptr->bucket[h]; node!=NULL; node=node->next) {
    if (strcmp(node->key, key)==0) {
      /* found entry, return data value that we found */
      return node->data;
    }
  }
  /* otherwise, we need to insert this (key,data) pair into the table */

  /* expand the table if needed */
  if (tptr->entries>=HASH_LIMIT*tptr->size) {
    do {
      if (rebuild_table(tptr))  return ADT_ERROR;
    } while (tptr->entries>=HASH_LIMIT*tptr->size);
    /* have to recompute hash function */
    h=hash(tptr, key);
  }

  /* insert the new entry */
  node=(struct hash_node_t *) malloc(sizeof(hash_node_t));
  if (node == NULL)  return ADT_ERROR;
  node->data=data;
  node->key=key;
  node->next=tptr->bucket[h];
  tptr->bucket[h]=node;
  tptr->entries++;

  return data;
}


/*
 *   - Remove an entry from a hash table and return a
 *  pointer to its data or ADT_ERROR if it wasn't found.
 *
 *  tptr: A pointer to the hash table
 *  key: The key to remove from the hash table
 */
int adt_deleteTable(adt_Table *tptr, const char *key) {
  hash_node_t *node, *last;
  int data;
  int h;

  /* find the node to remove */
  h=hash(tptr, key);
  for (node=tptr->bucket[h]; node; node=node->next) {
    if (strcmp(node->key, key)==0) {
      break;
    }
  }

  /* Didn't find anything, return ADT_ERROR */
  if (node==NULL)
    return ADT_ERROR;

  /* if node is at head of bucket, we have it easy */
  if (node==tptr->bucket[h])
    tptr->bucket[h]=node->next;
  else {
    /* find the node before the node we want to remove */
    for (last=tptr->bucket[h]; last && last->next; last=last->next) {
      if (last->next==node)
        break;
    }
    last->next=node->next;
  }

  /* free memory and return the data */
  data=node->data;
  free(node);

  return(data);
}



/*
 *  - Delete the entire table and all remaining entries.
 * 
 */
void adt_cleanupTable(adt_Table *tptr) {
  hash_node_t *node, *last;
  int i;

  for (i=0; i<tptr->size; i++) {
    node = tptr->bucket[i];
    while (node != NULL) { 
      last = node;   
      node = node->next;
      free(last);
    }
  }     

  /* free the entire array of buckets */
  free(tptr->bucket);
  memset(tptr, 0, sizeof(adt_Table));
}


/*
 *  alos() - Find the average length of search.
 *
 *  tptr: Pointer to a hash table
 */
static double alos(adt_Table *tptr) {
  int i,j;
  double alos=0;
  hash_node_t *node;

  for (i=0; i<tptr->size; i++) {
    for (node=tptr->bucket[i], j=0; node!=NULL; node=node->next, j++);
    if (j)
      alos+=((j*(j+1))>>1);
  } /* for */

  return(tptr->entries ? alos/tptr->entries : 0);
}


/*
 *   - Return a string with stats about hash table.
 *
 *  tptr: A pointer to the hash table
 */
const char *adt_getStatsTable(adt_Table *tptr) {
  sprintf(tptr->buf, "%d slots, %d entries, and %1.2f ALOS",
      (int)tptr->size, (int)tptr->entries, (double)alos(tptr));
  return (const char *) tptr->buf;
}
