/*
 * Copyright (C) 2003-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    array.h
 * @brief   Resize array container.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c adt_Array class provides a container for generic one-dimensional
 * arrays whose length can be resized.  Access routines perform error-checking
 * on the index bounds.  The array buffer space is by default dynamically
 * allocated, but it is also possible to pass preallocated buffer space
 * to be accessed, however a preallocated buffer cannot be resized.
 *
 * Functions returning @c int generally indicate success by returning 0
 * and failure by returning @c ADT_ERROR.  The access functions that accept
 * an array index will terminate the program if the index is out-of-range.
 * This means that error-checking is necessary only for calls to
 * the constructor @c adt_createArray()
 * (or alternatively @c adt_initializeArray() )
 * and to the resize function @c adt_resizeArray().
 */

#ifndef ADT_ARRAY_H
#define ADT_ARRAY_H

#ifdef __cplusplus
extern "C" {
#endif

#ifndef ADT_ERROR
#define ADT_ERROR  (-1)  /**< Return value from failed function call. */
#endif

  /**@brief Array container class.
   *
   * Members should be treated as private.
   */
  typedef struct adt_Array_t {
    void *data;
    int elemsz;
    int len;
    int flags;
  } adt_Array;


  /**@brief Constructor.
   *
   * Creates dynamically allocated array object.
   * @param[in] elemsz  The size in bytes of array elements, must be positive.
   * @param[in] len  The length of array measured in number of elements,
   *   must be nonnegative.
   * @param[in] buffer  Provides a preallocated buffer space to be used.
   *
   * If @p buffer is @c NULL, then the array buffer space is dynamically
   * allocated and can be resized.
   *
   * If @p buffer is not @c NULL, then the space provided must be at
   * least @p elemsz times @p len bytes.  The user retains control over
   * buffer space provided (e.g. user must free dynamic allocations).
   *
   * @return Pointer to array object or @c NULL if the parameters are
   * inconsistent or if memory allocation fails.
   */
  adt_Array *adt_createArray(int elemsz, int len, void *buffer);


  /**@brief Destructor.
   *
   * Frees dynamically allocated buffer space and dynamically allocated
   * array object.
   */
  void adt_destroyArray(adt_Array *);


  /**@brief Returns length of array. */
  int adt_getLengthArray(adt_Array *);


  /**@brief Returns array element size in bytes. */
  int adt_getElemsizeArray(adt_Array *);


  /**@brief Returns pointer to array buffer space.
   *
   * This permits direct access of array data for efficiency.
   * Note that resizing the array will most likely change the
   * memory location of the buffer, invalidating earlier calls
   * to this function.  @c NULL indicates an empty buffer.
   */
  void *adt_getDataArray(adt_Array *);


  /**@brief Index the array.
   *
   * @param[in] index  The array index using zero-based indexing.
   *
   * Must have @f$ 0 \leq @f$ @p index @f$ < @f$ length.
   * An @p index out-of-range will terminate program with error message.
   *
   * @return Pointer to the indicated array element.  Typecast to the
   * actual element type in order to use it.
   */
  void *adt_indexArray(adt_Array *, int index);


  /**@brief Update an array element.
   *
   * @param[in] index  The array index using zero-based indexing.
   * @param[in] pelem  Points to the desired element value.
   *
   * Must have @f$ 0 \leq @f$ @p index @f$ < @f$ length.
   * An @p index out-of-range will terminate program with error message.
   *
   * Performs a deep copy of the value located at @p pelem
   * into the indicated array element position.
   *
   * @return 0 (does not fail).
   */
  int adt_updateArray(adt_Array *, int index, void *pelem);


  /**@brief Resize the array.
   *
   * @param[in] len  The new length for the array.
   *
   * Must have @f$ 0 \leq @f$ @p len.
   * Fails if memory allocation is not possible.
   * (Terminates program as a bug if @p len is illegal or if an attempt is
   * made to resize a user-provided buffer.)
   *
   * @return 0 on success or @c ADT_ERROR on failure.
   */
  int adt_resizeArray(adt_Array *, int len);


  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated array object.
   * See @c adt_createArray() for a description of expected arguments.
   */
  int adt_initializeArray(adt_Array *, int elemsz, int len, void *buffer);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated array object
   * (i.e. one constructed using @c adt_initializeArray() )
   */
  void adt_cleanupArray(adt_Array *);

#ifdef __cplusplus
}
#endif

#endif /* ADT_ARRAY_H */
