/*
 * Copyright (C) 2003-2005 by David J. Hardy.  All rights reserved.
 *
 * array.c
 */

#include <stdlib.h>
#include <string.h>
#include "adt/array.h"
#include "debug/debug.h"


enum {
  DYNAMIC_BUFFER,
  STATIC_BUFFER
};


adt_Array *adt_createArray(int elemsz, int len, void *buffer)
{
  adt_Array *p;
  p = (adt_Array *) malloc(sizeof(adt_Array));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (adt_initializeArray(p, elemsz, len, buffer)) {
    free(p);
    return NULL;
  }
  return p;
}


int adt_initializeArray(adt_Array *p, int elemsz, int len, void *buffer)
{
  ASSERT(p != NULL);
  if (elemsz <= 0) {
    BUG("must have positive element size");
  }
  else if (len < 0) {
    BUG("must have nonnegative length");
  }
  else if (buffer != NULL && len == 0) {
    BUG("static buffer must have positive length");
  }
  p->elemsz = elemsz;
  p->len = len;
  p->data = buffer;
  if (p->data != NULL) {
    p->flags = STATIC_BUFFER;
  }
  else {
    p->flags = DYNAMIC_BUFFER;
    if (len != 0) {
      p->data = malloc(len * elemsz);
      if (p->data == NULL) {
        ERRMSG("out of memory");
        return ADT_ERROR;
      }
    } /* len == 0  implies  p->data == NULL */
  }
  INT(p->flags);
  INT(elemsz);
  return 0;
}


void adt_destroyArray(adt_Array *p)
{
  ASSERT(p != NULL);
  adt_cleanupArray(p);
  free(p);
}


void adt_cleanupArray(adt_Array *p)
{
  ASSERT(p != NULL);
  if (p->flags == DYNAMIC_BUFFER) {
    free(p->data);
  }
}


int adt_getLengthArray(adt_Array *p)
{
  ASSERT(p != NULL);
  return p->len;
}


int adt_getElemsizeArray(adt_Array *p)
{
  ASSERT(p != NULL);
  return p->elemsz;
}


void *adt_getDataArray(adt_Array *p)
{
  ASSERT(p != NULL);
  return p->data;
}


void *adt_indexArray(adt_Array *p, int index)
{
  ASSERT(p != NULL);
  if (index >= 0 && index < p->len) {
    return ((char *)(p->data)) + (index * p->elemsz);
  }
  else {
    BUG("index out of range");
  }
}


int adt_updateArray(adt_Array *p, int index, void *pelem)
{
  char *dest;
  ASSERT(p != NULL);
  ASSERT(pelem != NULL);
  if (index >= 0 && index < p->len) {
    dest = ((char *)(p->data)) + (index * p->elemsz);
    memcpy(dest, pelem, p->elemsz);
    return 0;
  }
  else {
    BUG("index out of range");
  }
}


int adt_resizeArray(adt_Array *p, int len)
{
  ASSERT(p != NULL);
  if (len < 0) {
    BUG("must have nonnegative length");
  }
  if (p->flags == DYNAMIC_BUFFER) {
    void *newdata;
    newdata = realloc(p->data, len * p->elemsz);
    if (len == 0) {
      /* have to explicitly set buffer to NULL when freeing memory */
      newdata = NULL;
    }
    else if (newdata == NULL) {
      ERRMSG("out of memory");
      return ADT_ERROR;
    }
    p->data = newdata;
    p->len = len;
    return 0;
  }
  else {
    BUG("cannot resize static buffer");
  }
}
