/*
 * Copyright (C) 2005 by David J. Hardy.  All rights reserved.
 */

/**@file    step.h
 * @brief   Time stepping library.
 * @author  David J. Hardy
 * @date    2005
 *
 * The @c Step_tag class computes time integration using leapfrog
 * (velocity Verlet) integration.  Also provided are routines for
 * random velocity initialization from an initial temperature
 * distribution and computation of kinetic energy and temperature
 * reductions.
 *
 * Two helper classes are required.  The @c StepParam_tag class is
 * used for setup.  Its fields are set to fixed simulation parameters
 * that should not change during the simulation.  Among these parameters
 * is a generic force routine that should by design accept the current
 * positions and provide the forces.  The @c StepSystem_tag class is
 * used for input and output data that is updated throughout the
 * simulation.  Input data includes the initial position and velocity
 * arrays, and output data includes updated position and veolcity
 * arrays along with current forces.
 *
 * The units are as outlined by the MDAPI documentation:
 * @li distance in Angstroms
 * @li time in femtoseconds
 * @li velocity in Angstroms/femtosecond
 * @li energy in kcal/mol
 * @li force in kcal/mol/Angstrom
 * @li temperature in degrees Kelvin
 */

#ifndef STEP_H
#define STEP_H

#include "mdapi/mdtypes.h"
#include "random/random.h"

#ifdef __cplusplus
extern "C" {
#endif

#define STEP_FAIL   (-1)
  /**< Return value indicating an error occurred. */

#define STEP_TRUE   (1)
  /**< Indicates @c StepSystem_tag::is_force_valid is true. */

#define STEP_FALSE  (0)
  /**< Indicates @c StepSystem_tag::is_force_valid is false. */


  /**@brief Helper class to configure parameters.
   *
   * The user is expected to fill in these values.  User retains ownership
   * of the atom array.  The only thing that might be modified as a
   * side-effect is @c StepParam_tag::force_object, and that depends on the
   * nature of the provided @c StepParam_tag::force_compute() routine.
   */
  typedef struct StepParam_tag {
    unsigned long random_seed;
    /**< Seed for the random number generator. */
    double timestep;
    /**< Time step (in femtoseconds). */
    void *force_object;
    /**< The @c force_object is intended to provide state to the
     *   force_compute() routine, if necessary. */
    int32 (*force_compute)(void *force_object, MD_Dvec *force, MD_Dvec *pos);
    /**< Must be provided by user.  Computes the @c force array for the
     *   given @c pos array of atomic positions.  Returns 0 for success. */
    MD_Atom *atom;
    /**< Array of atom data for system.  (Only the mass is needed.) */
    int32 natoms;
    /**< Number of atoms, length of @c atom array. */
    int32 ndegfreedom;
    /**< Number of degrees of freedom.  This is generally 3 times the
     *   number of atoms, less 3 if center of mass motion is removed.
     *   Needed to calculate the temperature. */
  } StepParam;


  /**@brief Helper class providing system data.
   *
   * The user is expected to provide buffer space for the
   * @c StepSystem_tag::force, @c StepSystem_tag::vel, and
   * @c StepSystem_tag::pos arrays.  They must be of length
   * @c StepParam_tag::natoms.
   * The @c StepSystem_tag::is_force_valid should be set to
   * @c STEP_FALSE for the initial call to @c step_compute().
   */
  typedef struct StepSystem_tag {
    double kinetic_energy;
    /**< Contains kinetic energy of the system upon return from
     *   @c step_find_reductions(). */
    double temperature;
    /**< Contains temperature of the system upon return from
     *   @c step_find_reductions(). */
    MD_Dvec *force;
    /**< Forces are updated in place during time stepping.
     *   Will contain forces for final positions upon return
     *   from @c step_compute(). */
    MD_Dvec *vel;
    /**< Must contain initial velocities for the system for the
     *   first call to @c step_compute().
     *   Velocities are updated in place during time stepping.
     *   Will contain final velocities upon return from
     *   @c step_compute(). */
    MD_Dvec *pos;
    /**< Must contain initial positions for the system for the
     *   first call to @c step_compute().
     *   Positions are updated in place during time stepping.
     *   Will contain final positions upon return from
     *   @c step_compute(). */
    int32 is_force_valid;
    /**< Set to either @c STEP_TRUE if @c force array contains valid
     *   forces for given @c pos array, or @c STEP_FALSE otherwise. */
  } StepSystem;


  /**@brief Time stepping class.
   *
   * Members should be treated as private.
   */
  typedef struct Step_tag {
    Random random;          /* random number generator */
    double tempkonst;       /* constant needed to compute temperature */
    StepParam *param;       /* shallow copy of user StepParam object */
    double *scal_inv_mass;  /* compute array of scaled inverse mass,
                               for efficiency */
  } Step;


  /**@brief Constructor.
   *
   * @param[in] param  The @c StepParam_tag object used for configuration.
   *
   * Initializes a @c Step_tag object.
   * Must call this routine first.
   *
   * @return 0 for success or @c STEP_FAIL on failure.
   */
  int step_init(Step *, StepParam *param);


  /**@brief Perform time stepping.
   *
   * @param[in,out] sys   Contains the state of the system.
   * @param[in] numsteps  Number of time steps to integrate the system.
   *
   * The provided system @c sys is integrated for @c numsteps
   * time steps using leapfrog (velocity Verlet).  The force routine
   * provided to @c StepParam_tag::force_compute() is called once per
   * time step and optionally once initially depending on the value
   * of @c StepSystem_tag::is_force_valid.  The buffer space provided
   * to @c StepSystem_tag is updated in place.
   *
   * Note that calls to the force routine account for the majority
   * of the computational cost.
   *
   * @return 0 for success or @c STEP_FAIL on failure.
   */
  int step_compute(Step *, StepSystem *sys, int32 numsteps);


  /**@brief Calculate reductions.
   *
   * @param[in,out] sys   Contains the state of the system.
   *
   * Calculates updates to the kinetic energy and the temperature
   * which are stored in @c StepSystem_tag.
   *
   * @return 0 for success or @c STEP_FAIL on failure.
   */
  int step_find_reductions(Step *, StepSystem *sys);


  /**@brief Set initial velocities.
   *
   * @param[out] sys       Contains the state of the system.
   * @param[in] init_temp  Initial temperature distribution.
   *
   * Determines random velocities for the initial temperature distribution.
   *
   * @return 0 for success or @c STEP_FAIL on failure.
   */
  int step_set_random_velocities(Step *, StepSystem *sys, double init_temp);


  /**@brief Remove center of mass motion.
   *
   * @param[in,out] sys    Contains the state of the system.
   *
   * Removes the center of mass motion for the system.  Doing so reduces
   * the number of degrees of freedom of the system by 3 (i.e. one atom),
   * so @c StepParam_tag::ndegfreedom should be set accordingly.
   */
  int step_remove_com_motion(Step *, StepSystem *sys);


  /**@brief Destructor.
   *
   * Frees memory allocations made internally during @c step_init().
   * Call when finished using @c Step_tag object.
   */
  void step_done(Step *);

#ifdef __cplusplus
}
#endif

#endif /* STEP_H */
