/*
 * Copyright (C) 2005 by David J. Hardy.  All rights reserved.
 *
 * step.c
 */

#include <stdlib.h>
#include "step/step.h"
#include "debug/debug.h"


int step_init(Step *s, StepParam *param)
{
  int32 i;

  /* check parameters */
  if (param->natoms <= 0) return STEP_FAIL;
  if (param->ndegfreedom <= 0) return STEP_FAIL;
  if (param->force_compute == NULL) return STEP_FAIL;
  if (param->atom == NULL) return STEP_FAIL;
  for (i = 0;  i < param->natoms;  i++) {
    if (param->atom[i].m <= 0.0) return STEP_FAIL;
  }

  /* set internals, allocate memory */
  s->tempkonst = 2.0 / (param->ndegfreedom * MD_BOLTZMAN);
  s->param = param;
  s->scal_inv_mass = (double *) malloc(param->natoms * sizeof(double));
  if (s->scal_inv_mass == NULL) return STEP_FAIL;
  for (i = 0;  i < param->natoms;  i++) {
    s->scal_inv_mass[i] = MD_FORCE_CONST / param->atom[i].m;
  }
  random_initseed(&(s->random), param->random_seed);
  return 0;
}


void step_done(Step *s)
{
  free(s->scal_inv_mass);
}


int step_compute(Step *s, StepSystem *sys, int32 numsteps)
{
  const double dt = s->param->timestep;
  const double half_dt = 0.5 * dt;
  double konst;
  void *f_obj = s->param->force_object;
  int32 (*f_eval)(void *, MD_Dvec *, MD_Dvec *) = s->param->force_compute;
  MD_Dvec *f = sys->force;
  MD_Dvec *vel = sys->vel;
  MD_Dvec *pos = sys->pos;
  const double *scal_inv_mass = s->scal_inv_mass;
  const int32 natoms = s->param->natoms;
  int32 n, i;

  /* compute force if not valid */
  if ( ! sys->is_force_valid) {
    if (f_eval(f_obj, f, pos)) return STEP_FAIL;
    sys->is_force_valid = STEP_TRUE;
  }

  /* perform leapfrog integration of system for numsteps */
  for (n = 0;  n < numsteps;  n++) {

    /* half kick */
    for (i = 0;  i < natoms;  i++) {
      konst = half_dt * scal_inv_mass[i];
      vel[i].x += konst * f[i].x;
      vel[i].y += konst * f[i].y;
      vel[i].z += konst * f[i].z;
    }

    /* drift */
    for (i = 0;  i < natoms;  i++) {
      pos[i].x += dt * vel[i].x;
      pos[i].y += dt * vel[i].y;
      pos[i].z += dt * vel[i].z;
    }

    /* compute force for updated position */
    if (f_eval(f_obj, f, pos)) {
      sys->is_force_valid = STEP_FALSE;
      return STEP_FAIL;
    }

    /* half kick */
    for (i = 0;  i < natoms;  i++) {
      konst = half_dt * scal_inv_mass[i];
      vel[i].x += konst * f[i].x;
      vel[i].y += konst * f[i].y;
      vel[i].z += konst * f[i].z;
    }

  }

  return 0;
}


int step_find_reductions(Step *s, StepSystem *sys)
{
  const MD_Atom *atom = s->param->atom;
  const MD_Dvec *vel = sys->vel;
  const int32 natoms = s->param->natoms;
  double accum = 0.0;
  int32 i;

  /* compute kinetic energy */
  for (i = 0;  i < natoms;  i++) {
    accum += (vel[i].x * vel[i].x + vel[i].y * vel[i].y
        + vel[i].z * vel[i].z) * atom[i].m;
  }
  sys->kinetic_energy = 0.5 * MD_KCAL_MOL * accum;
  sys->temperature = s->tempkonst * sys->kinetic_energy;
  return 0;
}


int step_set_random_velocities(Step *s, StepSystem *sys, double init_temp)
{
  const double kbtemp = MD_BOLTZMAN * MD_ENERGY_CONST * init_temp;
  double sqrt_kbtemp_div_mass;
#ifndef STEP_ALT_INITTEMP
  double rnum;
#endif
  const MD_Atom *atom = s->param->atom;
  MD_Dvec *vel = sys->vel;
  Random *r = &(s->random);
  const int32 natoms = s->param->natoms;
  int32 n, k;

  /* make sure initial temperature is valid */
  if (init_temp < 0.0) return STEP_FAIL;

  for (n = 0;  n < natoms;  n++) {
    sqrt_kbtemp_div_mass = sqrt(kbtemp / atom[n].m);

#ifndef STEP_ALT_INITTEMP
    /*
     * The following method and comments taken from NAMD WorkDistrib.C:
     *
     * //  The following comment was stolen from X-PLOR where
     * //  the following section of code was adapted from.
     *
     * //  This section generates a Gaussian random
     * //  deviate of 0.0 mean and standard deviation RFD for
     * //  each of the three spatial dimensions.
     * //  The algorithm is a "sum of uniform deviates algorithm"
     * //  which may be found in Abramowitz and Stegun,
     * //  "Handbook of Mathematical Functions", pg 952.
     */
    rnum = -6.0;
    for (k = 0;  k < 12;  k++) {
      rnum += random_uniform(r);
    }
    vel[n].x = sqrt_kbtemp_div_mass * rnum;

    rnum = -6.0;
    for (k = 0;  k < 12;  k++) {
      rnum += random_uniform(r);
    }
    vel[n].y = sqrt_kbtemp_div_mass * rnum;

    rnum = -6.0;
    for (k = 0;  k < 12;  k++) {
      rnum += random_uniform(r);
    }
    vel[n].z = sqrt_kbtemp_div_mass * rnum;
#else
    /*
     * Alternate method from NAMD Sequencer.C:
     */
    vel[n].x = sqrt_kbtemp_div_mass * random_gaussian(r);
    vel[n].y = sqrt_kbtemp_div_mass * random_gaussian(r);
    vel[n].z = sqrt_kbtemp_div_mass * random_gaussian(r);
#endif
  }
  return 0;
}


int step_remove_com_motion(Step *s, StepSystem *sys)
{
  const MD_Atom *atom = s->param->atom;
  MD_Dvec *vel = sys->vel;
  const int32 natoms = s->param->natoms;
  int32 i;
  MD_Dvec mv = { 0.0, 0.0, 0.0 };  /* accumulate net momentum */
  double mass = 0.0;               /* accumulate total mass */

  /* compute net momentum and total mass */
  for (i = 0;  i < natoms;  i++) {
    mv.x += atom[i].m * vel[i].x;
    mv.y += atom[i].m * vel[i].y;
    mv.z += atom[i].m * vel[i].z;
    mass += atom[i].m;
  }

  /* scale net momentum by total mass */
  mv.x /= mass;
  mv.y /= mass;
  mv.z /= mass;

  /* remove from atom velocities */
  for (i = 0;  i < natoms;  i++) {
    vel[i].x -= mv.x;
    vel[i].y -= mv.y;
    vel[i].z -= mv.z;
  }
  return 0;
}
