/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * split.c
 */

#include "mgrid/split.h"
#include "debug/debug.h"


#ifndef MGRID_FAST_SPLIT


void mgrid_gamma(double *pg, double s, int32 split)
{
  double g;
  /* double s = r_a * r_a; */

  ASSERT(s <= 1);
  ASSERT(pg != NULL);
  switch (split) {
    case MGRID_TAYLOR1:
      g = 1 + (s-1)*(-1./2);
      break;
    case MGRID_TAYLOR2:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8));
      break;
    case MGRID_TAYLOR3:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16)));
      break;
    case MGRID_TAYLOR4:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128))));
      break;
    case MGRID_TAYLOR5:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256)))));
      break;
    case MGRID_TAYLOR6:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024))))));
      break;
    case MGRID_TAYLOR7:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024 + (s-1)*(-429./2048)))))));
      break;
    case MGRID_TAYLOR8:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024 + (s-1)*(-429./2048
                      + (s-1)*(6435./32768))))))));
      break;
    case MGRID_ERRMIN3:
      g = MGRID_ERRMIN3_D + s*(MGRID_ERRMIN3_C + s*(MGRID_ERRMIN3_B
            + s*MGRID_ERRMIN3_A));
      break;
    default:
      BUG("invalid splitting function");
  }
  *pg = g;
}


void mgrid_dgamma(double *pg, double *pdg, double s, int32 split)
{
  double g, dg;
  /* double s = r_a * r_a; */

  ASSERT(s <= 1);
  ASSERT(pg != NULL);
  ASSERT(pdg != NULL);
  switch (split) {
    case MGRID_TAYLOR1:
      g = 1 + (s-1)*(-1./2);
      dg = -1./2;
      break;
    case MGRID_TAYLOR2:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8));
      dg = -1./2 + (s-1)*(3./4);
      break;
    case MGRID_TAYLOR3:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16)));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16));
      break;
    case MGRID_TAYLOR4:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32)));
      break;
    case MGRID_TAYLOR5:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256)))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
              + (s-1)*(-315./256))));
      break;
    case MGRID_TAYLOR6:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024))))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
              + (s-1)*(-315./256 + (s-1)*(693./512)))));
      break;
    case MGRID_TAYLOR7:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024 + (s-1)*(-429./2048)))))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
              + (s-1)*(-315./256 + (s-1)*(693./512
                  + (s-1)*(-3003./2048))))));
      break;
    case MGRID_TAYLOR8:
      g = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
              + (s-1)*(35./128 + (s-1)*(-63./256
                  + (s-1)*(231./1024 + (s-1)*(-429./2048
                      + (s-1)*(6435./32768))))))));
      dg = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
              + (s-1)*(-315./256 + (s-1)*(693./512
                  + (s-1)*(-3003./2048 + (s-1)*(6435./4096)))))));
      break;
    case MGRID_ERRMIN3:
      g = MGRID_ERRMIN3_D + s*(MGRID_ERRMIN3_C + s*(MGRID_ERRMIN3_B
            + s*MGRID_ERRMIN3_A));
      dg = MGRID_ERRMIN3_C + s*(2*MGRID_ERRMIN3_B + s*3*MGRID_ERRMIN3_A);
      break;
    case MGRID_EXSELF1:
      if (s > MGRID_EXSELF1_swdis) {
        g = MGRID_EXSELF1_D + s*(MGRID_EXSELF1_C + s*(MGRID_EXSELF1_B
              + s*MGRID_EXSELF1_A));
        dg = MGRID_EXSELF1_C + s*(2*MGRID_EXSELF1_B + s*3*MGRID_EXSELF1_A);
      }
      else {
        g = MGRID_EXSELF1_F + s*MGRID_EXSELF1_E;
        dg = MGRID_EXSELF1_E;
      }
      break;
    default:
      BUG("invalid splitting function");
  }
  *pg = g;
  /* *pdg = dg * (2.0 * r_a); */
  *pdg = dg;
}


void mgrid_gamma_odd(double *pg, double r, int32 split)
{
  double g;
  double r2 = r*r;

  ASSERT(r <= 1);
  ASSERT(pg != NULL);
  ASSERT(pdg != NULL);
  switch (split) {
    case MGRID_ODDPR1:
      g = 3./2 + r2*(-1./2);
      break;
    case MGRID_ODDPR2:
      g = 2 + r2*(-2 + r);
      break;
    case MGRID_ODDPR3:
      g = 9./4 + r2*(-5./2 + r2*(9./4 - r));
      break;
    case MGRID_ODDPR4:
      g = 21./8 + r2*(-35./8 + r2*(63./8 + r*(-7 + r*(15./8))));
      break;
    case MGRID_ODDPR5:
      g = 45./16 + r2*(-21./4 + r2*(63./8 + r2*(-45./4
              + r*(9 + r*(-35./16)))));
      break;
    case MGRID_ODDPR6:
      g = 25./8 + r2*(-15./2 + r2*(63./4 + r2*(-75./2
              + r*(45 + r*(-175./8 + r*4)))));
      break;
    case MGRID_ODDPR7:
      g = 105./32 + r2*(-275./32 + r2*(297./16 + r2*(-495./16
              + r2*(1925./32 + r*(-66 + r*(945./32 + r*(-5)))))));
      break;
    case MGRID_ODDPR8:
      g = 455./128 + r2*(-715./64 + r2*(3861./128 + r2*(-2145./32
              + r2*(25025./128 + r*(-286 + r*(12285./64 + r*(-65
                      + r*(1155./128))))))));
      break;
    default:
      BUG("invalid splitting function");
  }
  *pg = g;
}


void mgrid_dgamma_odd(double *pg, double *pdg, double r, int32 split)
{
  double g, dg;
  double r2 = r*r;

  ASSERT(r <= 1);
  ASSERT(pg != NULL);
  ASSERT(pdg != NULL);
  switch (split) {
    case MGRID_ODDPR1:
      g = 3./2 + r2*(-1./2);
      dg = -r;
      break;
    case MGRID_ODDPR2:
      g = 2 + r2*(-2 + r);
      dg = r*(-4 + r*3);
      break;
    case MGRID_ODDPR3:
      g = 9./4 + r2*(-5./2 + r2*(9./4 - r));
      dg = r*(-5 + r2*(9 + r*(-5)));
      break;
    case MGRID_ODDPR4:
      g = 21./8 + r2*(-35./8 + r2*(63./8 + r*(-7 + r*(15./8))));
      dg = r*(-35./4 + r2*(63./2 + r*(-35 + r*(45./4))));
      break;
    case MGRID_ODDPR5:
      g = 45./16 + r2*(-21./4 + r2*(63./8 + r2*(-45./4
              + r*(9 + r*(-35./16)))));
      dg = r*(-21./2 + r2*(63./2 + r2*(-135./2
              + r*(63 + r*(-35./2)))));
      break;
    case MGRID_ODDPR6:
      g = 25./8 + r2*(-15./2 + r2*(63./4 + r2*(-75./2
              + r*(45 + r*(-175./8 + r*4)))));
      dg = r*(-15 + r2*(63 + r2*(-225
              + r*(315 + r*(-175 + r*36)))));
      break;
    case MGRID_ODDPR7:
      g = 105./32 + r2*(-275./32 + r2*(297./16 + r2*(-495./16
              + r2*(1925./32 + r*(-66 + r*(945./32 + r*(-5)))))));
      dg = r*(-275./16 + r2*(297./4 + r2*(-1485./8
              + r2*(1925./4 + r*(-594 + r*(4725./16 + r*(-55)))))));
      break;
    case MGRID_ODDPR8:
      g = 455./128 + r2*(-715./64 + r2*(3861./128 + r2*(-2145./32
              + r2*(25025./128 + r*(-286 + r*(12285./64 + r*(-65
                      + r*(1155./128))))))));
      dg = r*(-715./32 + r2*(3861./32 + r2*(-6435./16
              + r2*(25025./16 + r*(-2574 + r*(61425./32 + r*(-715
                      + r*(3465./32)))))))); 
      break;
    default:
      BUG("invalid splitting function");
  }
  *pg = g;
  *pdg = dg;
}


#endif /* not defined MGRID_FAST_SPLIT */
/* otherwise expand these functions as macros */


void mgrid_ndgamma(double *pg, int32 n, double s, int32 split)
{
  int32 k = 0;

  ASSERT(s >= 0);
  ASSERT(n >= 0);
  ASSERT(pg != NULL || n == 0);
  if (k == n) return;
  if (s <= 1) {
    /* compute derivatives of smoothed part */
    switch (split) {
      case MGRID_TAYLOR2:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4);
        if (k == n) break;
        pg[k++] = 3./4;
        break;
      case MGRID_TAYLOR3:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16)));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8);
        if (k == n) break;
        pg[k++] = -15./8;
        break;
      case MGRID_TAYLOR4:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32)));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16);
        if (k == n) break;
        pg[k++] = 105./16;
        break;
      case MGRID_TAYLOR5:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128 + (s-1)*(-63./256)))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
                + (s-1)*(-315./256))));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32 + (s-1)*(-315./64)));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16 + (s-1)*(-945./64));
        if (k == n) break;
        pg[k++] = 105./16 + (s-1)*(-945./32);
        if (k == n) break;
        pg[k++] = -945./32;
        break;
      case MGRID_TAYLOR6:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128 + (s-1)*(-63./256 + (s-1)*(231./1024))))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
                + (s-1)*(-315./256 + (s-1)*(693./512)))));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32 + (s-1)*(-315./64
                + (s-1)*(3465./512))));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16 + (s-1)*(-945./64
              + (s-1)*(3465./128)));
        if (k == n) break;
        pg[k++] = 105./16 + (s-1)*(-945./32 + (s-1)*(10395./128));
        if (k == n) break;
        pg[k++] = -945./32 + (s-1)*(10395./64);
        if (k == n) break;
        pg[k++] = 10395./64;
        break;
      case MGRID_TAYLOR7:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128 + (s-1)*(-63./256
                    + (s-1)*(231./1024 + (s-1)*(-429./2048)))))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
                + (s-1)*(-315./256 + (s-1)*(693./512
                    + (s-1)*(-3003./2048))))));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32 + (s-1)*(-315./64
                + (s-1)*(3465./512 + (s-1)*(-9009./1024)))));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16 + (s-1)*(-945./64 + (s-1)*(3465./128
                + (s-1)*(-45045./1024))));
        if (k == n) break;
        pg[k++] = 105./16 + (s-1)*(-945./32 + (s-1)*(10395./128
              + (s-1)*(-45045./256)));
        if (k == n) break;
        pg[k++] = -945./32 + (s-1)*(10395./64 + (s-1)*(-135135./256));
        if (k == n) break;
        pg[k++] = 10395./64 + (s-1)*(-135135./128);
        if (k == n) break;
        pg[k++] = -135135./128;
        break;
      case MGRID_TAYLOR8:
        pg[k++] = 1 + (s-1)*(-1./2 + (s-1)*(3./8 + (s-1)*(-5./16
                + (s-1)*(35./128 + (s-1)*(-63./256
                    + (s-1)*(231./1024 + (s-1)*(-429./2048
                        + (s-1)*(6435./32768))))))));
        if (k == n) break;
        pg[k++] = -1./2 + (s-1)*(3./4 + (s-1)*(-15./16 + (s-1)*(35./32
                + (s-1)*(-315./256 + (s-1)*(693./512
                    + (s-1)*(-3003./2048 + (s-1)*(6435./4096)))))));
        if (k == n) break;
        pg[k++] = 3./4 + (s-1)*(-15./8 + (s-1)*(105./32 + (s-1)*(-315./64
                + (s-1)*(3465./512 + (s-1)*(-9009./1024
                    + (s-1)*(45045./4096))))));
        if (k == n) break;
        pg[k++] = -15./8 + (s-1)*(105./16 + (s-1)*(-945./64 + (s-1)*(3465./128
                + (s-1)*(-45045./1024 + (s-1)*(135135./2048)))));
        if (k == n) break;
        pg[k++] = 105./16 + (s-1)*(-945./32 + (s-1)*(10395./128
              + (s-1)*(-45045./256 + (s-1)*(675675./2048))));
        if (k == n) break;
        pg[k++] = -945./32 + (s-1)*(10395./64 + (s-1)*(-135135./256
              + (s-1)*(675675./512)));
        if (k == n) break;
        pg[k++] = 10395./64 + (s-1)*(-135135./128 + (s-1)*(2027025./512));
        if (k == n) break;
        pg[k++] = -135135./128 + (s-1)*(2027025./256);
        if (k == n) break;
        pg[k++] = 2027025./256;
        break;
      case MGRID_ERRMIN3:
        pg[k++] = MGRID_ERRMIN3_D + s*(MGRID_ERRMIN3_C + s*(MGRID_ERRMIN3_B
              + s*MGRID_ERRMIN3_A));
        if (k == n) break;
        pg[k++] = MGRID_ERRMIN3_C + s*(2*MGRID_ERRMIN3_B + s*3*MGRID_ERRMIN3_A);
        if (k == n) break;
        pg[k++] = 2*MGRID_ERRMIN3_B + s*2*3*MGRID_ERRMIN3_A;
        if (k == n) break;
        pg[k++] = 2*3*MGRID_ERRMIN3_A;
        break;
      case MGRID_EXSELF1:
        if (s > MGRID_EXSELF1_swdis) {
          pg[k++] = MGRID_EXSELF1_D + s*(MGRID_EXSELF1_C + s*(MGRID_EXSELF1_B
                + s*MGRID_EXSELF1_A));
          if (k == n) break;
          pg[k++] = MGRID_EXSELF1_C + s*(2*MGRID_EXSELF1_B
              + s*3*MGRID_EXSELF1_A);
          if (k == n) break;
          pg[k++] = 2*MGRID_EXSELF1_B + s*2*3*MGRID_EXSELF1_A;
          if (k == n) break;
          pg[k++] = 2*3*MGRID_EXSELF1_A;
        }
        else {
          pg[k++] = MGRID_EXSELF1_F + s*MGRID_EXSELF1_E;
          if (k == n) break;
          pg[k++] = MGRID_EXSELF1_E;
        }
        break;
      default:
        BUG("invalid splitting function");
    }
    /* higher derivatives are zero */
    while (k < n) pg[k++] = 0;
  }
  else { /*  (s > 1)  */
    /* compute derivatives of s^(-1/2) */
    const double s_1 = 1./s;
    double s_p = sqrt(s_1);
    double p = -0.5;
    double c = 1;
    pg[k++] = c * s_p;
    while (k < n) {
      s_p *= s_1;
      c *= p;
      p -= 1;
      pg[k++] = c * s_p;
    }
  }
}
