/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * lattice.c
 *
 * 3D grid lattice ADT in which user specifies element size and index
 * ranges.  Access function performs range checking.
 *
 * Storage in column-major order like Fortran, access memory sequentially
 * by looping over k in outer loop, j in middle loop, i in inner loop:
 *
 *     for k in ka..kb {
 *       for j in ja..jb {
 *         for i in ia..ib {
 *           access lattice element (i,j,k)
 *         }
 *       }
 *     }
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mgrid/lattice.h"
#include "mgrid/mgrid.h"
#include "debug/debug.h"


int mgrid_lattice_init(MgridLattice *g)
{
  ASSERT(g != NULL);
  memset(g, 0, sizeof(MgridLattice));
  return 0;
}


int mgrid_lattice_setup(MgridLattice *g, int32 elemsz,
    int32 ia, int32 ib, int32 ja, int32 jb, int32 ka, int32 kb)
{
  int32 ni, nj, nk, nelems, nbytes;

  ASSERT(g != NULL);

  /* check for argument sanity */
  if (elemsz <= 0 || ia > ib || ja > jb || ka > kb) return MGRID_FAIL;

  /* initialize size and range members */
  ni = ib - ia + 1;
  nj = jb - ja + 1;
  nk = kb - ka + 1;
  nelems = ni * nj * nk;
  nbytes = nelems * elemsz;

  /* grab more memory if needed */
  if (nbytes > g->nbytes) {
    void *tmp = realloc(g->databuffer, nbytes);
    if (tmp == NULL) return MGRID_FAIL;
    g->databuffer = tmp;
    g->nbytes = nbytes;
  }

  /* relocate data pointer to (0,0,0)th element */
  /* (by doing it once here, we never have to index range-shift again) */
  g->data = (char *)(g->databuffer) + ((-ka * nj - ja) * ni - ia) * elemsz;

  /* store size and range */
  g->elemsz = elemsz;
  g->ia = ia;
  g->ib = ib;
  g->ni = ni;
  g->ja = ja;
  g->jb = jb;
  g->nj = nj;
  g->ka = ka;
  g->kb = kb;
  g->nk = nk;
  g->nelems = nelems;
  return 0;
}


int mgrid_lattice_zero(MgridLattice *g)
{
  ASSERT(g != NULL);
  memset(g->databuffer, 0, g->nelems * g->elemsz);
  return 0;
}


void *mgrid_lattice_elem(MgridLattice *g, int32 i, int32 j, int32 k)
{
  ASSERT(g != NULL);

  if (g->ia <= i && i <= g->ib
      && g->ja <= j && j <= g->jb
      && g->ka <= k && k <= g->kb) {
    return (char *)(g->data) + ((k * g->nj + j) * g->ni + i) * g->elemsz;
  }
  return NULL;
}


void mgrid_lattice_done(MgridLattice *g)
{
  ASSERT(g != NULL);
  free(g->databuffer);
}


int mgrid_lattice_print(MgridLattice *g)
{
  int32 i, j, k;

  ASSERT(g != NULL);
  if (g->elemsz != sizeof(double)) return -1;
  printf("lattice:  ia=%d  ib=%d  ni=%d\n"
         "          ja=%d  jb=%d  nj=%d\n"
         "          ka=%d  kb=%d  nk=%d\n",
         g->ia, g->ib, g->ni, g->ja, g->jb, g->nj, g->ka, g->kb, g->nk);
  for (k = g->ka;  k <= g->kb;  k++) {
    printf("slice:  k=%d  (columns: j=%d..%d, rows: i=%d..%d)\n",
        k, g->ja, g->jb, g->ia, g->ib);
    for (j = g->ja;  j <= g->jb;  j++) {
      for (i = g->ia;  i <  g->ib;  i++) {
        printf("%13e ", *((double *)mgrid_lattice_elem(g, i, j, k)));
      }
      printf("%13e\n", *((double *)mgrid_lattice_elem(g, i, j, k)));
    }
  }
  return 0;
}
