/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * exact.c - compute exact smooth part (for nonperiodic boundaries)
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "mgrid/split.h"
#include "debug/debug.h"


int mgrid_exact_smooth(Mgrid *mg, MgridSystem *sys)
{
  const double *q = sys->charge;
  const MD_Dvec *pos = sys->pos;

  /*
   * store force to long range array if given
   * otherwise overwrite total force array
   */
  MD_Dvec *f = (sys->f_long != NULL ? sys->f_long : sys->f_elec);

  const double a2 = mg->param.cutoff * mg->param.cutoff;
  const double a_1 = mg->inv_cutoff;
  const double a_2 = a_1 * a_1;
  const double two_a3 = 2.0 * a_2 * a_1;
  double u = 0.0;
  MD_Dvec pj, fj, r_ij;
  double qj, r2, s, g, dg, r_2, r_1, qc, uc, fc, f_x, f_y, f_z;

  const int32 natoms = mg->param.natoms;
  const int32 split = mg->param.split;
  int32 i, j;


  if (mg->param.boundary != MGRID_NONPERIODIC) return MGRID_FAIL;

  /* clear memory before accumulation of force */
  memset(f, 0, natoms * sizeof(MD_Dvec));

  /* loop over all atoms */
  for (j = 0;  j < natoms;  j++) {

    pj.x = pos[j].x;
    pj.y = pos[j].y;
    pj.z = pos[j].z;
    qj = q[j];
    fj = f[j];

    /* loop over remaining atoms */
    for (i = j + 1;  i < natoms;  i++) {

      r_ij.x = pj.x - pos[i].x;
      r_ij.y = pj.y - pos[i].y;
      r_ij.z = pj.z - pos[i].z;
      r2 = r_ij.x * r_ij.x + r_ij.y * r_ij.y + r_ij.z * r_ij.z;

      r_2 = 1.0 / r2;
      r_1 = sqrt(r_2);

      qc = q[i] * qj;

      if (r2 < a2) {
        s = r2 * a_2;
        dgamma(&g, &dg, s, split);
        uc = qc * a_1 * g;
        fc = -qc * two_a3 * dg;
      }
      else {
        uc = qc * r_1;
        fc = uc * r_2;
      }

      f_x = fc * r_ij.x;
      f_y = fc * r_ij.y;
      f_z = fc * r_ij.z;

      fj.x += f_x;
      fj.y += f_y;
      fj.z += f_z;
      f[i].x -= f_x;
      f[i].y -= f_y;
      f[i].z -= f_z;
      u += uc;

    } /* end loop over remaining atoms */

    f[j] = fj;

  } /* end loop over all atoms */

  /* save to long range potential, leave others alone */
  sys->u_long = u;
  return 0;
}
