/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * demo_mgrid.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mgrid/mgrid.h"

int main(void)
{
#define N  2
  MgridParam p;
  MgridSystem sys;
  Mgrid mg;
  MD_Dvec pos[N] = { {-2, -2, -2}, {2, 2, 2} };
  MD_Dvec f[N];
  double q[N] = { 1, -1 };
  const MD_Dvec center = { 0, 0, 0 };
  const double h = 1.25;
  const double a = 9.0;
  const double len = 20.0;
  double u, du_r, r, r2;
  MD_Dvec r_ij;
  int32 i, j;

  /* print system */
  printf("  System\n");
  for (j = 0;  j < N;  j++) {
    printf("pos[%d] = { %g, %g, %g }", j, pos[j].x, pos[j].y, pos[j].z);
    printf("     charge[%d] = %g\n", j, q[j]);
  }

  /* direct evaluation */
  u = 0;
  memset(f, 0, N * sizeof(MD_Dvec));
  for (j = 1;  j < N;  j++) {
    for (i = 0;  i < j;  i++) {
      r_ij.x = pos[j].x - pos[i].x;
      r_ij.y = pos[j].y - pos[i].y;
      r_ij.z = pos[j].z - pos[i].z;
      r2 = r_ij.x * r_ij.x + r_ij.y * r_ij.y + r_ij.z * r_ij.z;
      r = sqrt(r2);
      u += q[i] * q[j] / r;
      du_r = -q[i] * q[j] / (r * r2);
      f[i].x += r_ij.x * du_r;
      f[i].y += r_ij.y * du_r;
      f[i].z += r_ij.z * du_r;
      f[j].x -= r_ij.x * du_r;
      f[j].y -= r_ij.y * du_r;
      f[j].z -= r_ij.z * du_r;
    }
  }
  printf("\n  Direct evaluation\n");
  printf("pe = %g\n", u);
  for (j = 0;  j < N;  j++) {
    printf("f[%d] = { %g, %g, %g }\n", j, f[j].x, f[j].y, f[j].z);
  }

  if (mgrid_init(&mg)) {
    fprintf(stderr, "mgrid_init() failed\n");
    exit(1);
  }
  memset(&p, 0, sizeof(MgridParam));
  memset(&sys, 0, sizeof(MgridSystem));

  /* setup system */
  sys.f_elec = f;
  sys.pos = pos;
  sys.charge = q;

  /* setup nonperiodic params */
  p.boundary = MGRID_NONPERIODIC;
  p.natoms = N;
  p.center = center;
  p.length = len;
  p.cutoff = a;
  p.spacing = h;
  p.nspacings = len / h;
  p.nlevels = 5;
  p.approx = MGRID_CUBIC;
  p.split = MGRID_TAYLOR2;

  if (mgrid_setup(&mg, &sys, &p)) {
    fprintf(stderr, "mgrid_setup() failed\n");
    exit(1);
  }

  printf("\n  Multigrid parameters\n");
  printf("center = (%g, %g, %g)\n", p.center.x, p.center.y, p.center.z);
  printf("length = %g\n", p.length);
  printf("cutoff = %g\n", p.cutoff);
  printf("spacing = %g\n", p.spacing);
  printf("nspacings = %d\n", p.nspacings);
  printf("nlevels = %d\n", p.nlevels);
  printf("boundary = (MGRID_%s)\n",
      (p.boundary == MGRID_NONPERIODIC ?  "NONPERIODIC" : "PERIODIC"));
  printf("natoms = %d\n", p.natoms);
  printf("approx = (MGRID_%s)\n",
      (p.approx == MGRID_CUBIC ? "CUBIC" :
       (p.approx == MGRID_QUINTIC1 ? "QUINTIC1" :
        (p.approx == MGRID_QUINTIC2 ? "QUINTIC2" :
         (p.approx == MGRID_HERMITE ? "HERMITE" :
          "\b\b\b\b\b\b***unknown***")))));
  printf("split = (MGRID_%s)\n",
      (p.approx == MGRID_TAYLOR2 ? "TAYLOR2" :
       (p.approx == MGRID_TAYLOR3 ? "TAYLOR3" :
        (p.approx == MGRID_TAYLOR4 ? "TAYLOR4" :
         "\b\b\b\b\b\b***unknown***"))));

  if (mgrid_force(&mg, &sys)) {
    fprintf(stderr, "mgrid_force() failed\n");
    exit(1);
  }

  printf("\n  Multigrid evaluation\n");
  printf("pe = %g\n", sys.u_elec);
  for (j = 0;  j < N;  j++) {
    printf("f[%d] = { %g, %g, %g }\n", j, f[j].x, f[j].y, f[j].z);
  }

  mgrid_done(&mg);

  return 0;
}
