/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * system.c
 */

#include <stdlib.h>
#include <string.h>
#include "mdio/mdio.h"
#include "mdsim/system.h"
#include "mdsim/simparam.h"
#include "mdsim/error.h"
#include "mdsim/const.h"
#include "debug/debug.h"


/*
 * System constructor
 * clear memory and initialize file readers
 */
int system_init(System *s)
{
  memset(s, 0, sizeof(System));

  if (mdio_initializeParam(&(s->prm))) {
    return error(MSG, "mdio_initializeParam() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->prm), mdio_errhandler);

  if (mdio_initializeTopo(&(s->topo))) {
    return error(MSG, "mdio_initializeTopo() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->topo), mdio_errhandler);

  if (mdio_initializePdbcoord(&(s->pdb))) {
    return error(MSG, "mdio_initializePdbcoord() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->pdb), mdio_errhandler);

  if (mdio_initializePdbcoord(&(s->pdbvel))) {
    return error(MSG, "mdio_initializePdbcoord() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->pdbvel), mdio_errhandler);

  if (mdio_initializeBincoord(&(s->bin))) {
    return error(MSG, "mdio_initializeBincoord() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->bin), mdio_errhandler);

  if (mdio_initializeBincoord(&(s->binvel))) {
    return error(MSG, "mdio_initializeBincoord() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->binvel), mdio_errhandler);

  return 0;
}


/*
 * validate access assumptions for engine
 * engine needs to have already been initialized
 */
int system_setup(System *s, MD_Engine *e)
{
  const int32 read = MD_READ;
  const int32 wlen = MD_WRITE | MD_SETLEN;
  const int32 rwlen = MD_READ | MD_WRITE | MD_SETLEN;

  /* store engine interface pointer */
  s->eng = e;

  /* check for essential engine data arrays */
  /*** treat "result" as essential ***/
  if ((s->atomprm_id = MD_idnum(e, "atomprm")) == MD_FAIL
      || (s->bondprm_id = MD_idnum(e, "bondprm")) == MD_FAIL
      || (s->angleprm_id = MD_idnum(e, "angleprm")) == MD_FAIL
      || (s->dihedprm_id = MD_idnum(e, "dihedprm")) == MD_FAIL
      || (s->imprprm_id = MD_idnum(e, "imprprm")) == MD_FAIL
      || (s->nbfixprm_id = MD_idnum(e, "nbfixprm")) == MD_FAIL
      || (s->atom_id = MD_idnum(e, "atom")) == MD_FAIL
      || (s->bond_id = MD_idnum(e, "bond")) == MD_FAIL
      || (s->angle_id = MD_idnum(e, "angle")) == MD_FAIL
      || (s->dihed_id = MD_idnum(e, "dihed")) == MD_FAIL
      || (s->impr_id = MD_idnum(e, "impr")) == MD_FAIL
      || (s->excl_id = MD_idnum(e, "excl")) == MD_FAIL
      || (s->pos_id = MD_idnum(e, "pos")) == MD_FAIL
      || (s->vel_id = MD_idnum(e, "vel")) == MD_FAIL
      || (s->result_id = MD_idnum(e, "result")) == MD_FAIL) {
    return error(MSG_ENGINE, "missing essential data array");
  }

  /* store attributes, might need to update later */
  s->atomprm_attr = MD_attrib(e, s->atomprm_id);
  s->bondprm_attr = MD_attrib(e, s->bondprm_id);
  s->angleprm_attr = MD_attrib(e, s->angleprm_id);
  s->dihedprm_attr = MD_attrib(e, s->dihedprm_id);
  s->imprprm_attr = MD_attrib(e, s->imprprm_id);
  s->nbfixprm_attr = MD_attrib(e, s->nbfixprm_id);
  s->atom_attr = MD_attrib(e, s->atom_id);
  s->bond_attr = MD_attrib(e, s->bond_id);
  s->angle_attr = MD_attrib(e, s->angle_id);
  s->dihed_attr = MD_attrib(e, s->dihed_id);
  s->impr_attr = MD_attrib(e, s->impr_id);
  s->excl_attr = MD_attrib(e, s->excl_id);
  s->pos_attr = MD_attrib(e, s->pos_id);
  s->vel_attr = MD_attrib(e, s->vel_id);
  s->result_attr = MD_attrib(e, s->result_id);

  /* check for type consistency */
  if (s->atomprm_attr.type != MD_ATOMPRM
      || s->bondprm_attr.type != MD_BONDPRM
      || s->angleprm_attr.type != MD_ANGLEPRM
      || s->dihedprm_attr.type != MD_TORSPRM
      || s->imprprm_attr.type != MD_TORSPRM
      || s->nbfixprm_attr.type != MD_NBFIXPRM
      || s->atom_attr.type != MD_ATOM
      || s->bond_attr.type != MD_BOND
      || s->angle_attr.type != MD_ANGLE
      || s->dihed_attr.type != MD_TORS
      || s->impr_attr.type != MD_TORS
      || s->excl_attr.type != MD_EXCL
      || (s->pos_attr.type != MD_DVEC /* && s->pos_attr.type != MD_FVEC */)
      || s->vel_attr.type != s->pos_attr.type
      || s->result_attr.type != MD_type(e, "Result")) {
    return error(MSG_ENGINE, "incorrect type on essential data array");
  }

  /* check minimal access permissions */
  if ((s->atomprm_attr.access & wlen) != wlen
      || (s->bondprm_attr.access & wlen) != wlen
      || (s->angleprm_attr.access & wlen) != wlen
      || (s->dihedprm_attr.access & wlen) != wlen
      || (s->imprprm_attr.access & wlen) != wlen
      || (s->nbfixprm_attr.access & wlen) != wlen
      || (s->atom_attr.access & wlen) != wlen
      || (s->bond_attr.access & wlen) != wlen
      || (s->angle_attr.access & wlen) != wlen
      || (s->dihed_attr.access & wlen) != wlen
      || (s->impr_attr.access & wlen) != wlen
      || (s->excl_attr.access & wlen) != wlen
      || (s->pos_attr.access & rwlen) != rwlen
      || (s->vel_attr.access & rwlen) != rwlen
      || (s->result_attr.access & read) != read) {
    return error(MSG_ENGINE,
        "incorrect access permissions on essential data array");
  }

  /* check attributes for result */
  if ((s->result_attr.access & (MD_RESIZE | MD_ERESIZE)) != 0
      || s->result_attr.len != 1
      || s->result_attr.max != 1) {
    return error(MSG_ENGINE, "incorrect attributes on \"result\"");
  }

  /* check for consistency of access permissions */
  if (s->atomprm_attr.access != s->bondprm_attr.access
      || s->atomprm_attr.access != s->angleprm_attr.access
      || s->atomprm_attr.access != s->dihedprm_attr.access
      || s->atomprm_attr.access != s->imprprm_attr.access
      || s->atomprm_attr.access != s->nbfixprm_attr.access
      || s->atom_attr.access != s->bond_attr.access
      || s->atom_attr.access != s->angle_attr.access
      || s->atom_attr.access != s->dihed_attr.access
      || s->atom_attr.access != s->impr_attr.access
      || s->atom_attr.access != s->excl_attr.access
      || s->pos_attr.access != s->vel_attr.access) {
    return error(MSG_ENGINE,
        "incorrect access permissions on essential data array");
  }
  return 0;
}


/*
 * use MDIO to read files (with file names given by SimParam)
 * initialize engine data arrays
 */
int system_input(System *s, SimParam *p)
{
  /* MDIO file reading objects */
  mdio_Param *prm = &(s->prm);
  mdio_Topo *topo = &(s->topo);
  mdio_Pdbcoord *pdb = &(s->pdb);
  mdio_Pdbcoord *pdbvel = &(s->pdbvel);
  mdio_Bincoord *bin = &(s->bin);
  mdio_Bincoord *binvel = &(s->binvel);

  /* define pointers for brevity */
  MD_Engine *e = s->eng;
  adt_List *pa;
  int32 len;

  /* other declarations */
  int32 k, npos, nvel;


  /* read force field param files */
  pa = &(p->parameters);
  len = (int32) adt_getLengthList(pa);
  for (k = 0;  k < len;  k++) {
    const char *fname = *((const char **) adt_indexList(pa, k));
    /* any errors parsing parameter files are unrecoverable */
    if (mdio_readParam(prm, fname)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) prm));
    }
  }

  /* read topology file */
  if (mdio_readTopo(topo, p->structure)) {
    return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) topo));
  }

  /* set topology indexing of params */
  if (mdio_indexParamTopo(topo, prm)) {
    return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) topo));
  }

  /* obtain param data */
  s->atomprm = mdio_getAtomParam(prm, &(s->atomprm_len));
  s->bondprm = mdio_getBondParam(prm, &(s->bondprm_len));
  s->angleprm = mdio_getAngleParam(prm, &(s->angleprm_len));
  s->dihedprm = mdio_getDihedParam(prm, &(s->dihedprm_len));
  s->imprprm = mdio_getImprParam(prm, &(s->imprprm_len));
  s->nbfixprm = mdio_getNbfixParam(prm, &(s->nbfixprm_len));

  /* obtain topology data */
  s->atom = mdio_getAtomTopo(topo, &(s->atom_len));
  s->bond = mdio_getBondTopo(topo, &(s->bond_len));
  s->angle = mdio_getAngleTopo(topo, &(s->angle_len));
  s->dihed = mdio_getDihedTopo(topo, &(s->dihed_len));
  s->impr = mdio_getImprTopo(topo, &(s->impr_len));
  s->excl = mdio_getExclTopo(topo, &(s->excl_len));

  /* check number of atoms and atomprms */
  if (s->atom_len <= 0) {
    return error("%d atoms in system", s->atom_len);
  }
  else if (s->atomprm_len <= 0) {
    return error("%d atom parameters", s->atomprm_len);
  }

  /* read positions from PDB file for expected length */
  if (mdio_readPdbcoord(pdb, p->coordinates, s->atom_len)) {
    return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) pdb));
  }
  s->pdbatompos = mdio_getAtomPdbcoord(pdb, &npos);
  if (npos != s->atom_len) {
    return error("number of atoms %d in coordinate PDB differs from "
        "%d atoms in PSF", npos, s->atom_len);
  }

  /* set "pdbatomvel" to also point to extended atom position info (for now) */
  s->pdbatomvel = s->pdbatompos;

  if (p->bincoordinates) { 
    /* read positions from binary coordinate file for expected length */
    if (mdio_readBincoord(bin, p->bincoordinates, s->atom_len)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) bin));
    }
    s->pos = mdio_getBincoord(bin, &npos);
    if (npos != s->atom_len) {
      return error("number of atoms %d in binary position file differs from "
          "%d atoms in PSF", npos, s->atom_len);
    }
  }
  else {
    /* otherwise use atom positions from PDB file */
    s->pos = mdio_getPdbcoord(pdb, &npos);
    ASSERT(npos == s->atom_len);
  }

  /*
   * note that we check first for "velocities" file, then for "binvelocities"
   * (it is turned around from "coordinates" to imitate behavior of NAMD)
   */
  if (p->velocities) {
    /* read velocities from PDB file for expected length */
    if (mdio_readPdbcoord(pdbvel, p->velocities, s->atom_len)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) pdbvel));
    }
    /* reset "pdbatomvel" to point to extended velocity info */
    s->pdbatomvel = mdio_getAtomPdbcoord(pdbvel, &nvel);
    s->vel = mdio_getPdbcoord(pdbvel, &nvel);
    if (nvel != s->atom_len) {
      return error("number of atoms %d in velocity PDB differs from "
          "%d atoms in PSF", nvel, s->atom_len);
    }
    /* must convert pdb velocities from A/ps to internal units A/fs */
    for (k = 0;  k < nvel;  k++) {
      s->vel[k].x *= MD_ANGSTROM_FS;
      s->vel[k].y *= MD_ANGSTROM_FS;
      s->vel[k].z *= MD_ANGSTROM_FS;
    }
  }
  else if (p->binvelocities) {
    /* read velocities from binary coordinate file for expected length */
    if (mdio_readBincoord(binvel, p->binvelocities, s->atom_len)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) binvel));
    }
    s->vel = mdio_getBincoord(binvel, &nvel);
    if (nvel != s->atom_len) {
      return error("number of atoms %d in binary velocity file differs from "
          "%d atoms in PSF", nvel, s->atom_len);
    }
  }
  else {
    /*
     * allocate zeroed velocity array and assume engine will set
     * using some initial temperature
     */
    nvel = s->atom_len;
    if ((s->vel = (MD_Dvec *) calloc(nvel, sizeof(MD_Dvec))) == NULL) {
      return error(MSG_NOMEM);
    }
    s->is_vel_alloc = TRUE;
  }

  ASSERT(npos == s->atom_len);
  ASSERT(nvel == s->atom_len);

  /* allocate extra buffers that might be needed */
  s->posbuf = (MD_Dvec *) calloc(s->atom_len, sizeof(MD_Dvec));
  s->velbuf = (MD_Dvec *) calloc(s->atom_len, sizeof(MD_Dvec));
  s->resultbuf = calloc(1, MD_SIZEOF(s->result_attr.type)); 
  if (s->posbuf == NULL || s->velbuf == NULL || s->resultbuf == NULL) {
    return error(MSG_NOMEM);
  }

  /* set force field parameter engine arrays */
  if (s->atomprm_attr.access & MD_SHARE) {
    /* all param arrays have SHARE access */
    if (MD_share(e, s->atomprm_id, s->atomprm, s->atomprm_len,
          s->atomprm_len)
        || MD_share(e, s->bondprm_id, s->bondprm, s->bondprm_len,
          s->bondprm_len)
        || MD_share(e, s->angleprm_id, s->angleprm, s->angleprm_len,
          s->angleprm_len)
        || MD_share(e, s->dihedprm_id, s->dihedprm, s->dihedprm_len,
          s->dihedprm_len)
        || MD_share(e, s->imprprm_id, s->imprprm, s->imprprm_len,
          s->imprprm_len)
        || MD_share(e, s->nbfixprm_id, s->nbfixprm, s->nbfixprm_len,
          s->nbfixprm_len)) {
      return error(MSG_MDAPI, "from MD_share()", MD_errmsg(e));
    }
  }
  else {
    /* fall back on WRITE access - but it increases memory use */
    /* resize arrays to desired length */
    if (MD_setlen(e, s->atomprm_id, s->atomprm_len)
        || MD_setlen(e, s->bondprm_id, s->bondprm_len)
        || MD_setlen(e, s->angleprm_id, s->angleprm_len)
        || MD_setlen(e, s->dihedprm_id, s->dihedprm_len)
        || MD_setlen(e, s->imprprm_id, s->imprprm_len)
        || MD_setlen(e, s->nbfixprm_id, s->nbfixprm_len)) {
      return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
    }
    /* write contents from our buffers to engine buffers */
    if (MD_write(e, s->atomprm_id, s->atomprm, s->atomprm_len)
        || MD_write(e, s->bondprm_id, s->bondprm, s->bondprm_len)
        || MD_write(e, s->angleprm_id, s->angleprm, s->angleprm_len)
        || MD_write(e, s->dihedprm_id, s->dihedprm, s->dihedprm_len)
        || MD_write(e, s->imprprm_id, s->imprprm, s->imprprm_len)
        || MD_write(e, s->nbfixprm_id, s->nbfixprm, s->nbfixprm_len)) {
      return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
    }
  }

  /* set topology engine arrays */
  if (s->atom_attr.access & MD_SHARE) {
    /* all topology arrays have SHARE access */
    if (MD_share(e, s->atom_id, s->atom, s->atom_len, s->atom_len)
        || MD_share(e, s->bond_id, s->bond, s->bond_len, s->bond_len)
        || MD_share(e, s->angle_id, s->angle, s->angle_len, s->angle_len)
        || MD_share(e, s->dihed_id, s->dihed, s->dihed_len, s->dihed_len)
        || MD_share(e, s->impr_id, s->impr, s->impr_len, s->impr_len)
        || MD_share(e, s->excl_id, s->excl, s->excl_len, s->excl_len)) {
      return error(MSG_MDAPI, "from MD_share()", MD_errmsg(e));
    }
  }
  else {
    /* fall back on WRITE access - but it increases memory use */
    /* resize arrays to desired length */
    if (MD_setlen(e, s->atom_id, s->atom_len)
        || MD_setlen(e, s->bond_id, s->bond_len)
        || MD_setlen(e, s->angle_id, s->angle_len)
        || MD_setlen(e, s->dihed_id, s->dihed_len)
        || MD_setlen(e, s->impr_id, s->impr_len)
        || MD_setlen(e, s->excl_id, s->excl_len)) {
      return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
    }
    /* write contents from our buffers to engine buffers */
    if (MD_write(e, s->atom_id, s->atom, s->atom_len)
        || MD_write(e, s->bond_id, s->bond, s->bond_len)
        || MD_write(e, s->angle_id, s->angle, s->angle_len)
        || MD_write(e, s->dihed_id, s->dihed, s->dihed_len)
        || MD_write(e, s->impr_id, s->impr, s->impr_len)
        || MD_write(e, s->excl_id, s->excl, s->excl_len)) {
      return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
    }
  }

  /* set trajectory engine arrays */
  if (s->pos_attr.access & MD_SHARE) {
    /* both pos and vel have SHARE access */
    if (MD_share(e, s->pos_id, s->pos, s->atom_len, s->atom_len)
        || MD_share(e, s->vel_id, s->vel, s->atom_len, s->atom_len)) {
      return error(MSG_MDAPI, "from MD_share()", MD_errmsg(e));
    }
  }
  else {
    /* fall back on WRITE access - but it increases memory use */
    /* resize arrays to desired length */
    if (MD_setlen(e, s->pos_id, s->atom_len)
        || MD_setlen(e, s->vel_id, s->atom_len)) {
      return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
    }
    /* write contents from our buffers to engine buffers */
    if (MD_write(e, s->pos_id, s->pos, s->atom_len)
        || MD_write(e, s->vel_id, s->vel, s->atom_len)) {
      return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
    }
  }

  /* need to refresh engine array attributes */
  s->atomprm_attr = MD_attrib(e, s->atomprm_id);
  s->bondprm_attr = MD_attrib(e, s->bondprm_id);
  s->angleprm_attr = MD_attrib(e, s->angleprm_id);
  s->dihedprm_attr = MD_attrib(e, s->dihedprm_id);
  s->imprprm_attr = MD_attrib(e, s->imprprm_id);
  s->nbfixprm_attr = MD_attrib(e, s->nbfixprm_id);
  s->atom_attr = MD_attrib(e, s->atom_id);
  s->bond_attr = MD_attrib(e, s->bond_id);
  s->angle_attr = MD_attrib(e, s->angle_id);
  s->dihed_attr = MD_attrib(e, s->dihed_id);
  s->impr_attr = MD_attrib(e, s->impr_id);
  s->excl_attr = MD_attrib(e, s->excl_id);
  s->pos_attr = MD_attrib(e, s->pos_id);
  s->vel_attr = MD_attrib(e, s->vel_id);

  return 0;
}


/*
 * System destructor
 * free array allocations (from MDIO and directly)
 * destroy engine before calling
 */
void system_done(System *s)
{
  /* cleanup MDIO file reader objects */
  mdio_cleanupParam(&(s->prm));
  mdio_cleanupTopo(&(s->topo));
  mdio_cleanupPdbcoord(&(s->pdb));
  mdio_cleanupPdbcoord(&(s->pdbvel));
  mdio_cleanupBincoord(&(s->bin));
  mdio_cleanupBincoord(&(s->binvel));

  /* cleanup array allocations */
  free(s->posbuf);
  free(s->velbuf);
  free(s->resultbuf);
  if (s->is_vel_alloc) {
    free(s->vel);
  }

  /* clear memory */
  memset(s, 0, sizeof(System));
}
