/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * spcheck.c - simparam module implementation of simparam_check()
 *   got too long for simparam.c file
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mdsim/const.h"
#include "mdsim/simparam.h"
#include "mdsim/system.h"
#include "mdsim/error.h"
#include "debug/debug.h"


/* prototypes for internal routines */
static int output(SimParam *p);
static int restart(SimParam *p);
static int dcd(SimParam *p);
static int results(SimParam *p, System *sys, MD_Engine *e);
static int unsupported(SimParam *);


/*
 * call to validate simparams and setup additional
 * filenames and related data
 *
 * to be called after simparam_set() calls
 */
int simparam_check(SimParam *p, System *sys, MD_Engine *e)
{
  /* must provide at least one force field "parameters" (X-Plor) file */
  if (adt_getLengthList(&(p->parameters)) == 0) {
    return error("must specify \"parameters\" at least once");
  }

  /* must provide one "structure" (X-Plor PSF) file */
  if (p->structure == NULL) {
    return error("must specify \"structure\"");
  }

  /*
   * must provide "coordinates" (coordinate PDB) for initial positions
   *
   * may also provide "bincoordinates" (binary) to be used instead
   * of "coordinates" for initial positions - although then both will
   * be read
   */
  if (p->coordinates == NULL) {
    return error("must specify \"coordinates\"");
  }

  /* must provide one "outputname" for output of trajectories */
  if (p->outputname == NULL) {
    return error("must specify \"outputname\"");
  }

  /* must provide "restartname" if "restartfreq" is defined */
  if (p->restartfreq != 0 && p->restartname == NULL) {
    return error("must specify \"restartname\" with \"restartfreq\"");
  }

  /* must provide "restartfreq" if "restartname" is defined */
  if (p->restartname && p->restartfreq == 0) {
    return error("must specify \"restartfreq\" with \"restartname\"");
  }

  /* require "restartfreq" to be positive */
  if (p->restartfreq < 0) {
    return error("must have \"restartfreq\" > 0");
  }

  /* must provide "dcdfile" if "dcdfreq" is defined */
  if (p->dcdfreq != 0 && p->dcdfile == NULL) {
    return error("must specify \"dcdfile\" with \"dcdfreq\"");
  }

  /* must provide "dcdfreq" if "dcdfile" is defined */
  if (p->dcdfile && p->dcdfreq == 0) {
    return error("must specify \"dcdfreq\" with \"dcdfile\"");
  }

  /* require "dcdfreq" to be positive */
  if (p->dcdfreq < 0) {
    return error("must have \"dcdfreq\" > 0");
  }

  /* must provide "veldcdfile" if "veldcdfreq" is defined */
  if (p->veldcdfreq != 0 && p->veldcdfile == NULL) {
    return error("must specify \"veldcdfile\" with \"veldcdfreq\"");
  }

  /* must provide "veldcdfreq" if "veldcdfile" is defined */
  if (p->veldcdfile && p->veldcdfreq == 0) {
    return error("must specify \"veldcdfreq\" with \"veldcdfile\"");
  }

  /* require "veldcdfreq" to be positive */
  if (p->veldcdfreq < 0) {
    return error("must have \"veldcdfreq\" > 0");
  }

  /* require "outputenergies" to be positive */
  if (p->outputenergies < 0) {
    return error("must have \"outputenergies\" > 0");
  }

  /* require "numsteps" to be nonnegative */
  if (p->numsteps < 0) {
    return error("must have \"numsteps\" >= 0");
  }

  /*** set default values ***/

  if (p->binaryoutput == NULL
      && (p->binaryoutput = strdup("yes")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->binaryrestart == NULL
      && (p->binaryrestart = strdup("yes")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->binaryresults == NULL
      && (p->binaryresults = strdup("no")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->restartsave == NULL
      && (p->restartsave = strdup("no")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->dcdunitcell == NULL
      && (p->dcdunitcell = strdup("no")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->outputenergies == 0) {
    p->outputenergies = 1;
  }

  /*** check more requirements ***/

  /* require "binaryoutput" to be either "yes" or "no" */
  if (strcmp(p->binaryoutput, "yes") != 0
      && strcmp(p->binaryoutput, "no") != 0) {
    return error("must have \"binaryoutput\" be \"yes\" or \"no\"");
  }

  /* require "binaryrestart" to be either "yes" or "no" */
  if (strcmp(p->binaryrestart, "yes") != 0
      && strcmp(p->binaryrestart, "no") != 0) {
    return error("must have \"binaryrestart\" be \"yes\" or \"no\"");
  }

  /* require "binaryresults" to be either "yes" or "no" */
  if (strcmp(p->binaryresults, "yes") != 0
      && strcmp(p->binaryresults, "no") != 0) {
    return error("must have \"binaryresults\" be \"yes\" or \"no\"");
  }

  /* require "restartsave" to be either "yes" or "no" */
  if (strcmp(p->restartsave, "yes") != 0
      && strcmp(p->restartsave, "no") != 0) {
    return error("must have \"restartsave\" be \"yes\" or \"no\"");
  }

  /* require "dcdunitcell" to be either "yes" or "no" */
  if (strcmp(p->dcdunitcell, "yes") != 0
      && strcmp(p->dcdunitcell, "no") != 0) {
    return error("must have \"dcdunitcell\" be \"yes\" or \"no\"");
  }

  /*** set additional helper variables ***/

  p->is_binaryoutput = (strcmp(p->binaryoutput, "yes") == 0);
  p->is_binaryrestart = (strcmp(p->binaryrestart, "yes") == 0);
  p->is_binaryresults = (strcmp(p->binaryresults, "yes") == 0);
  p->is_restartsave = (strcmp(p->restartsave, "yes") == 0);
  p->is_dcdunitcell = (strcmp(p->dcdunitcell, "yes") == 0);

  /* setup output filenames */
  if (output(p)) return error(MSG, "output() failed");

  /* setup restart filenames */
  if (p->restartname && restart(p)) return error(MSG, "restart() failed");

  /* setup dcd filenames */
  if ((p->dcdfile || p->veldcdfile) && dcd(p)) {
    return error(MSG, "dcd() failed");
  }

  /* setup results output */
  if (results(p, sys, e)) return error(MSG, "results() failed");

  /*** check for unsupported features ***/
  if (unsupported(p)) return error(MSG, "unsupported() failed");

  return 0;
}


/*
 * setup output filenames
 *
 * return 0 for success, nonzero for failure
 */
int output(SimParam *p)
{
  int len;

  len = strlen(p->outputname) + 1;
  p->output_pos = (char *) malloc(len + strlen(FN_POS_EXT));
  p->output_vel = (char *) malloc(len + strlen(FN_VEL_EXT));
  len += strlen(FN_BAK_EXT);
  p->output_posbak = (char *) malloc(len + strlen(FN_POS_EXT));
  p->output_velbak = (char *) malloc(len + strlen(FN_VEL_EXT));
  if (p->output_pos == NULL || p->output_vel == NULL
      || p->output_posbak == NULL || p->output_velbak == NULL) {
    return error(MSG_NOMEM);
  }
  sprintf(p->output_pos, "%s%s", p->outputname, FN_POS_EXT);
  sprintf(p->output_vel, "%s%s", p->outputname, FN_VEL_EXT);
  sprintf(p->output_posbak, "%s%s", p->output_pos, FN_BAK_EXT);
  sprintf(p->output_velbak, "%s%s", p->output_vel, FN_BAK_EXT);
  return 0;
}


/*
 * setup restart filename buffers
 *
 * return 0 for success, nonzero for failure
 */
int restart(SimParam *p)
{
  char buf[32];  /* convert step number to string, is enough space */
  int slenlo;    /* string length of lowest step number */
  int slenhi;    /* string length of highest step number */
  int slenmax;   /* max string length for step numbers */
  int len;

  ASSERT(p->restartname != NULL);

  /* convert first and last time steps to strings, find max length */
  sprintf(buf, "%d", (int) p->firsttimestep);
  slenlo = strlen(buf);
  sprintf(buf, "%d", (int) (p->firsttimestep + p->numsteps));
  slenhi = strlen(buf);
  slenmax = (slenhi > slenlo ? slenhi : slenlo);
  p->restart_stepnum_width = slenmax;

  /* allocate enough space for each filename */
  len = strlen(p->restartname) + slenmax + 2;
  p->restart_pos = (char *) calloc(len + strlen(FN_POS_EXT), 1);
  p->restart_vel = (char *) calloc(len + strlen(FN_VEL_EXT), 1);
  len += strlen(FN_BAK_EXT);
  p->restart_posbak = (char *) calloc(len + strlen(FN_POS_EXT), 1);
  p->restart_velbak = (char *) calloc(len + strlen(FN_VEL_EXT), 1);
  if (p->restart_pos == NULL || p->restart_vel == NULL
      || p->restart_posbak == NULL || p->restart_velbak == NULL) {
    return error(MSG_NOMEM);
  }
  return 0;
}


int dcd(SimParam *p)
{
  int len;

  ASSERT(p->dcdfile != NULL || p->veldcdfile != NULL);

  /* create backup file for position DCD file */
  if (p->dcdfile != NULL) {
    len = strlen(p->dcdfile) + 1;
    p->dcdfile_bak = (char *) malloc(len + strlen(FN_BAK_EXT));
    if (p->dcdfile_bak == NULL) return error(MSG_NOMEM);
    sprintf(p->dcdfile_bak, "%s%s", p->dcdfile, FN_BAK_EXT);
  }

  /* create backup file for velocity DCD file */
  if (p->veldcdfile != NULL) {
    len = strlen(p->veldcdfile) + 1;
    p->veldcdfile_bak = (char *) malloc(len + strlen(FN_BAK_EXT));
    if (p->veldcdfile_bak == NULL) return error(MSG_NOMEM);
    sprintf(p->veldcdfile_bak, "%s%s", p->veldcdfile, FN_BAK_EXT);
  }
  return 0;
}


/*
 * setup results output
 *
 * includes header line of labels and byte offsets into
 * engine "result" data
 *
 * return 0 for success, nonzero for failure
 */
int results(SimParam *p, System *sys, MD_Engine *e)
{
  char buf[32];  /* convert step number to string, is enough space */
  int slenlo;    /* string length of lowest step number */
  int slenhi;    /* string length of highest step number */
  int slenmax;   /* max string length for step numbers */
  const int slenmin = 6;  /* length of "# step" on header line */
#ifdef DEBUG_SUPPORT
  int32 nres = 0;  /* count number of "results_info" list elements */
#endif
  int32 nlab = 0;  /* count number of "results_label" list elements */
  MD_Member m;     /* for type and array length of each "result" member */
  const char *cs;  /* to process each string in results array */
  int32 cslen;     /* length of cs string */
  ResultsInfo res; /* keep track of byte offset, type, and len for member */
  int i, j, slen;
  int results_len;
  char *s;

  /* setup default list results */
  if (adt_getLengthList(&(p->results)) == 0) {

    /*** for now, just output "energy" by default ***/
    if ((s = strdup("energy")) == NULL
        || adt_appendList(&(p->results), &s)) {
      return error(MSG_NOMEM);
    }
  }

  /*
   * how often do results get printed?
   * default to "outputenergies" if unspecified
   */
  if (p->resultsfreq == 0) {
    p->resultsfreq = p->outputenergies;
  }
  else if (p->resultsfreq < 0) {
    return error("must have \"resultsfreq\" > 0");
  }

  /* how often do headers get printed? */
  if (p->resultsheaderfreq == 0) {
    p->resultsheaderfreq = RESULTS_HEADER_FREQ;
  }
  else if (p->resultsheaderfreq < 0) {
    return error("must have \"resultsheaderfreq\" > 0");
  }

  /* create backup file name if results are saved to log file */
  if (p->resultsname) {
    p->results_bak = (char *) malloc(strlen(p->resultsname)
        + strlen(FN_BAK_EXT) + 1);
    if (p->results_bak == NULL) return error(MSG_NOMEM);
    sprintf(p->results_bak, "%s%s", p->resultsname, FN_BAK_EXT);
  }

  /* compute offsets, expanding arrays and vectors as scalars */
  results_len = adt_getLengthList(&(p->results));
  for (i = 0;  i < results_len;  i++) {

    /* obtain each string in results array */
    cs = *((const char **) adt_indexList(&(p->results), i));
    ASSERT(cs != NULL);
    cslen = strlen(cs);

    /* lookup cs as member of engine "result", compute its byte offset */
    res.offset =
      ((char *) MD_type_member(e, sys->result_attr.type, NULL, cs, &m))
      - ((char *) NULL);
    res.type = m.type;
    res.len = m.len;

    /* make sure cs is the name of a member */
    if (MD_errnum(e)) {
      return error("\"%s\" is not valid for \"results\"", cs);
    }

    /* make sure engine "result" member has primary type */
    if (m.type != MD_CHAR && m.type != MD_INT32
        && m.type != MD_FLOAT && m.type != MD_DOUBLE
        && m.type != MD_FVEC && m.type != MD_DVEC
        && m.type != MD_NAME && m.type != MD_STRING
        && m.type != MD_MESSAGE) {
      char msg[100];
      sprintf(msg, "member \"%.60s\" does not have primary type", cs);
      return error(MSG_ENGINE, msg);
    }

    /* see if it is already a scalar (non-array or string) */
    if (m.len == 1 || m.type == MD_CHAR) {

      /* store offset */
      if (adt_appendList(&(p->results_info), &res)) {
        return error(MSG_NOMEM);
      }
      ASSERT( ++nres );

      /* expand 3D-vector "scalar" into separate _x, _y, _z labels */
      if (m.type == MD_DVEC || m.type == MD_FVEC) {
        char *s[3];

        /* store separate x,y,z labels */
        if ((s[0] = (char *) malloc(cslen + 3)) == NULL
            || (s[1] = (char *) malloc(cslen + 3)) == NULL
            || (s[2] = (char *) malloc(cslen + 3)) == NULL
            || adt_appendList(&(p->results_label), &(s[0]))
            || adt_appendList(&(p->results_label), &(s[1]))
            || adt_appendList(&(p->results_label), &(s[2]))) {
          return error(MSG_NOMEM);
        }
        nlab += 3;
        sprintf(s[0], "%s_x", cs);
        sprintf(s[1], "%s_y", cs);
        sprintf(s[2], "%s_z", cs);
      }

      /* otherwise, just copy label as is */
      else {
        if ((s = strdup(cs)) == NULL
            || adt_appendList(&(p->results_label), &s)) {
          return error(MSG_NOMEM);
        }
        ++nlab;
      }
      continue;
    }

    /* otherwise store ResultsInfo for each individual array element */
    res.len = 1;

    /* loop over array, expand labels as _1, _2, _3 (1-based indexing) */
    for (j = 0;  j < m.len;  j++) {
      slen = cslen;

      /* find string length of index */
      sprintf(buf, "%d", j+1);
      slen += strlen(buf) + 1;

      /* store each offset */
      if (adt_appendList(&(p->results_info), &res)) {
        return error(MSG_NOMEM);
      }
      ASSERT( ++nres );

      /* increment offset for next ResultsInfo element */
      res.offset += MD_SIZEOF(m.type);

      /*
       * expand 3D-vector "scalar" into separate _x, _y, _z labels
       * (e.g. for mth array element labels are _m_x, _m_y, _m_z)
       */
      if (m.type == MD_DVEC || m.type == MD_FVEC) {
        char *s[3];

        /* store separate x,y,z labels */
        if ((s[0] = (char *) malloc(cslen + 3)) == NULL
            || (s[1] = (char *) malloc(cslen + 3)) == NULL
            || (s[2] = (char *) malloc(cslen + 3)) == NULL
            || adt_appendList(&(p->results_label), &(s[0]))
            || adt_appendList(&(p->results_label), &(s[1]))
            || adt_appendList(&(p->results_label), &(s[2]))) {
          return error(MSG_NOMEM);
        }
        nlab += 3;
        sprintf(s[0], "%s_%s_x", cs, buf);
        sprintf(s[1], "%s_%s_y", cs, buf);
        sprintf(s[2], "%s_%s_z", cs, buf);
      }

      /* otherwise, just append index to label */
      else {
        if ((s = (char *) malloc(slen + 1)) == NULL
            || adt_appendList(&(p->results_label), &s)) {
          return error(MSG_NOMEM);
        }
        ++nlab;
        sprintf(s, "%s_%s", cs, buf);
      }
    }

  } /* end loop over "results" array */

  /* set default width if too small */
  if (p->resultswidth <= 1) {
    p->resultswidth = RESULTS_WIDTH;
  }

  /* convert first and last time steps to strings, find max length */
  sprintf(buf, "%d", (int) p->firsttimestep);
  slenlo = strlen(buf);
  sprintf(buf, "%d", (int) (p->firsttimestep + p->numsteps));
  slenhi = strlen(buf);
  slenmax = (slenhi > slenlo ? slenhi : slenlo);
  if (slenmax < slenmin) slenmax = slenmin;
  p->results_stepnum_width = slenmax;

  /* header contains entries from "results_label" */
  p->results_header =
    (char *) malloc(nlab * p->resultswidth + slenmax + 1);
  if (p->results_header == NULL) return error(MSG_NOMEM);

  /* create header */
  s = p->results_header;
  s += sprintf(p->results_header, "#%*s", slenmax - 1, "step");
  for (i = 0;  i < nlab;  i++) {
    cs = *((const char **) adt_indexList(&(p->results_label), i));
    ASSERT(cs != NULL);
    s += sprintf(s, "%*.*s", p->resultswidth, p->resultswidth - 1, cs);
    if (strlen(cs) >= p->resultswidth) {
      warn("results label \"%s\" will be truncated to \"%.*s\"",
          cs, p->resultswidth - 1, cs);
    }
  }
  ASSERT(adt_getLengthList(&(p->results_label)) == nlab);
  ASSERT(adt_getLengthList(&(p->results_info)) == nres);

#if 0
  /*** FOR DEBUGGING ***/
  printf("space allocated:  %d\n", nlab * p->resultswidth + slenmax + 1);
  printf("length of header: %d\n", strlen(p->results_header));
  printf("%s\n", p->results_header);
  printf("nlab = %d   p->results_label.len = %d\n",
      (int) nlab, (int) (p->results_label.len));
  printf("nres = %d   p->results_info.len = %d\n",
      (int) nres, (int) (p->results_info.len));
  printf("offset:");
  for (i = 0;  i < nres;  i++) {
    printf("  %d", (int) *((int32*) ADT_array_elem(&(p->results_info),i)));
  }
  printf("\n");
  printf("size of engine result = %d\n", MD_SIZEOF(sys->result_attr.type));
  exit(0);  /*** FOR DEBUGGING ***/
#endif
  return 0;
}


/*
 * check on features that are not yet supported
 *
 * return 0 for success, nonzero for failure
 */
int unsupported(SimParam *p)
{
  if (p->is_binaryresults) {
    return error("binary results file logging is not yet supported");
  }
  return 0;
}
