/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * simparam.h - type definition and prototypes for simparam module;
 *   purpose is to parse and store front end simulation parameters
 */

#ifndef SIMPARAM_H
#define SIMPARAM_H

#include "mdapi/mdfront.h"
#include "adt/list.h"
#include "adt/table.h"

#ifdef __cplusplus
extern "C" {
#endif

  /* need a forward definition of System type */
  struct System_tag;

  /* data needed for accessing members of engine "result" */
  typedef struct ResultsInfo_tag {
    int32 offset;  /* byte offset into "result" */
    int32 type;    /* number for primary type */
    int32 len;     /* len > 1 only for strings (type = MD_CHAR) */
  } ResultsInfo;

  /*
   * SimParam data structure
   *
   * stores simulation parameters recognized by front end
   */
  typedef struct SimParam_tag {
  /*** internal ***/

    /* table for storing member info */
    adt_Table member;

  /*** external ***/

    /* default directory for file I/O */
    char *cwd;

    /* simparams for input */
    adt_List parameters;
    char *structure;
    char *coordinates;
    char *bincoordinates;
    char *velocities;
    char *binvelocities;

    /* simparams for output */
    char *outputname;
    char *binaryoutput;
    char *binaryrestart;
    char *binaryresults;
    char *restartname;
    char *restartsave;
    int32 restartfreq;
    char *dcdfile;
    int32 dcdfreq;
    char *veldcdfile;
    int32 veldcdfreq;
    char *dcdunitcell;
    int32 outputenergies;
    adt_List results;
    char *resultsname;
    int32 resultsfreq;
    int32 resultswidth;
    int32 resultsheaderfreq;

    /* Boolean flags */
    int32 is_binaryoutput;
    int32 is_binaryrestart;
    int32 is_binaryresults;
    int32 is_restartsave;
    int32 is_dcdunitcell;

    /* filename support for final system output */
    char *output_pos;
    char *output_vel;
    char *output_posbak;
    char *output_velbak;

    /* filename support for restart files */
    int32 restart_stepnum_width;
    char *restart_pos;
    char *restart_vel;
    char *restart_posbak;
    char *restart_velbak;

    /* filename support for dcd files */
    char *dcdfile_bak;
    char *veldcdfile_bak;

    /* support for results output */
    adt_List results_info;     /* list of ResultInfo */
    adt_List results_label;
    char *results_header;
    char *results_bak;
    int32 results_stepnum_width;

    /* simparams for system initialization */
    char *engine;
/*    double temperature; */
    int32 firsttimestep;
    int32 numsteps;
  } SimParam;

  /*** prototypes ***/
  int simparam_init(SimParam *p);
  /*
   * constructor, initialize SimParam structure:
   *   clear memory, allocate member table, init array(s)
   *
   * returns 0 for success (or error code for failure)
   */

  int simparam_setup(SimParam *p, const char *cwd, const char *engine);
  /*
   * prepare for simparam_set() calls
   *
   * sets "cwd" (current working directory) and
   * "engine" (name of engine) simparams
   *
   * returns 0 for success (or error code for failure)
   */

  int simparam_set(SimParam *p, const char *keywd, const char *val);
  /*
   * set SimParam member identified by keyword (lowercase string of
   * member name) to value string (parsed and converted to correct type);
   * the (keywd,val)-pair are most likely read from config file
   *
   * note: simparam_setup() must be called first
   *
   * returns 0 for success (or error code for failure)
   * errors: NOTFOUND, REINIT, BADVAL
   */

  int simparam_check(SimParam *p, struct System_tag *s, MD_Engine *e);
  /*
   * check for initialization of mandatory parameters, provide default
   * values when appropriate, and check ranges on parameters
   * (need access to engine for checking certain parameters)
   *
   * returns 0 for success (or error code for failure)
   */

  int simparam_restart(SimParam *p, int32 stepnum);
  /*
   * generate new restart filenames using step number
   *
   * returns 0 for success (or error code for failure)
   */

  int simparam_restartbak(SimParam *p);
  /*
   * generate new backup restart filenames from current restart filenames
   *
   * returns 0 for success (or error code for failure)
   */

  void simparam_done(SimParam *p);
  /*
   * destructor, cleanup SimParam structure:
   *   free strings, free array strings, cleanup array(s), cleanup table
   */

#ifdef DEBUG_SUPPORT
  int simparam_debug(SimParam *p);
#endif

#ifdef __cplusplus
}
#endif

#endif /* SIMPARAM_H */
