/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * mdsim.c
 */

#include <stdio.h>
#include "mdsim/sim.h"
#include "mdsim/error.h"
#include "debug/debug.h"


#ifdef DEBUG_SUPPORT
static int print_config(mdio_ConfigData *c, int32 clen)
{
  int32 k;
  printf("===== listing of config file contents =====\n");
  for (k = 0;  k < clen;  k++) {
    printf("(%s,%d): %s = \"%s\"\n",
        c[k].filename, c[k].linenum, c[k].keyword, c[k].value);
  }
  return 0;
}
#endif


int main(int argc, char *argv[])
{
  Sim sim;

  /* print title */
  printf("#\n# MDSim - molecular dynamics simulation front end\n#\n");

  /* check command line arguments */
  if (argc < 2) {
    return error("usage: %s config_file", argv[0]);
  }
  /* skip command line args to start of config file list */
  argc--;
  argv++;

  /* initialize sim structure */
  if (sim_init(&sim)) {
    return error(MSG, "sim_init() failed");
  }

  /* read config files, setup default cwd and engine */
  if (sim_input_configfile(&sim, argc, argv)) {
    return error(MSG, "sim_input_configfile() failed");
  }

#ifdef DEBUG_SUPPORT
  if (print_config(sim.info, sim.info_len)) {
    return error(MSG, "print_config() failed");
  }
#endif

  /* start engine and check compatibility */
  if (sim_start_engine(&sim)) {
    return error(MSG, "sim_start_engine() failed");
  }

  /* parse config file parameters */
  if (sim_parse_configfile(&sim)) {
    return error(MSG, "sim_parse_configfile() failed");
  }

  /* input system data and setup engine data arrays */
  if (sim_input_sysdata(&sim)) {
    return error(MSG, "sim_input_sysdata() failed");
  }

  /* setup callbacks, if permitted by engine data array access */
  if (sim_setup_callbacks(&sim)) {
    return error(MSG, "sim_setup_callbacks() failed");
  }

  /* run simulation, either with callbacks or start and stop */
  if (sim_run(&sim)) {
    return error(MSG, "sim_run() failed");
  }

  /* save final trajectory (if desired) and other post-processing */
  if (sim_output_sysdata(&sim)) {
    return error(MSG, "sim_output_sysdata() failed");
  }

  /* cleanup sim structure */
  sim_done(&sim);

  printf("#\n# Success!\n#\n");

  return 0;
}
