/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * demo_simparam.c - demonstrate parsing and storing simparams
 */

#include <stdio.h>
#include <stdlib.h>
#include "debug/debug.h"
#include "mdio/mdio.h"
#include "mdsim/sim.h"
#include "mdsim/error.h"

int main(int32 argc, char *argv[])
{
  mdio_Config *cfg;
  mdio_ConfigData *info;
  int32 k, nelems;
  SimParam sp;

  /* check command line args */
  if (argc == 1) {
    fprintf(stderr, "usage: %s CONFIG1 [CONFIG2 ...]\n", argv[0]);
    return 0;
  }

  /* read config files */
  if ((cfg = mdio_createConfig()) == NULL) {
    fprintf(stderr, "mdio_createConfig() failed\n");
    exit(1);
  }
  for (k = 1;  k < argc;  k++) {
    if (mdio_readConfig(cfg, argv[k])) {
      fprintf(stderr, "mdio_readConfig() failed\n");
    }
  }
  info = mdio_getDataConfig(cfg, &nelems);

  /* set simparams */
  if (simparam_init(&sp)) {
    fprintf(stderr, "simparam_init() failed\n");
    exit(1);
  }
  if (simparam_setup(&sp, ".", "myengine")) {
    fprintf(stderr, "simparam_setup() failed\n");
    exit(1);
  }
  for (k = 0;  k < nelems;  k++) {
    switch (simparam_set(&sp, info[k].keyword, info[k].value)) {
      case 0:
        printf("\'%s\' successfully assigned\n", info[k].keyword);
        break;
      case NOTFOUND:
        printf("ERROR: \'%s\' is not a recognized keyword\n", info[k].keyword);
        break;
      case REINIT:
        printf("ERROR: attempt to re-initialize \'%s\'\n", info[k].keyword);
        break;
      case BADVAL:
        printf("ERROR: attempt to assign illegal value to \'%s\'\n",
            info[k].keyword);
        break;
      case (REINIT | BADVAL):
        printf("ERROR: attempt to re-initialize \'%s\'\n", info[k].keyword);
        printf("ERROR: attempt to assign illegal value to \'%s\'\n",
            info[k].keyword);
        break;
      case FAIL:
        fprintf(stderr, "ERROR: no memory available\n");
        exit(1);
      default:
        BUG("invalid value returned from simparam_set()");
    }
  }

  /* done with config and its data, free memory */
  mdio_destroyConfig(cfg);

  /* look at stored simparam values */
  printf("Assignments:\n");
  printf("  cwd = %s\n", sp.cwd);
  for (k = 0;  k < (int32) sp.parameters.len;  k++) {
    printf("  parameters[%d]  = %s\n", k,
        *((char **)adt_indexList(&(sp.parameters), k)));
  }
  printf("  structure      = %s\n", sp.structure);
  printf("  coordinates    = %s\n", sp.coordinates);
  printf("  bincoordinates = %s\n", sp.bincoordinates);
  printf("  velocities     = %s\n", sp.velocities);
  printf("  binvelocities  = %s\n", sp.binvelocities);
  printf("  outputname     = %s\n", sp.outputname);
  printf("  binaryoutput   = %s\n", sp.binaryoutput);
  printf("  binaryrestart  = %s\n", sp.binaryrestart);
  printf("  restartname    = %s\n", sp.restartname);
  printf("  restartsave    = %s\n", sp.restartsave);
  printf("  restartfreq    = %d\n", sp.restartfreq);
  printf("  outputenergies = %d\n", sp.outputenergies);
  printf("  engine         = %s\n", sp.engine);
  printf("  firsttimestep  = %d\n", sp.firsttimestep);
  printf("  numsteps       = %d\n", sp.numsteps);

  /* done with simparams, free memory */
  simparam_done(&sp);
  return 0;
}
