/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * callback.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "mdsim/const.h"
#include "mdsim/callback.h"
#include "mdsim/error.h"
#include "debug/debug.h"


/* internal function prototypes */
static int32 get_dcdprms(Callback *c, MD_Engine *e);
static int32 results_text(FILE *, Callback *c, void *result,
    int32 stepnum, int32 is_header);
static int32 results_binary(FILE *, Callback *c, void *result,
    int32 stepnum, int32 is_header);


int callback_init(Callback *c)
{
  memset(c, 0, sizeof(Callback));
  return 0;
}


int callback_setup(Callback *c, MD_Engine *e, SimParam *p, System *s)
{
  c->sp = p;
  c->sys = s;

  if (p->restartname != NULL
      && (s->pos_attr.access & MD_CBREAD)
      && (s->vel_attr.access & MD_CBREAD)) {

    /* setup restart callback */
    if (callback_setup_restart(c)) {
      return error(MSG, "callback_setup_restart() failed");
    }
    c->is_restart = TRUE;

    /* establish restart file callback */
    if (MD_callback(e, callback_restart, c, c->restart, 2, p->restartfreq)) {
      return error(MSG_MDAPI, "from MD_callback()", MD_errmsg(e));
    }
  }

  /* setup data needed for either position or velocity DCD file */
  if (p->dcdfile != NULL || p->veldcdfile != NULL) {
    if (get_dcdprms(c, e)) return FAIL;
  }

  if (p->dcdfile != NULL
      && (s->pos_attr.access & MD_CBREAD)) {
    const int32 n = (c->with_result ? 2 : 1);

    /* setup posdcd callback */
    if (callback_setup_posdcd(c)) {
      return error(MSG, "callback_setup_posdcd() failed");
    }
    c->is_posdcd = TRUE;

    /* establish posdcd callback */
    if (MD_callback(e, callback_posdcd, c, c->cbposdcd, n, p->dcdfreq)) {
      return error(MSG_MDAPI, "from MD_callback()", MD_errmsg(e));
    }
  }

  if (p->veldcdfile != NULL
      && (s->vel_attr.access & MD_CBREAD)) {
    const int32 n = (c->with_result ? 2 : 1);

    /* setup posdcd callback */
    if (callback_setup_veldcd(c)) {
      return error(MSG, "callback_setup_veldcd() failed");
    }
    c->is_veldcd = TRUE;

    /* establish posdcd callback */
    if (MD_callback(e, callback_veldcd, c, c->cbveldcd, n, p->veldcdfreq)) {
      return error(MSG_MDAPI, "from MD_callback()", MD_errmsg(e));
    }
  }

  if (s->result_attr.access & MD_CBREAD) {

    /* setup results callback */
    if (callback_setup_results(c)) {
      return error(MSG, "callback_setup_results() failed");
    }
    c->is_results = TRUE;

    /* establish results callback */
    if (MD_callback(e, callback_results, c, c->results, 1, p->resultsfreq)) {
      return error(MSG_MDAPI, "from MD_callback()", MD_errmsg(e));
    }
  }
  return 0;
}


int callback_setup_restart(Callback *c)
{
  SimParam *p = c->sp;
  System *s = c->sys;

  /* set flags for writing */
  ASSERT(s->pos_attr.type == MD_DVEC);
  ASSERT(s->vel_attr.type == MD_DVEC);
  c->is_binary = p->is_binaryrestart;

  /*
   * coord pdb restart files require conversion of velocity units
   * then must use separate system buffer
   */
  if ( ! c->is_binary) {
    /* point callback "vel" pointer to this buffer */
    c->vel = s->velbuf;
    /* init PDB file object */
    if (mdio_initializePdbcoord(&(c->pdb))) {
      return error(MSG, "mdio_initializePdbcoord() failed");
    }
    mdio_setErrorHandlerFile((mdio_File *) &(c->pdb), mdio_errhandler);
  }
  else {
    /* otherwise, no extra buffer space needed */
    c->vel = NULL;
    /* init binary coordinate file object */
    if (mdio_initializeBincoord(&(c->bin))) {
      return error(MSG, "mdio_initializeBincoord() failed");
    }
    mdio_setErrorHandlerFile((mdio_File *) &(c->bin), mdio_errhandler);
  }

  c->restart[0].idnum = s->pos_id;
  c->restart[0].access = MD_CBREAD;
  c->restart[0].nelems = s->atom_len;
  c->restart[0].first = 0;
  c->restart[1].idnum = s->vel_id;
  c->restart[1].access = MD_CBREAD;
  c->restart[1].nelems = s->atom_len;
  c->restart[1].first = 0;

  return 0;
}


int callback_setup_posdcd(Callback *c)
{
  SimParam *p = c->sp;
  System *s = c->sys;
  mdio_DcdHeader header;

  /* do not overwrite existing files */
  ASSERT(p->dcdfile != NULL);
  ASSERT(p->dcdfile_bak != NULL);
  if (rename(p->dcdfile, p->dcdfile_bak) && errno != ENOENT) {
    return warn("unable to rename \"%s\" to \"%s\"",
        p->dcdfile, p->dcdfile_bak);
  }

  /* init DCD object */
  if (mdio_initializeDcd(&(c->posdcd))) {
    return error(MSG, "mdio_initializeDcd() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(c->posdcd), mdio_errhandler);

  /* begin writing DCD file */
  if (mdio_writeBeginDcd(&(c->posdcd), p->dcdfile)) {
    return error(MSG_MDIO,
        mdio_getErrorMessageFile((mdio_File *) &(c->posdcd)));
  }

  /* setup DCD header */
  header.timestep = c->timestep;
  header.natoms = s->atom_len;
  header.firststep = p->firsttimestep;
  header.framestepcnt = p->dcdfreq;
  header.cellstatus = 0;
  if (p->is_dcdunitcell) {
    header.cellstatus |= MDIO_DCD_UNITCELL;
    if ( ! c->with_result) header.cellstatus |= MDIO_DCD_FIXEDCELL;
  }
  if (mdio_writeHeaderDcd(&(c->posdcd), &header)) {
    return error(MSG_MDIO,
        mdio_getErrorMessageFile((mdio_File *) &(c->posdcd)));
  }

  c->cbposdcd[0].idnum = s->pos_id;
  c->cbposdcd[0].access = MD_CBREAD;
  c->cbposdcd[0].nelems = s->atom_len;
  c->cbposdcd[0].first = 0;
  if (c->with_result) {
    /* also acquire sim cell info */
    c->cbposdcd[1].idnum = s->result_id;
    c->cbposdcd[1].access = MD_CBREAD;
    c->cbposdcd[1].nelems = 1;
    c->cbposdcd[1].first = 0;
  }

  return 0;
}


int callback_setup_veldcd(Callback *c)
{
  SimParam *p = c->sp;
  System *s = c->sys;
  mdio_DcdHeader header;

  /* do not overwrite existing files */
  ASSERT(p->veldcdfile != NULL);
  ASSERT(p->veldcdfile_bak != NULL);
  if (rename(p->veldcdfile, p->veldcdfile_bak) && errno != ENOENT) {
    return warn("unable to rename \"%s\" to \"%s\"",
        p->veldcdfile, p->veldcdfile_bak);
  }

  /* init DCD object */
  if (mdio_initializeDcd(&(c->veldcd))) {
    return error(MSG, "mdio_initializeDcd() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(c->posdcd), mdio_errhandler);

  /* begin writing DCD file */
  if (mdio_writeBeginDcd(&(c->veldcd), p->veldcdfile)) {
    return error(MSG_MDIO,
        mdio_getErrorMessageFile((mdio_File *) &(c->veldcd)));
  }

  /* setup DCD header */
  header.timestep = c->timestep;
  header.natoms = s->atom_len;
  header.firststep = p->firsttimestep;
  header.framestepcnt = p->veldcdfreq;
  header.cellstatus = 0;
  if (p->is_dcdunitcell) {
    header.cellstatus |= MDIO_DCD_UNITCELL;
    if ( ! c->with_result) header.cellstatus |= MDIO_DCD_FIXEDCELL;
  }
  if (mdio_writeHeaderDcd(&(c->veldcd), &header)) {
    return error(MSG_MDIO,
        mdio_getErrorMessageFile((mdio_File *) &(c->veldcd)));
  }

  c->cbveldcd[0].idnum = s->vel_id;
  c->cbveldcd[0].access = MD_CBREAD;
  c->cbveldcd[0].nelems = s->atom_len;
  c->cbveldcd[0].first = 0;
  if (c->with_result) {
    /* also acquire sim cell info */
    c->cbveldcd[1].idnum = s->result_id;
    c->cbveldcd[1].access = MD_CBREAD;
    c->cbveldcd[1].nelems = 1;
    c->cbveldcd[1].first = 0;
  }

  return 0;
}


int callback_setup_results(Callback *c)
{
  SimParam *p = c->sp;
  System *s = c->sys;

  c->results_info = (ResultsInfo *) adt_getDataList(&(p->results_info));
  c->results_info_len = (int32) adt_getLengthList(&(p->results_info));

  /* start header counter at max so that header is printed initially */
  c->cnt = p->resultsheaderfreq;

  if (p->resultsname) {
    /* also log results to a file */
    ASSERT(p->results_bak != NULL);

    /* try to rename if file already exists */
    if (rename(p->resultsname, p->results_bak) && errno != ENOENT) {
      return error("unable to rename \"%s\" to \"%s\"",
          p->resultsname, p->results_bak);
    }

    /* open results log file */
    if ((c->logfile = fopen(p->resultsname, "w")) == NULL) {
      return error("unable to open results log file \"%s\"", p->resultsname);
    }

    /* print intro line to log file */
    if (fprintf(c->logfile, "# results file created by mdsim\n") < 0) {
      return error("unable to write to results log file \"%s\"",
          p->resultsname);
    }
  }

  c->results[0].idnum = s->result_id;
  c->results[0].access = MD_CBREAD;
  c->results[0].nelems = 1;
  c->results[0].first = 0;

  return 0;
}


void callback_done(Callback *c)
{
  SimParam *p = c->sp;

  if (c->is_restart) {
    if ( ! (p->is_restartsave) ) {
      /* not saving restart files, so remove last backups */
      if (p->restart_posbak[0] != '\0') {
        ASSERT(p->restart_velbak[0] != '\0');
        if (remove(p->restart_posbak)) {
          warn("unable to remove \"%s\"", p->restart_posbak);
        }
        if (remove(p->restart_velbak)) { 
          warn("unable to remove \"%s\"", p->restart_velbak);
        }
      }
    }
    /* done with MDIO data structures */
    if (c->is_binary) {
      mdio_cleanupBincoord(&(c->bin));
    }
    else {
      mdio_cleanupPdbcoord(&(c->pdb));
    }
  }
  if (c->is_posdcd) {
    if (mdio_writeEndDcd(&(c->posdcd))) {
      warn(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) &(c->posdcd)));
    }
    mdio_cleanupDcd(&(c->posdcd));
  }
  if (c->is_veldcd) {
    if (mdio_writeEndDcd(&(c->veldcd))) {
      warn(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) &(c->veldcd)));
    }
    mdio_cleanupDcd(&(c->veldcd));
  }
  if (c->is_results && p->resultsname) {
    if (fclose(c->logfile)) {
      warn("unable to close \"%s\"", p->resultsname);
    }
  }
}


/* callback to write restart files */
int32 callback_restart(void *info, MD_Cbdata *cb, int32 cblen, int32 stepnum)
{
  Callback *c = (Callback *) info;
  SimParam *p = c->sp;
#ifdef DEBUG_SUPPORT
  System *s = c->sys;
#endif
  mdio_Bincoord *bin = &(c->bin);
  mdio_Pdbcoord *pdb = &(c->pdb);
  MD_Dvec *pos = cb[0].data;
  MD_Dvec *vel = cb[1].data;
  int32 len = cb[0].attrib.len;
#ifdef DEBUG_SUPPORT
  int32 type = cb[0].attrib.type;
#endif

  /* debugging assertions */
  ASSERT(c != NULL);
  ASSERT(cb != NULL);
  ASSERT(cblen == 2);
  ASSERT(len == cb[1].attrib.len);
  ASSERT(type == cb[1].attrib.type);
  ASSERT(type == MD_DVEC);
  ASSERT(len > 0);
  ASSERT(len == s->atom_len);
  ASSERT(pos != NULL);
  ASSERT(vel != NULL);

  /* message to user */
  printf("saving restart files at step number %d\n", stepnum);

  /* saving restart files */
  if (p->is_restartsave) {

    /* generate updated restart and backup filenames */
    if (simparam_restart(p, stepnum)) {
      return warn("unable to update restart filename on step %d", stepnum);
    }
    else if (simparam_restartbak(p)) {
      return warn("unable to update backup restart filename on step %d",
          stepnum);
    }

    /*
     * preserve any previously existing file by this name by
     * renaming it to the backup filename
     * (only fails if both files exist but rename isn't possible)
     */
    if (rename(p->restart_pos, p->restart_posbak) && errno != ENOENT) {
      return warn("unable to rename \"%s\" to \"%s\"",
          p->restart_pos, p->restart_posbak);
    }
    else if (rename(p->restart_vel, p->restart_velbak) && errno != ENOENT) {
      return warn("unable to rename \"%s\" to \"%s\"",
          p->restart_vel, p->restart_velbak);
    }
  }

  /* otherwise, not saving restart files */
  else {

    /*
     * here we let backup file name keep track of two callbacks ago
     * remove if it exists (checking for empty string - this branch
     * won't be true until third time we execute callback)
     */
    if (p->restart_posbak[0] != '\0') {
      ASSERT(p->restart_velbak[0] != '\0');
      if (remove(p->restart_posbak)) {
        return warn("unable to remove \"%s\"", p->restart_posbak);
      }
      else if (remove(p->restart_velbak)) {
        return warn("unable to remove \"%s\"", p->restart_velbak);
      }
    }

    /*
     * restart file name refers to previous callback
     * rename if it exists to updated backup name
     * (checking for empty string - this branch won't
     * be true until second time we execute callback)
     */
    if (p->restart_pos[0] != '\0') {
      ASSERT(p->restart_vel[0] != '\0');
      if (simparam_restartbak(p)) {
        return warn("unable to update backup restart filename on step %d",
            stepnum);
      }
      if (rename(p->restart_pos, p->restart_posbak)) {
        return warn("unable to rename \"%s\" to \"%s\"",
            p->restart_pos, p->restart_posbak);
      }
      else if (rename(p->restart_vel, p->restart_velbak)) {
        return warn("unable to rename \"%s\" to \"%s\"",
            p->restart_vel, p->restart_velbak);
      }
    }

    /* update restart file name to current step number before writing */
    if (simparam_restart(p, stepnum)) {
      return warn("unable to update restart filename on step %d", stepnum);
    }
  }

  /* convert pdb velocities in separate buffer */
  if (c->vel) {
    int32 k;
    /* convert from A/fs to A/ps */
    for (k = 0;  k < len;  k++) {
      c->vel[k].x = MD_ANGSTROM_PS * vel[k].x;
      c->vel[k].y = MD_ANGSTROM_PS * vel[k].y;
      c->vel[k].z = MD_ANGSTROM_PS * vel[k].z;
    }
    /* point "vel" to this buffer */
    vel = c->vel;
  }

  /* write restart files */
  if (c->is_binary) {
    if (mdio_setBincoord(bin, pos, len)  /* set position coordinates */
        || mdio_writeBincoord(bin, p->restart_pos)) {
      return warn(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) bin));
    }
    else if (mdio_setBincoord(bin, vel, len)  /* set velocity coordinates */
        || mdio_writeBincoord(bin, p->restart_vel)) {
      return warn(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) bin));
    }
  }
  else {
    if (mdio_setPdbcoord(pdb, pos, len)  /* set position coordinates */
        || mdio_writePdbcoord(pdb, p->restart_pos)) {
      return warn(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) pdb));
    }
    else if (mdio_setPdbcoord(pdb, vel, len)  /* set velocity coordinates */
        || mdio_writePdbcoord(pdb, p->restart_vel)) {
      return warn(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) pdb));
    }
  }
  return 0;
}


/* callback to write frame for DCD file */
int32 callback_posdcd(void *info, MD_Cbdata *cb, int32 cblen, int32 stepnum)
{
  Callback *c = (Callback *) info;
  SimParam *p = c->sp;
  MD_Dvec *pos = cb[0].data;
  mdio_Dcd *dcd = &(c->posdcd);
  mdio_DcdCell *cell = (p->is_dcdunitcell ? &(c->cell) : NULL);

  /* initial positions are not stored */
  if (stepnum == p->firsttimestep) return 0;

  /* message to user */
  printf("saving DCD coordinate file at step number %d\n", stepnum);

  ASSERT(c != NULL);
  ASSERT(cb != NULL);
  ASSERT(cb[0].data != NULL);
  ASSERT(cb[0].attrib.type == MD_DVEC);
  ASSERT(cb[0].attrib.len == c->sys->atom_len);
  ASSERT(cblen == 1 || cblen == 2);

  if (cblen == 2) {
    ASSERT(cb[1].data != NULL);
    ASSERT(cb[1].attrib.type == c->sys->result_attr.type);
    ASSERT(cb[1].attrib.len == 1);
    ASSERT(cell != NULL);
    cell->vec1 = *((MD_Dvec *)(((char *) cb[1].data) + c->offset_cellvec1));
    cell->vec2 = *((MD_Dvec *)(((char *) cb[1].data) + c->offset_cellvec2));
    cell->vec3 = *((MD_Dvec *)(((char *) cb[1].data) + c->offset_cellvec3));
  }

  if (mdio_writeFrameDcd(dcd, pos, cell, stepnum)) {
    return warn(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) dcd));
  }
  return 0;
}


/* callback to write frame for velocity DCD file */
int32 callback_veldcd(void *info, MD_Cbdata *cb, int32 cblen, int32 stepnum)
{
  Callback *c = (Callback *) info;
  SimParam *p = c->sp;
  MD_Dvec *vel = cb[0].data;
  mdio_Dcd *dcd = &(c->veldcd);
  mdio_DcdCell *cell = (p->is_dcdunitcell ? &(c->cell) : NULL);

  /* initial velocities are not stored */
  if (stepnum == p->firsttimestep) return 0;

  /* message to user */
  printf("saving DCD velocity file at step number %d\n", stepnum);

  ASSERT(c != NULL);
  ASSERT(cb != NULL);
  ASSERT(cb[0].data != NULL);
  ASSERT(cb[0].attrib.type == MD_DVEC);
  ASSERT(cb[0].attrib.len == c->sys->atom_len);
  ASSERT(cblen == 1 || cblen == 2);

  if (cblen == 2) {
    ASSERT(cb[1].data != NULL);
    ASSERT(cb[1].attrib.type == c->sys->result_attr.type);
    ASSERT(cb[1].attrib.len == 1);
    ASSERT(cell != NULL);
    cell->vec1 = *((MD_Dvec *)(((char *) cb[1].data) + c->offset_cellvec1));
    cell->vec2 = *((MD_Dvec *)(((char *) cb[1].data) + c->offset_cellvec2));
    cell->vec3 = *((MD_Dvec *)(((char *) cb[1].data) + c->offset_cellvec3));
  }

  if (mdio_writeFrameDcd(dcd, vel, cell, stepnum)) {
    return warn(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) dcd));
  }
  return 0;
}


int32 callback_results(void *info, MD_Cbdata *cb, int32 cblen, int32 stepnum)
{
  Callback *c = (Callback *) info;
  SimParam *p = c->sp;
  int32 is_header = FALSE;

  ASSERT(c != NULL);
  ASSERT(cb != NULL);
  ASSERT(cblen == 1);

  /* print out header line at regular intervals */
  if (c->cnt == p->resultsheaderfreq) {
    is_header = TRUE;
    c->cnt = 0;
  }
  c->cnt++;

  /* call routine to do printing to stdout */
  if (results_text(stdout, c, cb[0].data, stepnum, is_header)) return MD_FAIL;

  /* see if we are also logging results to a file */
  if (p->resultsname) {
    return (c->sp->is_binaryresults ? 
        results_binary(c->logfile, c, cb[0].data, stepnum, is_header) :
        results_text(c->logfile, c, cb[0].data, stepnum, is_header));
  }
  return 0;
}



/*** internal routines ***/

/*
 * Need to determine timestep: engine should have this value
 * - try to acquire "param" object and obtain member "timestep",
 * otherwise obtain engine data array "timestep".
 *
 * Next see if unitcell info is requested.  If so, engine should
 * offer "cellbasisvector[1-3]" either as "param" members or as
 * separate engine data arrays.  Check if "cellvec" (or "cellvec1",
 * "cellvec2", "cellvec3") is member of "result"; if so, provide
 * "result" to callback for dynamically updated cell basis vector
 * info; otherwise, consider cell basis vectors to be fixed.
 */
int32 get_dcdprms(Callback *c, MD_Engine *e)
{
  SimParam *p = c->sp;
  System *s = c->sys;
  void *v;
  int32 param_id = MD_FAIL;
  int32 param_type = MD_FAIL;
  int32 id = MD_FAIL;
  MD_Member mem = { MD_FAIL, MD_FAIL, NULL };
  MD_Attrib attr = { MD_FAIL, MD_FAIL, MD_FAIL, MD_FAIL };

  ASSERT(c != NULL);
  ASSERT(e != NULL);
  ASSERT(c->param_obj == NULL);

  /* see if engine offers data array "param" (of type "Param") */
  if ((param_id = MD_idnum(e, "param")) != MD_FAIL
      && (param_type = MD_attrib(e, param_id).type) != MD_FAIL) {
    /* assume that "param" has passed "consistency check" */
    if ((c->param_obj = malloc(MD_SIZEOF(param_type))) == NULL) {
      return error(MSG_NOMEM);
    }
    if (MD_read(e, param_id, c->param_obj, 1) || MD_wait(e)) {
      return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
    }
  }
  else if (MD_reset(e)) {
    return error(MSG_MDAPI, "from MD_reset()", MD_errmsg(e));
  }

  /* get "timestep" from engine, store value into c->timestep */
  if ((v = MD_type_member(e, param_type, c->param_obj,
          "timestep", &mem)) != NULL
      /* also check sanity of member */
      && mem.type == MD_DOUBLE && mem.len == 1) {
    c->timestep = *((double *) v);
  }
  else if (MD_errnum(e) != 0 && MD_reset(e) == 0
      && (id = MD_idnum(e, "timestep")) != MD_FAIL
      /* also check sanity of engine data array */
      && (attr = MD_attrib(e, id)).type == MD_DOUBLE
      && attr.len == 1 && attr.max == 1 && (attr.access & MD_READ)) {
    if (MD_read(e, id, &(c->timestep), 1) || MD_wait(e)) {
      return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
    }
  }
  else {
    return error(MSG_ENGINE, "DCD writer needs \"timestep\"");
  }

  if (p->is_dcdunitcell) {
    /*
     * first must get initial cell info
     */

    /* get "cellbasisvector1" from engine, store value into c->cell.vec1 */
    if ((v = MD_type_member(e, param_type, c->param_obj,
            "cellbasisvector1", &mem)) != NULL
        /* also check sanity of member */
        && mem.type == MD_DVEC && mem.len == 1) {
      c->cell.vec1 = *((MD_Dvec *) v);
    }
    else if (MD_errnum(e) != 0 && MD_reset(e) == 0
        && (id = MD_idnum(e, "cellbasisvector1")) != MD_FAIL
        /* also check sanity of engine data array */
        && (attr = MD_attrib(e, id)).type == MD_DVEC
        && attr.len == 1 && attr.max == 1 && (attr.access & MD_READ)) {
      if (MD_read(e, id, &(c->cell.vec1), 1) || MD_wait(e)) {
        return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
      }
    }
    else {
      return error(MSG_ENGINE, "DCD writer needs \"cellbasisvector1\"");
    }

    /* get "cellbasisvector2" from engine, store value into c->cell.vec2 */
    if ((v = MD_type_member(e, param_type, c->param_obj,
            "cellbasisvector2", &mem)) != NULL
        /* also check sanity of member */
        && mem.type == MD_DVEC && mem.len == 1) {
      c->cell.vec2 = *((MD_Dvec *) v);
    }
    else if (MD_errnum(e) != 0 && MD_reset(e) == 0
        && (id = MD_idnum(e, "cellbasisvector2")) != MD_FAIL
        /* also check sanity of engine data array */
        && (attr = MD_attrib(e, id)).type == MD_DVEC
        && attr.len == 1 && attr.max == 1 && (attr.access & MD_READ)) {
      if (MD_read(e, id, &(c->cell.vec2), 1) || MD_wait(e)) {
        return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
      }
    }
    else {
      return error(MSG_ENGINE, "DCD writer needs \"cellbasisvector2\"");
    }

    /* get "cellbasisvector3" from engine, store value into c->cell.vec3 */
    if ((v = MD_type_member(e, param_type, c->param_obj,
            "cellbasisvector3", &mem)) != NULL
        /* also check sanity of member */
        && mem.type == MD_DVEC && mem.len == 1) {
      c->cell.vec3 = *((MD_Dvec *) v);
    }
    else if (MD_errnum(e) != 0 && MD_reset(e) == 0
        && (id = MD_idnum(e, "cellbasisvector3")) != MD_FAIL
        /* also check sanity of engine data array */
        && (attr = MD_attrib(e, id)).type == MD_DVEC
        && attr.len == 1 && attr.max == 1 && (attr.access & MD_READ)) {
      if (MD_read(e, id, &(c->cell.vec3), 1) || MD_wait(e)) {
        return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
      }
    }
    else {
      return error(MSG_ENGINE, "DCD writer needs \"cellbasisvector3\"");
    }

    /*
     * next see if cell info is offered by engine "result"
     * (might be either 3-element array "cellvec" of MD_Dvec
     * or 3 MD_Dvec values "cellvec1", "cellvec2", "cellvec3")
     */
    if ((v = MD_type_member(e, s->result_attr.type, s->resultbuf,
            "cellvec", &mem)) != NULL) {
      /* sanity check on "cellvec" */
      if (mem.type != MD_DVEC || mem.len != 3) {
        return error(MSG_ENGINE, "\"result\" member \"cellvec\" "
            "has unexpected definition");
      }
      c->offset_cellvec1 = ((char *)v) - ((char *)(s->resultbuf));
      c->offset_cellvec2 = c->offset_cellvec1 + MD_SIZEOF(MD_DVEC);
      c->offset_cellvec3 = c->offset_cellvec2 + MD_SIZEOF(MD_DVEC);
      c->with_result = TRUE;
    }
    else if (MD_errnum(e) != 0 && MD_reset(e) == 0
        && (v = MD_type_member(e, s->result_attr.type, s->resultbuf,
            "cellvec1", &mem)) != NULL) {
      /* sanity check on "cellvec1" */
      if (mem.type != MD_DVEC || mem.len != 1) {
        return error(MSG_ENGINE, "\"result\" member \"cellvec1\" "
            "has unexpected definition");
      }
      c->offset_cellvec1 = ((char *)v) - ((char *)(s->resultbuf));
      /* since we have "cellvec1", we expect "cellvec2" and "cellvec3" */
      if ((v = MD_type_member(e, s->result_attr.type, s->resultbuf,
              "cellvec2", &mem)) == NULL) {
        return error(MSG_ENGINE, "\"result\" does not have expected "
            "member \"cellvec2\"");
      }
      else if (mem.type != MD_DVEC || mem.len != 1) {
        return error(MSG_ENGINE, "\"result\" member \"cellvec2\" "
            "has unexpected definition");
      }
      c->offset_cellvec2 = ((char *)v) - ((char *)(s->resultbuf));
      if ((v = MD_type_member(e, s->result_attr.type, s->resultbuf,
              "cellvec3", &mem)) == NULL) {
        return error(MSG_ENGINE, "\"result\" does not have expected "
            "member \"cellvec3\"");
      }
      else if (mem.type != MD_DVEC || mem.len != 1) {
        return error(MSG_ENGINE, "\"result\" member \"cellvec3\" "
            "has unexpected definition");
      }
      c->offset_cellvec3 = ((char *)v) - ((char *)(s->resultbuf));
      c->with_result = TRUE;
    }
    else if (MD_errnum(e) != 0 && MD_reset(e) != 0) {
      return error(MSG_MDAPI, "from MD_reset()", MD_errmsg(e));
    }
    else {
      c->with_result = FALSE;
    }

    /*
     * make sure that none of the cell basis vectors are zero
     * (otherwise, print warning and *turn off* dcdunitcell option)
     */
    if ((c->cell.vec1.x==0 && c->cell.vec1.y==0 && c->cell.vec1.z==0)
        || (c->cell.vec2.x==0 && c->cell.vec2.y==0 && c->cell.vec2.z==0)
        || (c->cell.vec3.x==0 && c->cell.vec3.y==0 && c->cell.vec3.z==0)) {
      warn("simparam \"dcdunitcell\" requires nonzero cell basis vectors, "
          "turning off this option");
      p->is_dcdunitcell = FALSE;
      c->with_result = FALSE;
      memset(&(c->cell), 0, sizeof(mdio_DcdCell));
      c->offset_cellvec1 = 0;
      c->offset_cellvec2 = 0;
      c->offset_cellvec3 = 0;
    }
  }
  return 0;
}


int32 results_text(FILE *f, Callback *c, void *result, int32 stepnum,
    int32 is_header)
{
  SimParam *p = c->sp;
  char *start = (char *) result;
  ResultsInfo *r = c->results_info;
  int32 len = c->results_info_len;
  int32 width = p->resultswidth - 1;
  int32 k;
  MD_Fvec fv;
  MD_Dvec dv;

  ASSERT(f != NULL);

  /* see if we need to print out header line */
  if (is_header) {
    fprintf(f, "%s\n", p->results_header);
  }

  /*
   * print results line
   * perform error checking only for first and last fprintf()
   */
  if (fprintf(f, "%*d", p->results_stepnum_width, stepnum) < 0) {
    return warn("unable to write to results log file \"%s\"", p->resultsname);
  }
  for (k = 0;  k < len;  k++) {
    switch (r[k].type) {
      case MD_CHAR:
        if (r[k].len == 1) {
          fprintf(f, " %*c", width, *(start + r[k].offset));
          break;
        }
      case MD_NAME:
      case MD_STRING:
      case MD_MESSAGE:
        fprintf(f, " %*s", width, *((char **)(start + r[k].offset)));
        break;
      case MD_INT32:
        fprintf(f, " %*d", width, (int) *((int32 *)(start + r[k].offset)));
        break;
      case MD_FLOAT:
        fprintf(f, " %*f", width, (double) *((float *)(start + r[k].offset)));
        break;
      case MD_DOUBLE:
        fprintf(f, " %*f", width, *((double *)(start + r[k].offset)));
        break;
      case MD_FVEC:
        fv = *((MD_Fvec *)(start + r[k].offset));
        fprintf(f, " %*g %*g %*g", width, (double) (fv.x),
            width, (double) (fv.y), width, (double) (fv.z));
        break;
      case MD_DVEC:
        dv = *((MD_Dvec *)(start + r[k].offset));
        fprintf(f, " %*g %*g %*g", width, dv.x, width, dv.y, width, dv.z);
        break;
      default:
        BUG("unexpected type found in ResultsInfo array");
    }
  }
  if (fprintf(f, "\n") < 0) {
    return warn("unable to write to results log file \"%s\"", p->resultsname);
  }

  return 0;
}


int32 results_binary(FILE *f, Callback *c, void *result, int32 stepnum,
    int32 is_header)
{
  BUG("results_binary() not yet implemented");
  return 0;
}
