/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    topo.h
 * @brief   Topology file reader.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c mdio_Topo class reads an X-Plor PSF topology file, storing
 * the results into arrays of MDAPI topology data types defined in
 * @c mdtypes.h.  There is also a routine to setup the cross-indexing
 * of the topology arrays into the force field parameter arrays,
 * given an @c mdio_Param object.
 *
 * The @c mdio_Topo class is derivde from @c mdio_File so that error
 * handling is performed by simply typecasting (@c mdio_Topo @c *)
 * to a (@c mdio_File @c *).
 */

#ifndef MDIO_TOPO_H
#define MDIO_TOPO_H

#include "adt/array.h"
#include "mdio/file.h"
#include "mdio/param.h"
#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**@brief Topology file reader class.
   *
   * Members should be treated as private.
   */
  typedef struct mdio_Topo_t {
    mdio_File file;
    adt_Array atom;
    adt_Array bond;
    adt_Array angle;
    adt_Array dihed;
    adt_Array impr;
    adt_Array excl;
    char *buffer;
    int bufferlen;
  } mdio_Topo;


/* constructor and destructor */

  /**@brief Constructor.
   *
   * Creates dynamically allocated topology file reader object.
   *
   * @return Pointer to new object or @c NULL on failure.
   */
  mdio_Topo *mdio_createTopo(void);


  /**@brief Destructor.
   *
   * Frees dynamically allocated topology file reader object after
   * freeing the arrays of collected topology data types.
   *
   * Note that the topology arrays obtained do not persist beyond
   * destruction of @c mdio_Topo object.
   */
  void mdio_destroyTopo(mdio_Topo *);


/* reading parameter files, indexing params, obtaining parsed data */

  /**@brief Read topology file.
   *
   * @param[in] name  A nil-terminated string naming the topology file.
   *
   * Opens the file, reads and parses the file contents, and closes the
   * file.  Incrementally builds arrays storing the topology data types.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_readTopo(mdio_Topo *, const char *name);


  /**@brief Index force field parameter arrays from topology arrays.
   *
   * @param[in,out] p  Force field parameters have already been read.
   *
   * Computes the indexing that matches the particular bonded interactions
   * described by the topology arrays to their bond types described by
   * the force field parameter arrays.  The @c MD_NbfixPrm array that is
   * part of @c p is also updated to index the correct @c MD_AtomPrm array
   * entries.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   *
   * Note that if multiple definitions for a force field parameter occurs,
   * a warning will be issued.  The last such definition will be the one
   * indexed by the corresponding topology array.  A warning will not
   * result in returning the @c MDIO_ERROR value, but the error handler
   * will be invoked for each warning.  A way of determining after the
   * call that some warning occurred is to compare @c mdio_getErrnumFile()
   * against the value @c MDIO_ERROR_WARN.
   */
  int mdio_indexParamTopo(mdio_Topo *, mdio_Param *p);


  /**@brief Obtain array of atoms.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The atom array as a pointer.
   */
  MD_Atom *mdio_getAtomTopo(mdio_Topo *, int *nelems);


  /**@brief Obtain array of bonds.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The bond array as a pointer.
   */
  MD_Bond *mdio_getBondTopo(mdio_Topo *, int *nelems);


  /**@brief Obtain array of angles.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The angle array as a pointer.
   */
  MD_Angle *mdio_getAngleTopo(mdio_Topo *, int *nelems);


  /**@brief Obtain array of dihedrals.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The dihedral array as a pointer.
   */
  MD_Tors *mdio_getDihedTopo(mdio_Topo *, int *nelems);


  /**@brief Obtain array of impropers.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The improper array as a pointer.
   */
  MD_Tors *mdio_getImprTopo(mdio_Topo *, int *nelems);


  /**@brief Obtain array of exclusions.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The exclusion array as a pointer.
   */
  MD_Excl *mdio_getExclTopo(mdio_Topo *, int *nelems);


/* constructor and destructor for pre-allocated mdio_Topo object */

  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated @c mdio_Topo object.
   * See @c mdio_createTopo() for a description of expected arguments.
   */
  int mdio_initializeTopo(mdio_Topo *);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated @c mdio_Topo object
   * (i.e. one constructed using @c mdio_initializeTopo() ).
   */
  void mdio_cleanupTopo(mdio_Topo *);

#ifdef __cplusplus
}
#endif

#endif /* MDIO_TOPO_H */
