/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * demo_topo.c - demonstrate reading X-Plor PSF file
 */

#include <stdlib.h>
#include <stdio.h>
#include "mdio/topo.h"
#include "mdio/param.h"


int main(int argc, char *argv[])
{
  mdio_Param *prm;
  MD_AtomPrm *atomprm = NULL;
  MD_BondPrm *bondprm = NULL;
  MD_AnglePrm *angleprm = NULL;
  MD_TorsPrm *dihedprm = NULL;
  MD_TorsPrm *imprprm = NULL;
  MD_NbfixPrm *nbfixprm = NULL;
  int natomprm;
  int nbondprm;
  int nangleprm;
  int ndihedprm;
  int nimprprm;
  int nnbfixprm;
  mdio_Topo *topo;
  MD_Atom *atom;
  MD_Bond *bond;
  MD_Angle *angle;
  MD_Tors *dihed;
  MD_Tors *impr;
  MD_Excl *excl;
  int natom;
  int nbond;
  int nangle;
  int ndihed;
  int nimpr;
  int nexcl;
  int k;

  if (argc < 2) {
    fprintf(stderr, "usage:  %s  PSF-file  [param-file-1  param-file2 ... ]\n",
        argv[0]);
    exit(1);
  }

  /* setup */
  if ((prm = mdio_createParam()) == NULL) {
    fprintf(stderr, "failed to create param reader object\n");
    exit(1);
  }
  if ((topo = mdio_createTopo()) == NULL) {
    fprintf(stderr, "failed to create topo reader object\n");
    exit(1);
  }

  if (argc >= 3) {
    /* read param files */
    for (k = 2;  k < argc;  k++) {
      if (mdio_readParam(prm, argv[k])) {
        fprintf(stderr, "unable to read params from file \"%s\"\n", argv[k]);
        exit(1);
      }
      else {
        printf("Successfully read params from file \"%s\"\n", argv[k]);
      }
    }

    /* obtain data */
    atomprm = mdio_getAtomParam(prm, &natomprm);
    bondprm = mdio_getBondParam(prm, &nbondprm);
    angleprm = mdio_getAngleParam(prm, &nangleprm);
    dihedprm = mdio_getDihedParam(prm, &ndihedprm);
    imprprm = mdio_getImprParam(prm, &nimprprm);
    nbfixprm = mdio_getNbfixParam(prm, &nnbfixprm);
  }


  /* read topology file */
  if (mdio_readTopo(topo, argv[1])) {
    fprintf(stderr, "unable to read topology from file \"%s\"\n", argv[1]);
    exit(1);
  }

  /* obtain data */
  atom = mdio_getAtomTopo(topo, &natom);
  bond = mdio_getBondTopo(topo, &nbond);
  angle = mdio_getAngleTopo(topo, &nangle);
  dihed = mdio_getDihedTopo(topo, &ndihed);
  impr = mdio_getImprTopo(topo, &nimpr);
  excl = mdio_getExclTopo(topo, &nexcl);

  if (argc >= 3) {
    if (mdio_indexParamTopo(topo, prm)) {
      fprintf(stderr, "failed to index parameter data from topology\n");
      exit(1);
    }
  }

  /* print topology data */
  printf("*** TOPOLOGY DATA ***\n\n");
  printf("(atom indices are incremented by one to agree with PSF values)\n");
  printf("\natoms: %d\n", natom);
  for (k = 0;  k < natom;  k++) {
    printf(" %5d  m=%-8g  q=%-8g  prm=%-5d  name=%-4s  type=%-4s\n",
        k+1, atom[k].m, atom[k].q, atom[k].prm, atom[k].name, atom[k].type);
  }
  printf("\nbonds: %d\n", nbond);
  for (k = 0;  k < nbond;  k++) {
    printf(" %5d  atom[0]=%-5d  atom[1]=%-5d  prm=%-5d\n",
        k+1, bond[k].atom[0]+1, bond[k].atom[1]+1, bond[k].prm);
  }
  printf("\nangles: %d\n", nangle);
  for (k = 0;  k < nangle;  k++) {
    printf(" %5d  atom[0]=%-5d  atom[1]=%-5d  atom[2]=%-5d  prm=%-5d\n",
        k+1, angle[k].atom[0]+1, angle[k].atom[1]+1, angle[k].atom[2]+1,
        angle[k].prm);
  }
  printf("\ndihedrals: %d\n", ndihed);
  for (k = 0;  k < ndihed;  k++) {
    printf(" %5d  atom[0]=%-5d  atom[1]=%-5d  atom[2]=%-5d  atom[3]=%-5d  "
        "prm=%-5d\n", k+1, dihed[k].atom[0]+1, dihed[k].atom[1]+1,
        dihed[k].atom[2]+1, dihed[k].atom[3]+1, dihed[k].prm);
  }
  printf("\nimpropers: %d\n", nimpr);
  for (k = 0;  k < nimpr;  k++) {
    printf(" %5d  atom[0]=%-5d  atom[1]=%-5d  atom[2]=%-5d  atom[3]=%-5d  "
        "prm=%-5d\n", k+1, impr[k].atom[0]+1, impr[k].atom[1]+1,
        impr[k].atom[2]+1, impr[k].atom[3]+1, impr[k].prm);
  }
  printf("\nexclusions: %d\n", nexcl);
  for (k = 0;  k < nexcl;  k++) {
    printf(" %5d  atom[0]=%-5d  atom[1]=%-5d\n",
        k+1, excl[k].atom[0]+1, excl[k].atom[1]+1);
  }

  if (argc >= 3) {
    /* print param data */
    printf("\n\n*** FORCE FIELD PARAMETER DATA ***\n\n");
    for (k = 0;  k < natomprm;  k++) {
      MD_AtomPrm *p = &atomprm[k];
      printf("atomprm[%d]: emin=%g rmin=%g emin14=%g rmin14=%g %s\n",
          k, p->emin, p->rmin, p->emin14, p->rmin14, p->type);
    }
    for (k = 0;  k < nbondprm;  k++) {
      MD_BondPrm *p = &bondprm[k];
      printf("bondprm[%d]: k=%g r0=%g %s %s\n",
          k, p->k, p->r0, p->type[0], p->type[1]);
    }
    for (k = 0;  k < nangleprm;  k++) {
      MD_AnglePrm *p = &angleprm[k];
      printf("angleprm[%d]: k_theta=%g theta0=%g k_ub=%g r_ub=%g %s %s %s\n",
          k, p->k_theta, p->theta0, p->k_ub, p->r_ub,
          p->type[0], p->type[1], p->type[2]);
    }
    for (k = 0;  k < ndihedprm;  k++) {
      MD_TorsPrm *p = &dihedprm[k];
      printf("dihedprm[%d]: k_tor=%g phi=%g n=%d mult=%d %s %s %s %s\n",
          k, p->k_tor, p->phi, p->n, p->mult,
          p->type[0], p->type[1], p->type[2], p->type[3]);
    }
    for (k = 0;  k < nimprprm;  k++) {
      MD_TorsPrm *p = &imprprm[k];
      printf("imprprm[%d]: k_tor=%g phi=%g n=%d mult=%d %s %s %s %s\n",
          k, p->k_tor, p->phi, p->n, p->mult,
          p->type[0], p->type[1], p->type[2], p->type[3]);
    }
    for (k = 0;  k < nnbfixprm;  k++) {
      MD_NbfixPrm *p = &nbfixprm[k];
      printf("nbfixprm[%d]: emin=%g rmin=%g emin14=%g rmin14=%g %d %d %s %s\n",
          k, p->emin, p->rmin, p->emin14, p->rmin14,
          p->prm[0], p->prm[1], p->type[0], p->type[1]);
    }
  }

  /* cleanup file readers */
  mdio_destroyTopo(topo);
  mdio_destroyParam(prm);

  return 0;
}
