/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * demo_pdbcoord.c - demonstrate reading and writing PDB coordinate
 *   files by reading in an existing PDB coordinate file and writing
 *   the contents to a new PDB file
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "mdio/pdbcoord.h"
#include "debug/debug.h"


int main(int argc, char *argv[])
{
  /* MDIO objects */
  mdio_Pdbcoord *pdb;

  /* coordinate arrays and lengths */
  mdio_Pdbatom *atom;
  MD_Dvec *coord;
  int atom_len;
  int coord_len;

  /* filenames */
  const char *fnread;
  const char *fnwrite;

  /* check expected command line arguments */
  if (argc != 3) {
    fprintf(stderr, "usage:  %s PDB1 PDB2\n", argv[0]);
    exit(1);
  }

  /* file names */
  fnread = argv[1];
  fnwrite = argv[2];
  if (strcmp(fnread, fnwrite) == 0) {
    fprintf(stderr, "file names must be distinct\n");
    exit(1);
  }

  if ((pdb = mdio_createPdbcoord()) == NULL) {
    fprintf(stderr, "failed to create pdb object\n");
    exit(1);
  }

  /* read PDB file */
  if (mdio_readPdbcoord(pdb, fnread, 0)) {
    fprintf(stderr, "failed to read pdb file \"%s\"\n", fnread);
    exit(1);
  }
  atom = mdio_getAtomPdbcoord(pdb, &atom_len);
  if (atom == NULL) {
    fprintf(stderr, "failed to obtain pdb atoms\n");
    exit(1);
  }
  coord = mdio_getPdbcoord(pdb, &coord_len);
  if (coord == NULL) {
    fprintf(stderr, "failed to obtain pdb coordinates\n");
    exit(1);
  }
  if (atom_len != coord_len) {
    fprintf(stderr, "data arrays are different lengths\n");
    exit(1);
  }
  printf("successful read from %s\n", fnread);

  /*
   * write PDB file
   *
   * note:  we depend on pdb object retaining its data
   */
  if (mdio_writePdbcoord(pdb, fnwrite)) {
    fprintf(stderr, "failed to write pdb file \"%s\"\n", fnwrite);
    exit(1);
  }
  printf("successful write to %s\n", fnwrite);

  /* success */
  mdio_destroyPdbcoord(pdb);

  return 0;
}
