/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * demo_bincoord.c - demonstrate by reading PDB file, then writing the
 *   binary file, reading it back, then printing those coordinates
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "mdio/bincoord.h"
#include "mdio/pdbcoord.h"
#include "debug/debug.h"


int main(int argc, char *argv[])
{
  /* MDIO objects */
  mdio_Bincoord *bin;
  mdio_Pdbcoord *pdb;

  /* coordinate arrays and lengths */
  MD_Dvec *coord;
  MD_Dvec *newcoord;
  int n, nn, k;

  /* filenames */
  const char *fnpdb;
  const char *fnbin;

  /* check expected command line arguments */
  if (argc != 3) {
    fprintf(stderr, "usage:  %s PDB BIN\n", argv[0]);
    exit(1);
  }

  /* initialize file names */
  fnpdb = argv[1];
  fnbin = argv[2];
  if (strcmp(fnpdb, fnbin) == 0) {
    fprintf(stderr, "file names must be distinct\n");
    exit(1);
  }

  /* I/O object constructors */
  if ((bin = mdio_createBincoord()) == NULL) {
    fprintf(stderr, "failed to create Bincoord object\n");
    exit(1);
  }
  if ((pdb = mdio_createPdbcoord()) == NULL) {
    fprintf(stderr, "failed to create Pdbcoord object\n");
    exit(1);
  }

  /* read PDB file */
  if (mdio_readPdbcoord(pdb, fnpdb, 0)) {
    fprintf(stderr, "failed to read pdb file \"%s\"\n", fnpdb);
    exit(1);
  }
  coord = mdio_getPdbcoord(pdb, &n);
  if (coord == NULL) {
    fprintf(stderr, "failed to obtain coordinates from Pdbcoord object\n");
    exit(1);
  }
  printf("successful read from %s\n", fnpdb);

  /* write binary file */
  if (mdio_setBincoord(bin, coord, n)) {
    fprintf(stderr, "failed to set coordinates for Bincoord object\n");
    exit(1);
  }
  if (mdio_writeBincoord(bin, fnbin)) {
    fprintf(stderr, "failed to write bincoord file \"%s\"\n", fnbin);
    exit(1);
  }
  printf("successful write to %s\n", fnbin);

  /*
   * read binary file
   *
   * note:  reset coordinates so as not to clobber pdb coordinate data
   */
  if (mdio_setBincoord(bin, NULL, 0)) {
    fprintf(stderr, "failed to reset coordinates for Bincoord object\n");
    exit(1);
  }
  if (mdio_readBincoord(bin, fnbin, n)) {
    fprintf(stderr, "failed to read bincoord file \"%s\"\n", fnbin);
    exit(1);
  }
  newcoord = mdio_getBincoord(bin, &nn);
  if (newcoord == NULL) {
    fprintf(stderr, "failed to obtain coordinates from Bincoord object\n");
    exit(1);
  }
  printf("successful read from %s\n", fnbin);

  /* success */
  for (k = 0;  k < nn;  k++) {
    printf("ATOM  %5d %8.3f %8.3f %8.3f\n", k+1,
        newcoord[k].x, newcoord[k].y, newcoord[k].z);
  }

  /* destroy I/O objects */
  mdio_destroyPdbcoord(pdb);
  mdio_destroyBincoord(bin);
  return 0;
}
