/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    dcd.h
 * @brief   DCD trajectory file writer.
 * @author  David J. Hardy
 * @date    2004-2005
 *
 * The @c mdio_Dcd class writes DCD trajectory files equivalent to
 * those produced by NAMD.  For now, support is only available for
 * file sizes less than 2 gigabytes.
 *
 * Writing the DCD file is performed by the following sequence
 * of routines:
 * @li calling @c mdio_writeBeginDcd() once to open the file
 * @li calling @c mdio_writeHeaderDcd() once to write the required
 *   header information
 * @li calling @c mdio_writeFrameDcd() multiple times, once for each
 *   frame of atomic coordinates
 * @li calling @c mdio_writeEndDcd() once to close the file
 *
 * The @c mdio_Dcd class is derived from @c mdio_File so that error
 * handling is performed by typecasting (@c mdio_Bincoord @c *) to
 * (@c mdio_File @c *).
 */

#ifndef MDIO_DCD_H
#define MDIO_DCD_H

#include "mdio/file.h"
#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**
   * Set appropriate @c cellstatus flag to indicate periodic cell.
   */
  enum {
    MDIO_DCD_UNITCELL  = 0x001, /**< Flag for a periodic cell. */
    MDIO_DCD_FIXEDCELL = 0x002  /**< Flag showing cell dimensions are fixed. */
  };

  /**@brief DCD header information provided by user.
   */
  typedef struct mdio_DcdHeader_t {
    double timestep;    /**< Size of timestep in femtoseconds. */
    int32 natoms;       /**< Number of atoms in system. */
    int32 firststep;    /**< Numbering of the first timestep. */
    int32 framestepcnt; /**< Number of steps between each frame. */
    int32 cellstatus;   /**< Cell status flag to indicate a periodic cell. */
  } mdio_DcdHeader;

  /**@brief DCD periodic cell information provided by user.
   */
  typedef struct mdio_DcdCell_t {
    MD_Dvec vec1;  /**< First vector direction. */
    MD_Dvec vec2;  /**< Second vector direction. */
    MD_Dvec vec3;  /**< Third vector direction. */
  } mdio_DcdCell;

  /**@brief DCD trajectory file writer class.
   *
   * Members should be treated as private.
   */
  typedef struct mdio_Dcd_t {
    mdio_File file;
    struct {
      int32 nbhead_begin;   /* = 84 (bytes for "head" section) */
      char title[4];        /* = "CORD" */
      int32 numframes;      /*  number of frames, updated as we write */
      int32 firststep;      /*  first step number */
      int32 framestepcnt;   /*  number of steps per frame */
      int32 numsteps;       /*  total number sim steps + first step, updated */
      int32 zero5[5];       /* = 0 */
      float timestep;       /*  time step */
      int32 iscell;         /* = 1 if unitcell info, = 0 if not */
      int32 zero8[8];       /* = 0 */
      int32 charmmversion;  /* = 24 (CHARMM 24 format DCD file) */
      int32 nbhead_end;     /* = 84 (bytes for "head" section) */

      int32 nbtitle_begin;  /* = 164 (bytes for "title" section) */
      int32 numtitle;       /* = 2 (i.e. two 80-char title strings) */
      char title_str[80];   /*  remarks title string */
      char create_str[80];  /*  remarks create time string */
      int32 nbtitle_end;    /* = 164 (bytes for "title" section) */

      int32 nbnatoms_begin; /* = 4 (bytes for "natoms" section) */
      int32 natoms;         /*  number of atoms */
      int32 nbnatoms_end;   /* = 4 (bytes for "natoms" section) */
    } header;
    struct {
      int32 dummy1;         /*  (padding to 8-byte word alignment) */
      int32 nbcell_begin;   /* = 48 (bytes for "unitcell" section) */
      double unitcell[6];   /*  describes periodic cell */
      int32 nbcell_end;     /* = 48 (bytes for "unitcell" section) */
      int32 dummy2;         /*  (padding to 8-byte word alignment) */
    } cell;
    int32 *coordbuf;        /*  buffer for coordinates with byte num info */
    float *x;               /*  points to x-coordinates in coordbuf */
    float *y;               /*  points to y-coordinates in coordbuf */
    float *z;               /*  points to z-coordinates in coordbuf */
    int32 coordbuflen;      /* = 3*(N + 2) elements */
    int32 nbcell;           /* = 56 (bytes written for "unitcell") */
    int32 stepnum;
    int32 status;
  } mdio_Dcd;


/* constructor and destructor */

  /**@brief Constructor.
   *
   * Creates dynamically allocated DCD trajectory file writer object.
   *
   * @return Pointer to new object or @c NULL on failure.
   */
  mdio_Dcd *mdio_createDcd(void);


  /**@brief Destructor.
   *
   * Frees dynamically allocated DCD trajectory file writer object.
   */
  void mdio_destroyDcd(mdio_Dcd *);


/* write DCD file */

  /**@brief Begin writing DCD file.
   *
   * @param[in] name  A nil-terminated string naming the DCD file.
   *
   * Call this once when writing a DCD file.  Opens file for writing.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_writeBeginDcd(mdio_Dcd *, const char *name);


  /**@brief Write header information to DCD file.
   *
   * @param[in] header  Contains header information from user.
   *
   * User must supply appropriate values to @c mdio_DcdHeader fields.
   * Call this once when writing a DCD file.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_writeHeaderDcd(mdio_Dcd *, const mdio_DcdHeader *header);


  /**@brief Write set of coordinates to DCD file.
   *
   * @param[in] coord  Array of coordinates.
   * @param[in] cell  Periodic cell information.
   * @param[in] stepnum  The number of this timestep.
   *
   * User should call this repeatedly for every frame of coordinates
   * supplied to the DCD file during the simulation.
   * The length of @p coord array is supplied to the @c mdio_DcdHeader
   * structure.  If system is not periodic, @p cell should be set to @c NULL.
   * The value of @p stepnum is validated against the
   * @c framestepcnt supplied to @c mdio_DcdHeader.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_writeFrameDcd(mdio_Dcd *, const MD_Dvec *coord,
      const mdio_DcdCell *cell, int32 stepnum);


  /**@brief End writing DCD file.
   *
   * User calls this once at end of DCD file writing.
   * Closes the file.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_writeEndDcd(mdio_Dcd *);


/* constructor and destructor for pre-allocated mdio_Dcd object */

  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated @c mdio_Dcd object.
   * See @c mdio_createDcd() for a description of expected arguments.
   */
  int mdio_initializeDcd(mdio_Dcd *);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated @c mdio_Dcd object
   * (i.e. one constructed using @c mdio_initializeDcd() ).
   */
  void mdio_cleanupDcd(mdio_Dcd *);

#ifdef __cplusplus
}
#endif

#endif /* MDIO_DCD_H */
