/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * dcd.c - write DCD trajectory file
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "mdio/dcd.h"
#include "debug/debug.h"

/* internal status indicator flags */
enum {
  BEGIN        = 0x010,
  HEADER       = 0x020,
  FRAME        = 0x040,
  SET_UNITCELL = 0x080
};


mdio_Dcd *mdio_createDcd(void)
{
  mdio_Dcd *p;
  p = (mdio_Dcd *) malloc(sizeof(mdio_Dcd));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (mdio_initializeDcd(p)) {
    free(p);
    return NULL;
  }
  return p;
}


int mdio_initializeDcd(mdio_Dcd *p)
{
  ASSERT(p != NULL);
  memset(p, 0, sizeof(mdio_Dcd));      /* zero memory */

  if (mdio_initializeFile(&(p->file))) {
    return MDIO_ERROR;
  }

  /* make sure structure lengths are correct */
  if (sizeof(int32) != 4 || sizeof(float) != 4 || sizeof(double) != 8
      || sizeof(p->header) != 276 || sizeof(p->cell) != 64) {
    COND(sizeof(int32) != 4);
    COND(sizeof(float) != 4);
    COND(sizeof(double) != 8);
    COND(sizeof(p->header) != 276);
    COND(sizeof(p->cell) != 64);
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_BADVAL,
        "incorrect byte count for internal data structures");
    return MDIO_ERROR;
  }

  /* set status flags in anticipation of next method call */
  p->status = BEGIN;

  return 0;
}


void mdio_destroyDcd(mdio_Dcd *p)
{
  ASSERT(p != NULL);
  mdio_cleanupDcd(p);
  free(p);
}


void mdio_cleanupDcd(mdio_Dcd *p)
{
  ASSERT(p != NULL);
  free(p->coordbuf);
  mdio_cleanupFile(&(p->file));
}


int mdio_writeBeginDcd(mdio_Dcd *p, const char *name)
{
  /* assert correct arguments */
  ASSERT(p != NULL);
  ASSERT(name != NULL);

  /* check status flags */
  if (p->status != BEGIN) {
    BUG("incorrect use of DCD writer");
  }

  /* open file */
  if (mdio_openFile(&(p->file), name, MDIO_FILE_BINARY | MDIO_FILE_WRITE)) {
    return MDIO_ERROR;
  }

  /* set status flags in anticipation of next method call */
  p->status = HEADER;

  return 0;
}


int mdio_writeHeaderDcd(mdio_Dcd *p, const mdio_DcdHeader *dcdheader)
{
  mdio_File *f;
  time_t abstime;
  struct tm *tmbuf;
  int32 cellstatus = 0;

  ASSERT(p != NULL);
  ASSERT(dcdheader != NULL);
  f = &(p->file);

  /* check status flags */
  if (p->status != HEADER) {
    BUG("incorrect use of DCD writer");
  }

  /* check DcdHeader input */
  if (dcdheader->cellstatus & ~(MDIO_DCD_UNITCELL | MDIO_DCD_FIXEDCELL)) {
    COND(dcdheader->cellstatus & ~(MDIO_DCD_UNITCELL | MDIO_DCD_FIXEDCELL));
    mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL,
       	"illegal value for DcdHeader::cellstatus indicator");
    return MDIO_ERROR;
  }
  else if (dcdheader->timestep <= 0) {
    COND(dcdheader->timestep <= 0);
    mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL,
       	"DcdHeader::timestep must be positive");
    return MDIO_ERROR;
  }
  else if (dcdheader->natoms <= 0) {
    COND(dcdheader->natoms <= 0);
    mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL,
       	"DcdHeader::natoms must be positive");
    return MDIO_ERROR;
  }
  else if (dcdheader->framestepcnt <= 0) {
    COND(dcdheader->framestepcnt <= 0);
    mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL,
       	"DcdHeader::framestepcnt must be positive");
    return MDIO_ERROR;
  }

  /* set constant values in header */
  memset(&(p->header), 0, sizeof(p->header));
  p->header.nbhead_begin = 84;
  strncpy(p->header.title, "CORD", 4);
  p->header.numframes = 0;
  p->header.firststep = dcdheader->firststep;
  p->header.framestepcnt = dcdheader->framestepcnt;
  p->header.numsteps = dcdheader->firststep;
  p->header.timestep = (float)(dcdheader->timestep);
  p->header.iscell = (dcdheader->cellstatus != 0);
  p->header.charmmversion = 24;
  p->header.nbhead_end = 84;

  p->header.nbtitle_begin = 164;
  p->header.numtitle = 2;
  snprintf(p->header.title_str, sizeof(p->header.title_str),
      "REMARKS FILENAME=\"%s\"", mdio_getNameFile(f));
  abstime = time(NULL);
  tmbuf = localtime(&abstime);
  if (tmbuf != NULL) {
    strftime(p->header.create_str, sizeof(p->header.create_str),
        "REMARKS MDIO DCD file created %d %b %Y at %H:%M", tmbuf);
  }
  p->header.nbtitle_end = 164;

  p->header.nbnatoms_begin = 4;
  p->header.natoms = dcdheader->natoms;
  p->header.nbnatoms_end = 4;

  /* check byte sizes within structure */
  ASSERT(sizeof(p->header) == 276);
  ASSERT((char*)&(p->header.nbhead_end) - (char*)&(p->header.nbhead_begin)
      - 4 == p->header.nbhead_begin);
  ASSERT((char*)&(p->header.nbtitle_end) - (char*)&(p->header.nbtitle_begin)
      - 4 == p->header.nbtitle_begin);
  ASSERT((char*)&(p->header.nbnatoms_end) - (char*)&(p->header.nbnatoms_begin)
      - 4 == p->header.nbnatoms_begin);

  /* allocate memory for coordinate data */
  p->coordbuflen = 3 * dcdheader->natoms + 6;
  p->coordbuf = (int32 *) calloc(p->coordbuflen, sizeof(int32));
  if (p->coordbuf == NULL) {
    ERRCOND(p->coordbuf == NULL);
    mdio_setErrorFile(f, MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }

  /* set const values in coordbuf array, number of bytes for each coord set */
  ASSERT(sizeof(int32) == 4);
  p->coordbuf[0] = 4 * dcdheader->natoms;
  p->coordbuf[dcdheader->natoms + 1] = 4 * dcdheader->natoms;
  p->coordbuf[dcdheader->natoms + 2] = 4 * dcdheader->natoms;
  p->coordbuf[2*dcdheader->natoms + 3] = 4 * dcdheader->natoms;
  p->coordbuf[2*dcdheader->natoms + 4] = 4 * dcdheader->natoms;
  p->coordbuf[3*dcdheader->natoms + 5] = 4 * dcdheader->natoms;

  /* set coordinate pointers to point into coordbuf array */
  ASSERT(sizeof(float) == sizeof(int32));
  p->x = (float *)(&(p->coordbuf[1]));
  p->y = (float *)(&(p->coordbuf[dcdheader->natoms + 3]));
  p->z = (float *)(&(p->coordbuf[2*dcdheader->natoms + 5]));

  /* see if we need to fill out cell values */
  if (dcdheader->cellstatus != 0) {
    /* make sure UNITCELL flag is set, might have set FIXEDCELL flag */
    cellstatus = (MDIO_DCD_UNITCELL | dcdheader->cellstatus);
    /* set const values, number of bytes for unitcell array */
    memset(&(p->cell), 0, sizeof(p->cell));
    p->cell.nbcell_begin = 48;
    p->cell.nbcell_end = 48;
    p->nbcell = 56;
    ASSERT(sizeof(double) == 8);
    ASSERT(sizeof(p->cell) == 64);
    ASSERT((char*)&(p->cell.nbcell_end) - (char*)&(p->cell.nbcell_begin) - 4
        == p->cell.nbcell_begin);
    /* expect non-NULL MDIO_DcdCell pointer passed to dcd_set routine */
    cellstatus |= SET_UNITCELL;
  }

  /* write header as a single chunk */
  if (mdio_writeBinaryFile(f, &(p->header), sizeof(p->header), 1) != 1) {
    return MDIO_ERROR;
  }

  /* set status flags in anticipation of next method call */
  p->status = (FRAME | cellstatus);

  return 0;
}


/* file position offsets */
#define OFFSET_NUMFRAMES  8L
#define OFFSET_NUMSTEPS  20L

int mdio_writeFrameDcd(mdio_Dcd *p, const MD_Dvec *coord,
    const mdio_DcdCell *cell, int32 stepnum)
{
  mdio_File *f;
  int32 k, natoms;

  ASSERT(p != NULL);
  ASSERT(coord != NULL);
  ASSERT(cell == NULL || (p->status & MDIO_DCD_UNITCELL));
  f = &(p->file);
  natoms = p->header.natoms;

  /* check status flags */
  if ((p->status & FRAME) == 0) {
    BUG("incorrect use of DCD writer");
  }

  /* check input */
  if (p->header.numsteps + p->header.framestepcnt != stepnum) {
    COND(p->header.numsteps + p->header.framestepcnt != stepnum);
    mdio_setErrorMessageFile(f, MDIO_ERROR_BADVAL,
       	"incorrect value for stepnum");
    return MDIO_ERROR;
  }

  /* set DCD unitcell array */
  if (p->status & SET_UNITCELL) {
    const MD_Dvec a = cell->vec1;
    const MD_Dvec b = cell->vec2;
    const MD_Dvec c = cell->vec3;
    const double alen = sqrt(a.x * a.x + a.y * a.y + a.z * a.z);
    const double blen = sqrt(b.x * b.x + b.y * b.y + b.z * b.z);
    const double clen = sqrt(c.x * c.x + c.y * c.y + c.z * c.z);
    double cos_ab = (a.x * b.x + a.y * b.y + a.z * b.z) / (alen * blen);
    double cos_ac = (a.x * c.x + a.y * c.y + a.z * c.z) / (alen * clen);
    double cos_bc = (b.x * c.x + b.y * c.y + b.z * c.z) / (blen * clen);
    /* adjust in case of roundoff error */
    if (cos_ab < -1.0) cos_ab = -1.0;
    else if (cos_ab > 1.0) cos_ab = 1.0;
    if (cos_ab < -1.0) cos_ab = -1.0;
    else if (cos_ab > 1.0) cos_ab = 1.0;
    if (cos_ab < -1.0) cos_ab = -1.0;
    else if (cos_ab > 1.0) cos_ab = 1.0;
    /* store values into unitcell */
    p->cell.unitcell[0] = alen;
    p->cell.unitcell[1] = MD_DEGREES * acos(cos_ab);
    p->cell.unitcell[2] = blen;
    p->cell.unitcell[3] = MD_DEGREES * acos(cos_ac);
    p->cell.unitcell[4] = MD_DEGREES * acos(cos_bc);
    p->cell.unitcell[5] = clen;
    /* if cell remains fixed, no need to recompute it */
    if (p->status & MDIO_DCD_FIXEDCELL) {
      p->status &= ~SET_UNITCELL;
    }
  }

  /* set DCD coordinates */
  for (k = 0;  k < natoms;  k++) {
    p->x[k] = (float)(coord[k].x);
    p->y[k] = (float)(coord[k].y);
    p->z[k] = (float)(coord[k].z);
  }

  if (p->status & MDIO_DCD_UNITCELL) {
    /* write unitcell as single chunk */
    if (mdio_writeBinaryFile(f, &(p->cell.nbcell_begin), p->nbcell, 1) != 1) {
      return MDIO_ERROR;
    }
  }

  /* write coordinates as array of int32 */
  if (mdio_writeBinaryFile(f, p->coordbuf, sizeof(int32), p->coordbuflen)
      != p->coordbuflen) {
    return MDIO_ERROR;
  }

  /* update "numframes" and "numsteps" fields stored in file */
  p->header.numframes++;
  p->header.numsteps += p->header.framestepcnt;
  if (mdio_seekFile(f, OFFSET_NUMFRAMES, MDIO_SEEK_SET)
      || mdio_writeBinaryFile(f, &(p->header.numframes), sizeof(int32), 1) != 1
      || mdio_seekFile(f, OFFSET_NUMSTEPS, MDIO_SEEK_SET)
      || mdio_writeBinaryFile(f, &(p->header.numsteps), sizeof(int32), 1) != 1
      || mdio_seekFile(f, 0, MDIO_SEEK_END)) {
    return MDIO_ERROR;
  }

  return 0;
}


int mdio_writeEndDcd(mdio_Dcd *p)
{
  ASSERT(p != NULL);

  /* close file */
  if (mdio_closeFile(&(p->file))) {
    return MDIO_ERROR;
  }

  /* free coordinate buffer */
  free(p->coordbuf);
  p->coordbuf = NULL;

  /* reset status */
  p->status = BEGIN;

  return 0;
}
