/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * config.c
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "mdio/config.h"
#include "debug/debug.h"


/* constant char delimiters - used in read routine */
enum {
  ASSIGN_DELIM = '=',
  COMMENT_DELIM = '#'
};


mdio_Config *mdio_createConfig(void)
{
  mdio_Config *p;
  p = (mdio_Config *) malloc(sizeof(mdio_Config));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (mdio_initializeConfig(p)) {
    free(p);
    return NULL;
  }
  return p;
}


int mdio_initializeConfig(mdio_Config *p)
{
  ASSERT(p != NULL);
  memset(p, 0, sizeof(mdio_Config));
  if (mdio_initializeFile(&(p->file))) {
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->configdata), sizeof(mdio_ConfigData), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"configdata\" list");
    return MDIO_ERROR;
  }
  return 0;
}


void mdio_destroyConfig(mdio_Config *p)
{
  ASSERT(p != NULL);
  mdio_cleanupConfig(p);
  free(p);
}


void mdio_cleanupConfig(mdio_Config *p)
{
  int len, k;
  mdio_ConfigData *data;

  ASSERT(p != NULL);

  /* free allocated strings */
  len = adt_getLengthList(&(p->configdata));
  data = (mdio_ConfigData *) adt_getDataList(&(p->configdata));
  for (k = 0;  k < len;  k++) {
    free((void *) data[k].keyword);
    free((void *) data[k].value);
  }

  adt_cleanupList(&(p->configdata));
  mdio_cleanupFile(&(p->file));
}


int mdio_readConfig(mdio_Config *p, const char *name)
{
  char *buf = NULL;
  char *endkey, *val, *endbuf;
  int buflen = 0;
  int retval = 0;
  mdio_ConfigData cfg;

  ASSERT(p != NULL);
  ASSERT(name != NULL);

  /* open file */
  if (mdio_openFile(&(p->file), name, MDIO_FILE_TEXT | MDIO_FILE_READ)) {
    return MDIO_ERROR;
  }

  /* read each line */
  while ((retval = mdio_readLineFile(&(p->file), &buf, &buflen)) > 0) {

    /* find end of buffer, dropping any comments */
    endbuf = strchr(buf, COMMENT_DELIM);
    if (endbuf == NULL) {
      endbuf = strchr(buf, '\n');
      ASSERT(endbuf != NULL);
    }
    *endbuf = '\n';

    /* remove trailing whitespace */
    while (isspace(*endbuf) && endbuf > buf) endbuf--;
    if (endbuf == buf) {
      continue;  /* empty line, continue loop */
    }
    ++endbuf;
    *endbuf = '\0';

    /* remove leading whitespace in front of keyword */
    while (isspace(*buf)) buf++;

    /* find end of keyword */
    endkey = buf;
    while (!isspace(*endkey) && *endkey != ASSIGN_DELIM && endkey < endbuf) {
      endkey++;
    }

    /* remove leading whitespace and optional assign delim before value */
    val = endkey;
    while (isspace(*val) && val < endbuf) val++;
    if (*val == ASSIGN_DELIM) val++;
    while (isspace(*val) && val < endbuf) val++;

    /* mark end of keyword */
    *endkey = '\0';

    /* append keyword and value to list */
    ASSERT(buf != NULL);
    ASSERT(val != NULL);
    cfg.keyword = (const char *) strdup(buf);
    cfg.value = (const char *) strdup(val);
    if (cfg.keyword == NULL || cfg.value == NULL) {
      free((void *) cfg.keyword);
      free((void *) cfg.value);
      ERRMSG("out of memory");
      mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
      mdio_closeFile(&(p->file));  /* close file before returning */
      return MDIO_ERROR;
    }
    cfg.filename = name;
    cfg.linenum = mdio_getLinenumFile(&(p->file));
    if (adt_appendList(&(p->configdata), &cfg)) {
      mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
      mdio_closeFile(&(p->file));  /* close file before returning */
      return MDIO_ERROR;
    }

  } /* end read line loop */

  /* close file */
  if (mdio_closeFile(&(p->file))) {
    return MDIO_ERROR;
  }

  ASSERT(retval == 0 || retval == MDIO_ERROR);
  return retval;  /* return error status */
}


mdio_ConfigData *mdio_getDataConfig(mdio_Config *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);

  *nelems = adt_getLengthList(&(p->configdata));
  return (mdio_ConfigData *) adt_getDataList(&(p->configdata));
}
