/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * bincoord.c - read and write binary coordinate files
 */

#include <stdlib.h>
#include <string.h>
#include "mdapi/mdtypes.h"
#include "mdio/bincoord.h"
#include "debug/debug.h"

/* prototypes for internal functions */
static int bin_read(mdio_Bincoord *p, int32 nexpect);
static int bin_write(mdio_Bincoord *p);


mdio_Bincoord *mdio_createBincoord(void)
{
  mdio_Bincoord *p;
  p = (mdio_Bincoord *) malloc(sizeof(mdio_Bincoord));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (mdio_initializeBincoord(p)) {
    free(p);
    return NULL;
  }
  return p;
}


int mdio_initializeBincoord(mdio_Bincoord *p)
{
  ASSERT(p != NULL);

  /* check required data size */
  if (sizeof(int32) != 4) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_BADVAL,
        "bincoord reading and writing requires 4-byte integers");
    return MDIO_ERROR;
  }

  memset(p, 0, sizeof(mdio_Bincoord));   /* zero memory */

  if (mdio_initializeFile(&(p->file))) {
    return MDIO_ERROR;
  }
  if (adt_initializeArray(&(p->dvec), sizeof(MD_Dvec), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"dvec\" array");
    return MDIO_ERROR;
  }
  return 0;
}


void mdio_destroyBincoord(mdio_Bincoord *p)
{
  ASSERT(p != NULL);
  mdio_cleanupBincoord(p);
  free(p);
}


void mdio_cleanupBincoord(mdio_Bincoord *p)
{
  adt_cleanupArray(&(p->dvec));
  mdio_cleanupFile(&(p->file));
}


int mdio_readBincoord(mdio_Bincoord *p, const char *name, int n_expect)
{
  ASSERT(p != NULL);
  ASSERT(name != NULL);
  ASSERT(n_expect >= 0);

  /* open file */
  if (mdio_openFile(&(p->file), name, MDIO_FILE_BINARY | MDIO_FILE_READ)) {
    return MDIO_ERROR;
  }

  /* read file */
  if (bin_read(p, (int32) n_expect)) {
    mdio_closeFile(&(p->file));
    return MDIO_ERROR;
  }

  /* close file */
  if (mdio_closeFile(&(p->file))) {
    return MDIO_ERROR;
  }
  return 0;
}


MD_Dvec *mdio_getBincoord(mdio_Bincoord *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthArray(&(p->dvec));
  return (MD_Dvec *) adt_getDataArray(&(p->dvec));
}


int mdio_setBincoord(mdio_Bincoord *p, MD_Dvec *dvec, int nelems)
{
  ASSERT(p != NULL);

  /* trap for an easy bug */
  if (adt_getDataArray(&(p->dvec)) == dvec) {
    BUG("cannot set coordinates using memory that Bincoord object owns");
  }

  /* destroy existing array and reconstruct with data provided */
  adt_cleanupArray(&(p->dvec));
  if (adt_initializeArray(&(p->dvec), sizeof(MD_Dvec), nelems, dvec)) {
    mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}


int mdio_writeBincoord(mdio_Bincoord *p, const char *name)
{
  ASSERT(p != NULL);
  ASSERT(name != NULL);

  /* open file */
  if (mdio_openFile(&(p->file), name, MDIO_FILE_BINARY | MDIO_FILE_WRITE)) {
    return MDIO_ERROR;
  }

  /* write file */
  if (bin_write(p)) {
    mdio_closeFile(&(p->file));
    return MDIO_ERROR;
  }

  /* close file */
  if (mdio_closeFile(&(p->file))) {
    return MDIO_ERROR;
  }
  return 0;
}


/******************************************************************************
 *
 * internal routines to perform actual file reading and writing
 *
 ******************************************************************************/

/* in-place byte swapping */
#define  SWAP(a,b)    (b)^=(a); (a)^=(b); (b)^=(a)
#define  SWAP_INT(b)  SWAP((b)[0],(b)[3]); SWAP((b)[1],(b)[2])
#define  SWAP_DBL(b)  SWAP((b)[0],(b)[7]); SWAP((b)[1],(b)[6]); \
                      SWAP((b)[2],(b)[5]); SWAP((b)[3],(b)[4])

/* call these macros with type (int *) and (double *), respectively */
#define  BYTE_FLIP_INT(pi)  \
{ char *pc = (char *)(pi);  SWAP_INT(pc); }

#define  BYTE_FLIP_DOUBLE(pd)  \
{ char *pc = (char *)(pd);  SWAP_DBL(pc); }


/*
 * use "nexpect > 0" as our "magic number" to determine whether or not
 * we need byte flipping
 *
 * otherwise ("nexpect == 0") read in NAMD binary file format:
 * int*1 (natoms), double*3*natoms (atom coordinates)
 */
int bin_read(mdio_Bincoord *p, int32 nexpect)
{
  mdio_File *f;
  MD_Dvec *dvec;
  int32 natoms, nelems;
  char c;
  char s[80];

  ASSERT(p != NULL);
  f = &(p->file);

  /* read number of atoms */
  if ((nelems = (int32) mdio_readBinaryFile(f, &natoms, sizeof(int32), 1))
      != 1) {
    COND(nelems != 1);
    if (mdio_getErrorFile(f) == MDIO_ERROR_NONE) {
      mdio_setErrorMessageFile(f, MDIO_ERROR_READ,
          "expected to read number of atoms");
    }
    return MDIO_ERROR;
  }

  /* see if we need byte flipping */
  if (nexpect > 0 && natoms != nexpect) {
    /* try flipping "natoms" to see if we match number atoms expected */
    BYTE_FLIP_INT(&natoms);
    if (natoms != nexpect) {
      snprintf(s, sizeof(s), 
          "number of atoms (%d) != (%d) expected", natoms, nexpect);
      mdio_setErrorMessageFile(f, MDIO_ERROR_READ, s);
      return MDIO_ERROR;
    }
  }
  else {
    /* _assume_ we don't need to flip, use "nexpect" as "flip flag" */
    nexpect = 0;
  }

  /* validate number of atoms (to whatever extent possible) */
  if (natoms < 0) {
    snprintf(s, sizeof(s), "invalid number (%d) of atoms", natoms);
    mdio_setErrorMessageFile(f, MDIO_ERROR_READ, s);
    return MDIO_ERROR;
  }

  /* resize array based on expected length */
  if (adt_resizeArray(&(p->dvec), natoms)) {
    mdio_setErrorMessageFile(f, MDIO_ERROR_NOMEM,
        "cannot resize \"dvec\" array");
    return MDIO_ERROR;
  }

  /* read file */
  dvec = adt_getDataArray(&(p->dvec));
  if ((nelems = (int32) mdio_readBinaryFile(f, dvec, sizeof(MD_Dvec), natoms))
      != natoms) {
    if (mdio_isEndOfFile(f)) {
      COND(mdio_isEndOfFile(f));
      snprintf(s, sizeof(s),
          "reached end-of-file before reading %d coordinates", natoms);
      mdio_setErrorMessageFile(f, MDIO_ERROR_UNXEOF, s);
    }
    COND(nelems != natoms);
    return MDIO_ERROR;
  }
  else if (nexpect) {
    /* flip atom coordinates in place */
    double *d = (double *) dvec;
    int k;
    for (k = 0;  k < natoms;  k++) {
      BYTE_FLIP_DOUBLE(d++);
      BYTE_FLIP_DOUBLE(d++);
      BYTE_FLIP_DOUBLE(d++);
    }
  }

  /* find EOF */
  if ((nelems = mdio_readBinaryFile(f, &c, sizeof(char), 1)) != 0
      || !mdio_isEndOfFile(f)) {
    COND(nelems != 0);
    COND(!mdio_isEndOfFile(f));
    mdio_setErrorMessageFile(f, MDIO_ERROR_READ,
        "extra information found in file");
    return MDIO_ERROR;
  }
  return 0;
}


int bin_write(mdio_Bincoord *p)
{
  mdio_File *f;
  MD_Dvec *dvec;
  int natoms;

  ASSERT(p != NULL);
  f = &(p->file);
  dvec = adt_getDataArray(&(p->dvec));
  natoms = adt_getLengthArray(&(p->dvec));
  ASSERT(natoms >= 0);

  /* write number of atoms */
  if (mdio_writeBinaryFile(f, &natoms, sizeof(int), 1) < 1) {
    return MDIO_ERROR;
  }

  /* write coordinate array */
  if (mdio_writeBinaryFile(f, dvec, sizeof(MD_Dvec), natoms) < natoms) {
    return MDIO_ERROR;
  }
  return 0;
}
