/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mdtypes.c - function definitions for handling MDAPI data types
 */

#include <stdlib.h>
#include <string.h>
#include "adt/list.h"
#include "adt/table.h"
#include "mdapi/mddef.h"
#include "debug/debug.h"


#define NELEMS(a)  (sizeof(a) / sizeof(a[0]))

/*
 * type encoding
 */

/* internal type used to initialize MDAPI predefined data types */
typedef struct TypeInit_tag {
  const int32 typnum;         /* for verification */
  const char *typname;
  const MD_Member member[6];  /* max member list length below is 6 */
  const int32 memberlen;
  const int32 numbytes;
} TypeInit;

static const TypeInit TypeInitList[] = {
  { MD_CHAR, "char",
    { { MD_CHAR, 1, "" } }, 1, MD_SIZEOF(MD_CHAR) },
  { MD_INT32, "int32",
    { { MD_INT32, 1, "" } }, 1, MD_SIZEOF(MD_INT32) },
  { MD_FLOAT, "float",
    { { MD_FLOAT, 1, "" } }, 1, MD_SIZEOF(MD_FLOAT) },
  { MD_DOUBLE, "double",
    { { MD_DOUBLE, 1, "" } }, 1, MD_SIZEOF(MD_DOUBLE) },
  { MD_FVEC, "MD_Fvec",
    { { MD_FVEC, 1, "" } }, 1, MD_SIZEOF(MD_FVEC) },
  { MD_DVEC, "MD_Dvec",
    { { MD_DVEC, 1, "" } }, 1, MD_SIZEOF(MD_DVEC) },
  { MD_NAME, "MD_Name",
    { { MD_NAME, 1, "" } }, 1, MD_SIZEOF(MD_NAME) },
  { MD_STRING, "MD_String",
    { { MD_STRING, 1, "" } }, 1, MD_SIZEOF(MD_STRING) },
  { MD_MESSAGE, "MD_Message",
    { { MD_MESSAGE, 1, "" } }, 1, MD_SIZEOF(MD_MESSAGE) },
  { MD_ATOMPRM, "MD_AtomPrm",
    { { MD_DOUBLE, 1, "emin" },
      { MD_DOUBLE, 1, "rmin" },
      { MD_DOUBLE, 1, "emin14" },
      { MD_DOUBLE, 1, "rmin14" },
      { MD_NAME, 1, "type" } }, 5, MD_SIZEOF(MD_ATOMPRM) },
  { MD_BONDPRM, "MD_BondPrm",
    { { MD_DOUBLE, 1, "k" },
      { MD_DOUBLE, 1, "r0" },
      { MD_NAME, 2, "type" } }, 3, MD_SIZEOF(MD_BONDPRM) },
  { MD_ANGLEPRM, "MD_AnglePrm",
    { { MD_DOUBLE, 1, "k_theta" },
      { MD_DOUBLE, 1, "theta0" },
      { MD_DOUBLE, 1, "k_ub" },
      { MD_DOUBLE, 1, "r_ub" },
      { MD_NAME, 3, "type" } }, 5, MD_SIZEOF(MD_ANGLEPRM) },
  { MD_TORSPRM, "MD_TorsPrm",
    { { MD_DOUBLE, 1, "k_tor" },
      { MD_DOUBLE, 1, "phi" },
      { MD_INT32, 1, "n" },
      { MD_INT32, 1, "mult" },
      { MD_NAME, 4, "type" } }, 5, MD_SIZEOF(MD_TORSPRM) },
  { MD_NBFIXPRM, "MD_NbfixPrm",
    { { MD_DOUBLE, 1, "emin" },
      { MD_DOUBLE, 1, "rmin" },
      { MD_DOUBLE, 1, "emin14" },
      { MD_DOUBLE, 1, "rmin14" },
      { MD_INT32, 2, "prm" },
      { MD_NAME, 2, "type" } }, 6, MD_SIZEOF(MD_NBFIXPRM) },
  { MD_ATOM, "MD_Atom",
    { { MD_DOUBLE, 1, "m" },
      { MD_DOUBLE, 1, "q" },
      { MD_INT32, 1, "prm" },
      { MD_INT32, 1, "notused" },
      { MD_NAME, 1, "name" },
      { MD_NAME, 1, "type" } }, 6, MD_SIZEOF(MD_ATOM) },
  { MD_BOND, "MD_Bond",
    { { MD_INT32, 2, "atom" },
      { MD_INT32, 1, "prm" } }, 2, MD_SIZEOF(MD_BOND) },
  { MD_ANGLE, "MD_Angle",
    { { MD_INT32, 3, "atom" },
      { MD_INT32, 1, "prm" } }, 2, MD_SIZEOF(MD_ANGLE) },
  { MD_TORS, "MD_Tors",
    { { MD_INT32, 4, "atom" },
      { MD_INT32, 1, "prm" } }, 2, MD_SIZEOF(MD_TORS) },
  { MD_EXCL, "MD_Excl",
    { { MD_INT32, 2, "atom" } }, 1, MD_SIZEOF(MD_EXCL) },
};


/*
 * front end API methods
 */

int32 MD_type(MD_Engine *s, const char *name)
{
  int32 typnum;

  ASSERT(s != NULL);
  ASSERT(name != NULL);
  typnum = adt_lookupTable(&(s->typnum), name);
  if (typnum == ADT_ERROR) {
    COND(typnum == ADT_ERROR);
    return MD_error(s, MD_ERR_NAME);
  }
  return typnum;
}


const char *MD_type_name(MD_Engine *s, int32 type)
{
  adt_List *list;
  TypeInfo *typinfo;
  int32 len, index;


  ASSERT(s != NULL);
  list = &(s->typinfo);
  len = (int32) adt_getLengthList(list);
  ASSERT(adt_getLengthList(&(s->typname)) == len);
  typinfo = (TypeInfo *) adt_getDataList(list);
  index = type >> MD_SHIFT_TYPE;
  /* check validity of type number */
  if (index >= len || index < 0 || type != typinfo[index].typnum) {
    COND(index >= len);
    COND(index < 0);
    COND(type != typinfo[index].typnum);
    MD_error(s, MD_ERR_TYPENUM);
    return NULL;
  }
  return typinfo[index].typname;
}


const char **MD_type_namelist(MD_Engine *s, int32 *listlen)
{
  ASSERT(s != NULL);
  ASSERT(listlen != NULL);
  *listlen = adt_getLengthList(&(s->typname));
  return (const char **) adt_getDataList(&(s->typname));
}


const MD_Member *MD_type_memberlist(MD_Engine *s, int32 type, int32 *listlen)
{
  adt_List *list;
  TypeInfo *typinfo;
  int32 len, index;

  ASSERT(s != NULL);
  ASSERT(listlen != NULL);
  list = &(s->typinfo);
  len = (int32) adt_getLengthList(list);
  typinfo = (TypeInfo *) adt_getDataList(list);
  index = type >> MD_SHIFT_TYPE;
  /* check validity of type number */
  if (index >= len || index < 0 || type != typinfo[index].typnum) {
    COND(index >= s->typinfo.len);
    COND(index < 0);
    COND(type != typinfo[index].typnum);
    MD_error(s, MD_ERR_TYPENUM);
    return NULL;
  }
  *listlen = typinfo[index].memberlistlen;
  return (const MD_Member *) typinfo[index].memberlist;
}


void *MD_type_member(MD_Engine *s, int32 type, const void *obj,
    const char *member_name, MD_Member *pmember)
{
  adt_List *list;
  TypeInfo *typinfo;
  int32 len, index, member_index;

  ASSERT(s != NULL);
  /* ASSERT(obj != NULL); */
  ASSERT(member_name != NULL);
  list = &(s->typinfo);
  len = (int32) adt_getLengthList(list);
  typinfo = (TypeInfo *) adt_getDataList(list);
  index = type >> MD_SHIFT_TYPE;
  /* check validity of type number */
  if (index >= len || index < 0 || type != typinfo[index].typnum) {
    COND(index >= s->typinfo.len);
    COND(index < 0);
    COND(type != typinfo[index].typnum);
    MD_error(s, MD_ERR_TYPENUM);
    return NULL;
  }
  /* create member table if it does not exist */
  if (typinfo[index].member == NULL) {
    int32 offset = 0;  /* offset in bytes from start of object type */
    int32 mlistlen = typinfo[index].memberlistlen;
    int32 k, val;
    const MD_Member *mlist = typinfo[index].memberlist;
    adt_Table *t = adt_createTable(0);
    int32 *offsetlist = (int32 *) calloc(mlistlen, sizeof(int32));
    ASSERT(mlist != NULL);
    if (t == NULL) {
      ERRCOND(t == NULL);
      MD_error(s, MD_ERR_MEMALLOC);
      return NULL;
    }
    if (offsetlist == NULL) {
      ERRCOND(offsetlist == NULL);
      MD_error(s, MD_ERR_MEMALLOC);
      return NULL;
    }
    for (k = 0;  k < mlistlen;  k++) {
      /* store member index in hash table to be accessed by member name */
      if ((val = adt_insertTable(t, mlist[k].name, k)) != k) {
        if (val == ADT_ERROR) {
          COND(val == ADT_ERROR);
          MD_error(s, MD_ERR_MEMALLOC);
          return NULL;
        }
        /* two members have the same name, this is a type creation error! */
        COND(val != k);
        MD_error(s, MD_ERR_NEWTYPE);
        return NULL;
      }
      offsetlist[k] = offset;
      offset += mlist[k].len * MD_SIZEOF(mlist[k].type);
    }
    /* all member type lengths must sum to length of containing type */
    ASSERT(offset == MD_SIZEOF(type));
    /* store newly created member table and byte offset list */
    typinfo[index].member = t;
    typinfo[index].byteoffset = offsetlist;
  }
  /* hash table lookup of member name to find index in member list */
  member_index = adt_lookupTable(typinfo[index].member, member_name);
  if (member_index == ADT_ERROR) {
    /* COND(member_index == ADT_ERROR); */
    MD_error(s, MD_ERR_NAME);
    return NULL;
  }
  ASSERT(member_index < typinfo[index].memberlistlen);
  /* if pmember points to variable, store MD_Member data for this member */
  if (pmember != NULL) {
    *pmember = typinfo[index].memberlist[member_index];
  }
  /* return byte offset from obj based on table value */
  return (void*)((const char *)obj + typinfo[index].byteoffset[member_index]);
}


/*
 * engine API method
 */

int32 MD_new_type(MD_Front *s, const char *name,
    const MD_Member *member, int32 memberlen, int32 nbytes)
{
  adt_List flipq;       /* place flipoffset info into queue */
  adt_List stack;       /* recursively expand typnum of members on stack */
  int32 *elem;          /* for new element in stack or queue */
  TypeInfo typinfo;     /* new element for typinfo list */
  const MD_Member *tm;  /* member list for previously defined type */
  int32 tmlen;          /* length of member list */
  int32 flipflag;       /* will get flip flag for new type */
  int32 index;          /* index of type in typinfo array */
  int32 j, k;           /* to step through member list */
  int32 offset;         /* used to count bytes from member types */
  int32 last;           /* end of last offset location needing flipping */
  int32 alignmask;      /* keep track of word alignment */
  int32 n, t;
  adt_List *typinfo_list, *typname_list;
  int32 typinfo_listlen, typname_listlen;
  TypeInfo *typinfo_listdata;
  int32 *flipq_listdata = NULL;  /* by default assume empty flipq list */
  int32 flipq_listlen = 0;

  ASSERT(s != NULL);
  ASSERT(name != NULL);
  ASSERT(member != NULL);
  ASSERT(memberlen > 0);
  ASSERT(nbytes > 0 && nbytes <= MD_SIZEOF_MASK);
  typinfo_list = &(s->typinfo);
  typinfo_listlen = (int32) adt_getLengthList(typinfo_list);
  typname_list = &(s->typname);
  typname_listlen = (int32) adt_getLengthList(typname_list);
  /*
   * the first number of types added will be primary types
   * (if so, treat as special case)
   */
  if (typinfo_listlen < MD_NUMBER_PRIMARY_TYPES) {
    ASSERT(memberlen == 1);
    ASSERT(member[0].len == 1);
    ASSERT(strcmp(member[0].name, "") == 0);
    ASSERT(MD_SIZEOF(member[0].type) == nbytes);
    index = typinfo_listlen;
    flipflag = MD_FLIP_MASK & member->type;
    ASSERT(flipflag != MD_FLIP_DERIVED);
    t = (index << MD_SHIFT_TYPE) | flipflag | nbytes;
    /* place (name,t) into typnum table */
    if ((k = adt_insertTable(&(s->typnum), name, t)) == ADT_ERROR) {
      COND(k == ADT_ERROR);
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    else if (k != t) {
      /* conflicting type name */
      COND(k != t);
      return MD_error(s, MD_ERR_NEWTYPE);
    }
    /* append name to typname list */
    ASSERT(typinfo_listlen == typname_listlen);
    if (adt_appendList(typname_list, &name)) {
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    typinfo.typnum = t;
    typinfo.typname = name;
    typinfo.memberlist = member;
    typinfo.memberlistlen = memberlen;
    typinfo.member = NULL;  /* member hash table may be created later */
    typinfo.byteoffset = NULL;  /* member byte offset may be created later */
    typinfo.flipoffset = NULL;  /* need flipoffset only for derived types */
    typinfo.flipoffsetlen = 0;
    /* append to typinfo list */
    if (adt_appendList(typinfo_list, &typinfo)) {
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    ASSERT(typinfo_listlen == typname_listlen);
    return typinfo.typnum;
  }
  /*
   * otherwise we are attempting to add a derived type;
   * expand member array into type array having only primary types
   *
   * flipq will gather what will become flipoffset array
   *
   * flipoffset entries contain increment to next set of bytes that need to
   * be flipped (i.e. when communicating between different endian machines)
   * ORed with flip type (4-byte, 8-byte, 12-byte, 24-byte);
   * the inclusion of 12-byte and 24-byte allows us to operate
   * on MD_Fvec and MD_Dvec types more efficiently
   *
   * the last flipoffset entry contains increment to end of element
   * (to do fast flipping of data arrays of types encoded here)
   *
   * while constructing flipoffset, we are also verifying word alignment
   * of the given type; require that 4-byte (12-byte) subtypes be aligned
   * on 4-byte boundaries and 8-byte (24-byte) on 8-byte boundaries
   */
  typinfo_listdata = (TypeInfo *) adt_getDataList(typinfo_list);
  if (adt_initializeList(&flipq, sizeof(int32), 0, NULL)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  if (adt_initializeList(&stack, sizeof(int32), 0, NULL)) {
    adt_cleanupList(&flipq);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  /* push types from member array backwards onto stack */
  for (k = memberlen - 1;  k >= 0;  k--) {
    index = member[k].type >> MD_SHIFT_TYPE;
    /* check validity of type number */
    if (index >= typinfo_listlen || index < 0
        || member[k].type != typinfo_listdata[index].typnum) {
      COND(index >= typinfo_listlen);
      COND(index < 0);
      COND(member[k].type != typinfo_listdata[index].typnum);
      adt_cleanupList(&stack);
      adt_cleanupList(&flipq);
      return MD_error(s, MD_ERR_TYPENUM);
    }
    ASSERT(member[k].len > 0);
    /* push it as many times as indicated by len */
    for (j = 0;  j < member[k].len;  j++) {
      if (adt_appendList(&stack, (void *) &(member[k].type))) {
        adt_cleanupList(&stack);
        adt_cleanupList(&flipq);
        return MD_error(s, MD_ERR_MEMALLOC);
      }
    }
  }
  offset = last = 0;
  alignmask = 0;
  while (adt_getLengthList(&stack) > 0) {
    /* pop type off of stack */
    if ((elem = (int32 *) adt_indexList(&stack,
            adt_getLengthList(&stack) - 1)) == NULL) {
      adt_cleanupList(&stack);
      adt_cleanupList(&flipq);
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    t = *elem;
    if (adt_deleteList(&stack)) {
      adt_cleanupList(&stack);
      adt_cleanupList(&flipq);
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    switch (t & MD_FLIP_MASK) {
      case MD_FLIP_NONE:
        offset += MD_SIZEOF(t);
        break;
      case MD_FLIP_8BYTE:
      case MD_FLIP_24BYTE:
        alignmask = 0x07;
        if (offset & 0x07) {
          /* type is not word aligned! */
          COND(offset & 0x07);
          adt_cleanupList(&stack);
          adt_cleanupList(&flipq);
          return MD_error(s, MD_ERR_NEWTYPE);
        }
      case MD_FLIP_4BYTE:
      case MD_FLIP_12BYTE:
        if (alignmask < 0x03) alignmask = 0x03;
        if (offset & 0x03) {
          /* type is not word aligned! */
          COND(offset & 0x03);
          adt_cleanupList(&stack);
          adt_cleanupList(&flipq);
          return MD_error(s, MD_ERR_NEWTYPE);
        }
        /* store flip info into queue */
        n = ((t & MD_FLIP_MASK) | (offset - last));
        if (adt_appendList(&flipq, &n)) {
          adt_cleanupList(&stack);
          adt_cleanupList(&flipq);
          return MD_error(s, MD_ERR_MEMALLOC);
        }
        last = offset;
        offset += MD_SIZEOF(t);
        break;
      case MD_FLIP_DERIVED:
        /* push types from member array of derived type backwards onto stack */
        /* (we don't have to re-validate a type that was previously defined) */
        tm = typinfo_listdata[t >> MD_SHIFT_TYPE].memberlist;
        tmlen = typinfo_listdata[t >> MD_SHIFT_TYPE].memberlistlen;
        for (k = tmlen - 1;  k >= 0;  k--) {
          for (j = 0;  j < tm[k].len;  j++) {
            if (adt_appendList(&stack, (void *) &(tm[k].type))) {
              adt_cleanupList(&stack);
              adt_cleanupList(&flipq);
              return MD_error(s, MD_ERR_MEMALLOC);
            }
          }
        }
        break;
      default:
        BUG("Invalid flip flag value");
    }
  }
  /* done with stack, free its data array */
  adt_cleanupList(&stack);
  /*
   * check to make sure byte offset matches predicted number of bytes
   * and check that total number of bytes has correct word alignment
   */
  if (offset != nbytes || (offset & alignmask)) {
    COND(offset != nbytes);
    COND(offset & alignmask);
    adt_cleanupList(&flipq);
    return MD_error(s, MD_ERR_NEWTYPE);
  }
  /* check ranges imposed by type encoding using int32 */
  if (nbytes > 0xFFFF || typinfo_listlen == 0x0FFF) {
    /* it is doubtful these ranges will ever be exceeded */
    COND(nbytes > 0xFFFF);
    COND(s->typinfo.len == 0x0FFF);
    adt_cleanupList(&flipq);
    return MD_error(s, MD_ERR_NEWTYPE);
  } 
  /* we have a valid type member list */
  if (adt_getLengthList(&flipq) == 0) {
    flipflag = MD_FLIP_NONE;
    /* in this case, we can free flipoffset list */
    adt_cleanupList(&flipq);
  }
  else if (memberlen == 1 && member[0].len == 1
      && (member[0].type & MD_FLIP_MASK) != MD_FLIP_DERIVED) {
    flipflag = (member[0].type & MD_FLIP_MASK);
    /* in this case, we can free flipoffset list */
    adt_cleanupList(&flipq);
  }
  else {
    flipflag = MD_FLIP_DERIVED;
    /* append to flipq number of remaining bytes in type */
    n = offset - last;
    if (adt_appendList(&flipq, &n)) {
      adt_cleanupList(&flipq);
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    /* flipq list is nonempty, obtain its data array and length */
    flipq_listdata = (int32 *) adt_getDataList(&flipq);
    flipq_listlen = (int32) adt_getLengthList(&flipq);
    /* in this case don't cleanup flipq, instead let data array persist */
  }
  /* get next index for typinfo array */
  index = typinfo_listlen;
  /* compute new type number */
  t = (index << MD_SHIFT_TYPE) | flipflag | nbytes;
  /* place (name,t) into typnum table */
  if ((k = adt_insertTable(&(s->typnum), name, t)) == ADT_ERROR) {
    COND(k == ADT_ERROR);
    adt_cleanupList(&flipq);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  else if (k != t) {
    /* conflicting type name */
    COND(k != t);
    adt_cleanupList(&flipq);
    return MD_error(s, MD_ERR_NEWTYPE);
  }
  /* append name to typname list */
  ASSERT(typinfo_listlen == typname_listlen);
  if (adt_appendList(typname_list, &name)) {
    adt_cleanupList(&flipq);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  typinfo.typnum = t;
  typinfo.typname = name;
  typinfo.memberlist = member;
  typinfo.memberlistlen = memberlen;
  typinfo.member = NULL;  /* member hash table may be created later */
  typinfo.byteoffset = NULL;  /* member byte offset may be created later */
  typinfo.flipoffset = flipq_listdata;
  typinfo.flipoffsetlen = flipq_listlen;
  /* append to typinfo list */
  if (adt_appendList(typinfo_list, &typinfo)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  ASSERT(typinfo_listlen == typname_listlen);
  return typinfo.typnum;
}


/*
 * internal methods
 */

int32 MD_init_types(MD_Interface *s)
{
  int32 k, t;

  ASSERT(s != NULL);
  /* init type handling containers */
  if (adt_initializeTable(&(s->typnum), 0)
      || adt_initializeList(&(s->typname), sizeof(const char *), 0, NULL)
      || adt_initializeList(&(s->typinfo), sizeof(TypeInfo), 0, NULL)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  /* put predefined data types into containers */
  for (k = 0;  k < NELEMS(TypeInitList);  k++) {
    t = MD_new_type(s, TypeInitList[k].typname, TypeInitList[k].member,
        TypeInitList[k].memberlen, TypeInitList[k].numbytes);
    if (t < 0) {
      return MD_FAIL;
    }
    else if (t != TypeInitList[k].typnum) {
      BUG("Invalid type number");
    }
  }
  return 0;
}


void MD_done_types(MD_Interface *s)
{
  TypeInfo *typinfo;
  int32 len, k;

  ASSERT(s != NULL);
  /* need to free memory used by flipoffset arrays in typinfo array */
  typinfo = (TypeInfo *) adt_getDataList(&(s->typinfo));
  len = (int32) adt_getLengthList(&(s->typinfo));
  for (k = 0;  k < len;  k++) {
    free(typinfo[k].flipoffset);
    if (typinfo[k].member) {
      adt_destroyTable(typinfo[k].member);
    }
    free(typinfo[k].byteoffset);
  }
  /* free type handling containers */
  adt_cleanupTable(&(s->typnum));
  adt_cleanupList(&(s->typname));
  adt_cleanupList(&(s->typinfo));
}


int32 MD_flipdata(MD_Interface *s, void *vbuf, int32 nelems, int32 typnum)
{
  TypeInfo *typinfo;
  char *buf = (char *) vbuf;
  int32 index = typnum >> MD_SHIFT_TYPE;
  int32 *flipoffset;
  int32 flipoffsetlen;
  int32 len, i, j;
  char tmp;

  ASSERT(s != NULL);
  ASSERT(vbuf != NULL);
  /* check validity of type number */
  typinfo = (TypeInfo *) adt_getDataList(&(s->typinfo));
  len = (int32) adt_getLengthList(&(s->typinfo));
  if (index >= len || index < 0 || typnum != typinfo[index].typnum) {
    COND(index >= len);
    COND(index < 0);
    COND(typnum != typinfo[index].typnum);
    return MD_error(s, MD_ERR_TYPENUM);
  }
  flipoffset = typinfo[index].flipoffset;
  /* set array length to one less (see increment after j loop below) */
  flipoffsetlen = typinfo[index].flipoffsetlen - 1;

  switch (typnum & MD_FLIP_MASK) {
    case MD_FLIP_NONE:
      break;
    case MD_FLIP_4BYTE:
      ASSERT(MD_SIZEOF(typnum) == 4);
      for (i = 0;  i < nelems;  i++) {
        tmp=buf[0]; buf[0]=buf[3]; buf[3]=tmp;
        tmp=buf[1]; buf[1]=buf[2]; buf[2]=tmp;
        buf += 4;
      }
      break;
    case MD_FLIP_8BYTE:
      ASSERT(MD_SIZEOF(typnum) == 8);
      for (i = 0;  i < nelems;  i++) {
        tmp=buf[0]; buf[0]=buf[7]; buf[7]=tmp;
        tmp=buf[1]; buf[1]=buf[6]; buf[6]=tmp;
        tmp=buf[2]; buf[2]=buf[5]; buf[5]=tmp;
        tmp=buf[3]; buf[3]=buf[4]; buf[4]=tmp;
        buf += 8;
      }
      break;
    case MD_FLIP_12BYTE:
      ASSERT(MD_SIZEOF(typnum) == 12);
      for (i = 0;  i < nelems;  i++) {
        tmp=buf[0]; buf[0]=buf[3]; buf[3]=tmp;
        tmp=buf[1]; buf[1]=buf[2]; buf[2]=tmp;
        tmp=buf[4]; buf[4]=buf[7]; buf[7]=tmp;
        tmp=buf[5]; buf[5]=buf[6]; buf[6]=tmp;
        tmp=buf[8]; buf[8]=buf[11]; buf[11]=tmp;
        tmp=buf[9]; buf[9]=buf[10]; buf[10]=tmp;
        buf += 12;
      }
      break;
    case MD_FLIP_24BYTE:
      ASSERT(MD_SIZEOF(typnum) == 24);
      for (i = 0;  i < nelems;  i++) {
        tmp=buf[0]; buf[0]=buf[7]; buf[7]=tmp;
        tmp=buf[1]; buf[1]=buf[6]; buf[6]=tmp;
        tmp=buf[2]; buf[2]=buf[5]; buf[5]=tmp;
        tmp=buf[3]; buf[3]=buf[4]; buf[4]=tmp;
        tmp=buf[8]; buf[8]=buf[15]; buf[15]=tmp;
        tmp=buf[9]; buf[9]=buf[14]; buf[14]=tmp;
        tmp=buf[10]; buf[10]=buf[13]; buf[13]=tmp;
        tmp=buf[11]; buf[11]=buf[12]; buf[12]=tmp;
        tmp=buf[16]; buf[16]=buf[23]; buf[23]=tmp;
        tmp=buf[17]; buf[17]=buf[22]; buf[22]=tmp;
        tmp=buf[18]; buf[18]=buf[21]; buf[21]=tmp;
        tmp=buf[19]; buf[19]=buf[20]; buf[20]=tmp;
        buf += 24;
      }
      break;
    case MD_FLIP_DERIVED:
      for (i = 0;  i < nelems;  i++) {
        for (j = 0;  j < flipoffsetlen;  j++) {
          buf += (flipoffset[j] & MD_SIZEOF_MASK);
          switch (flipoffset[j] & MD_FLIP_MASK) {
            case MD_FLIP_4BYTE:
              tmp=buf[0]; buf[0]=buf[3]; buf[3]=tmp;
              tmp=buf[1]; buf[1]=buf[2]; buf[2]=tmp;
              break;
            case MD_FLIP_8BYTE:
              tmp=buf[0]; buf[0]=buf[7]; buf[7]=tmp;
              tmp=buf[1]; buf[1]=buf[6]; buf[6]=tmp;
              tmp=buf[2]; buf[2]=buf[5]; buf[5]=tmp;
              tmp=buf[3]; buf[3]=buf[4]; buf[4]=tmp;
              break;
            case MD_FLIP_12BYTE:
              tmp=buf[0]; buf[0]=buf[3]; buf[3]=tmp;
              tmp=buf[1]; buf[1]=buf[2]; buf[2]=tmp;
              tmp=buf[4]; buf[4]=buf[7]; buf[7]=tmp;
              tmp=buf[5]; buf[5]=buf[6]; buf[6]=tmp;
              tmp=buf[8]; buf[8]=buf[11]; buf[11]=tmp;
              tmp=buf[9]; buf[9]=buf[10]; buf[10]=tmp;
              break;
            case MD_FLIP_24BYTE:
              tmp=buf[0]; buf[0]=buf[7]; buf[7]=tmp;
              tmp=buf[1]; buf[1]=buf[6]; buf[6]=tmp;
              tmp=buf[2]; buf[2]=buf[5]; buf[5]=tmp;
              tmp=buf[3]; buf[3]=buf[4]; buf[4]=tmp;
              tmp=buf[8]; buf[8]=buf[15]; buf[15]=tmp;
              tmp=buf[9]; buf[9]=buf[14]; buf[14]=tmp;
              tmp=buf[10]; buf[10]=buf[13]; buf[13]=tmp;
              tmp=buf[11]; buf[11]=buf[12]; buf[12]=tmp;
              tmp=buf[16]; buf[16]=buf[23]; buf[23]=tmp;
              tmp=buf[17]; buf[17]=buf[22]; buf[22]=tmp;
              tmp=buf[18]; buf[18]=buf[21]; buf[21]=tmp;
              tmp=buf[19]; buf[19]=buf[20]; buf[20]=tmp;
              break;
            default:
              BUG("Invalid flip flag value in flipoffset array");
          }
        } /* end for j loop */
        ASSERT((flipoffset[j] & MD_FLIP_MASK) == 0);
        buf += flipoffset[j];  /* final offset puts us to next element */
      } /* end for i loop */
      break;
    default:
      BUG("Invalid flip flag value defined by type number");
  }
  return 0;
}
