/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mdsetup.c - subsystem for setting up front end and engine APIs
 */

#include <stdlib.h>
#include <string.h>
#include "mdapi/mddef.h"
#include "debug/debug.h"


/*
 * front end API
 */

int32 MD_init_version(MD_Engine *s, const char *mdapi_version,
    const char *engname, int32 flags,
    int (*engine_init)(MD_Engine *, int32 flags),
    void (*engine_done)(MD_Engine *))
{
  int32 retval;

  ASSERT(s != NULL);

  /*
   * check that front end included correct version of header files
   *
   * (later we can make this check everything up to minor version number)
   */
  if (strcmp(MDAPI_VERSION, mdapi_version) != 0) {
    COND(strcmp(MDAPI_VERSION, mdapi_version) != 0);
    return MD_error_version(s);
  }

  /* clear memory */
  memset(s, 0, sizeof(MD_Engine));

  /* setup error handling */
  if (MD_init_error(s)) return MD_FAIL;

  /* setup types */
  if (MD_init_types(s)) return MD_FAIL;

  /* setup data handling */
  if (MD_init_data(s)) return MD_FAIL;

  /* setup callbacks */
  if (MD_init_callback(s)) return MD_FAIL;

  /* setup engine name */
  s->engname = (engname != NULL ? engname : "");

  /*
   * setup engine constructor and destructor
   *
   * (until we support dynamic loading, require init & done be non-NULL)
   */
  ASSERT(engine_init != NULL);
  s->engine_init = engine_init;
  ASSERT(engine_done != NULL);
  s->engine_done = engine_done;

  /* call engine constructor */
  retval = s->engine_init(s, flags);
  if (retval != 0) {
    COND(retval != 0);
    /*
     * MD_ERR_VERSION from engine_init() needs to be changed to MD_ERR_INIT
     * so that engine_done() is still properly called.  Note that the
     * MD_Interface structure being passed is correct size since the
     * front end version check already tested OK.
     *
     * On non-fatal error reset to fatal error MD_ERR_INIT.
     */
    if (s->errnum == MD_ERR_VERSION || !s->error.isfatal) {
      return MD_error(s, MD_ERR_INIT);
    }
    else return MD_FAIL;
  }

  return 0;
}


void MD_done(MD_Engine *s)
{
  ASSERT(s != NULL);

  /* for version error, nothing was initialized */
  if (s->errnum == MD_ERR_VERSION) return;

  /* call engine destructor */
  if (s->engine_done != NULL) s->engine_done(s);

  /* cleanup callbacks */
  MD_done_callback(s);

  /* cleanup data handling */
  MD_done_data(s);

  /* cleanup types */
  MD_done_types(s);

  /* cleanup error handling */
  MD_done_error(s);
}


const char *MD_engine_name(MD_Engine *s)
{
  ASSERT(s != NULL);
  return s->engname;
}


/*
 * engine API
 */

int32 MD_setup_engine_version(MD_Front *s,
    const char *mdapi_version, void *engine)
{
  ASSERT(s != NULL);

  /*
   * check that engine included correct version of header files
   *
   * (later we can make this check everything up to minor version number)
   */
  if (strcmp(MDAPI_VERSION, mdapi_version) != 0) {
    COND(strcmp(MDAPI_VERSION, mdapi_version) != 0);
    return MD_error_version(s);
  }

  s->engine = engine;
  return 0;
}


void *MD_engine_data(MD_Front *s)
{
  ASSERT(s != NULL);
  return s->engine;
}
