/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mdrun.c - subsystem for running the simulation engine
 */

#include <stdlib.h>
#include <string.h>
#include "mdapi/mddef.h"
/* #define DEBUG_WATCH */
#include "debug/debug.h"


/*
 * front end API
 */

int32 MD_firststep(MD_Engine *s, int32 firststep)
{
  ASSERT(s != NULL);
  s->stepnum = firststep;
  /*
   * we have to reset standard callbacks to be processed together
   * starting from this initial step number value
   */
  s->flags |= MD_CBRESET;
  return 0;
}


int32 MD_run(MD_Engine *s, int32 numsteps, int32 runflags)
{
  int32 finalstepnum, retval;

  ASSERT(s != NULL);
  ASSERT(s->run != NULL);

  /* check numsteps argument */
  if (numsteps < 0) {
    COND(numsteps < 0);
    return MD_error(s, MD_ERR_RANGE);
  }

  /* interface status flags that front end is permitted to set */
  s->flags |= (runflags & MD_CBFIRST);

  /* otherwise, front end can set any engine-dependent flags */
  /* (the low order bits are available for this purpose) */
  runflags &= MD_ENGINE_FLAG_MASK;

  /* pass interface status flags that affect engine run routine */
  /* (i.e. MD_UPDATE flag) */
  runflags |= (s->flags & ~MD_ENGINE_FLAG_MASK);

  /* validate final step number */
  finalstepnum = s->stepnum + numsteps;

  /* call the engine run routine, handle errors */
  retval = s->run(s, numsteps, runflags);
  if (retval != 0) {
    STR(MD_errmsg(s));
    COND(retval != 0);
    /* set errnum if unset */
    return (s->errnum == MD_ERR_NONE ? MD_error(s, MD_ERR_RUN) : MD_FAIL);
  }
  else if (s->stepnum != finalstepnum) {
    COND(s->stepnum != finalstepnum);
    return MD_error(s, MD_ERR_CHECK);
  }

  /* attempt to reset MD_UPDATE flag */
  MD_clear_update(s);
  return 0;
}


/*
 * engine API
 */

int32 MD_setup_run(MD_Front *s,
    int32 (*run)(MD_Front *, int32 numsteps, int32 flags))
{
  ASSERT(s != NULL);
  ASSERT(run != NULL);
  s->run = run;
  return 0;
}


int32 MD_incrstep(MD_Front *s)
{
  ASSERT(s != NULL);
  s->stepnum++;
  return 0;
}


/*
 * both APIs
 */

int32 MD_stepnum(MD_Engine *s)
{
  ASSERT(s != NULL);
  return s->stepnum;
}
