/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mddata.c - subsystem for manipulating data arrays
 */

#include <stdlib.h>
#include <string.h>
#include "mdapi/mddef.h"
#include "debug/debug.h"


/*
 * front end API
 */

int32 MD_idnum(MD_Engine *s, const char *name)
{
  int32 idnum;

  ASSERT(s != NULL);
  ASSERT(name != NULL);
  /* get idnum from table lookup of name */
  if ((idnum = adt_lookupTable(&(s->idnum), name)) == ADT_ERROR) {
    /* COND(idnum == ADT_ERROR); */
    return MD_error(s, MD_ERR_NAME);
  }
  return idnum;
}


const char *MD_name(MD_Engine *s, int32 idnum)
{
  MD_Engdata *e;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    MD_error(s, MD_ERR_IDNUM);
    return NULL;
  }
  e = data[idnum];
  return e->name;
}


const char **MD_namelist(MD_Engine *s, int32 *listlen)
{
  const char **namelist;

  ASSERT(s != NULL);
  ASSERT(listlen != NULL);
  *listlen = (int32) adt_getLengthList(&(s->name));
  namelist = (const char **) adt_getDataList(&(s->name));
  return namelist;
}


MD_Attrib MD_attrib(MD_Engine *s, int32 idnum)
{
  MD_Engdata *e;
  /* for error check any field against MD_FAIL */
  MD_Attrib err = { MD_FAIL, MD_FAIL, MD_FAIL, MD_FAIL };
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    MD_error(s, MD_ERR_IDNUM);
    return err;
  }
  e = data[idnum];
  return e->attrib;
}


int32 MD_setlen(MD_Engine *s, int32 idnum, int32 newlen)
{
  MD_Engdata *e;
  void *v;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    return MD_error(s, MD_ERR_IDNUM);
  }
  e = data[idnum];
  /* validate SETLEN access */
  if ((e->attrib.access & MD_SETLEN) != MD_SETLEN) {
    COND((e->attrib.access & MD_SETLEN) != MD_SETLEN);
    return MD_error(s, MD_ERR_ACCESS);
  }
  if (newlen < 0) {
    COND(newlen < 0);
    return MD_error(s, MD_ERR_RANGE);
  }
  if (newlen > e->attrib.max) {
    /* can't resize beyond max if no SETMAX access */
    if ((e->attrib.access & MD_SETMAX) != MD_SETMAX) {
      COND((e->attrib.access & MD_SETMAX) != MD_SETMAX);
      return MD_error(s, MD_ERR_RANGE);
    }
    v = e->reallocmem(e->buf, newlen * MD_SIZEOF(e->attrib.type));
    /* note:  newlen > 0, so NULL is an error */
    if (v == NULL) {
      COND(v == NULL);
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    /* update SHARE access */
    e->attrib.access &= ~MD_SHARE;  /* clear SHARE */
    e->buf = v;
    e->attrib.max = newlen;
  }
  e->attrib.len = newlen;
  e->attrib.access |= MD_MODIFY;    /* set MODIFY "dirty bit" */
  /* set global UPDATE engine flag */
  if (e->attrib.access & MD_NOTIFY) {
    s->flags |= MD_UPDATE;
  }
  /* force setup for any callbacks involving this engine data buffer */
  if (MD_engdata_cbsetup(s, e)) return MD_FAIL;
  return 0;
}


int32 MD_setmax(MD_Engine *s, int32 idnum, int32 newmax)
{
  MD_Engdata *e;
  void *v;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    return MD_error(s, MD_ERR_IDNUM);
  }
  e = data[idnum];
  /* validate SETMAX access */
  if ((e->attrib.access & MD_SETMAX) != MD_SETMAX) {
    COND((e->attrib.access & MD_SETMAX) != MD_SETMAX);
    return MD_error(s, MD_ERR_ACCESS);
  }
  if (newmax < 0) {
    COND(newmax < 0);
    return MD_error(s, MD_ERR_RANGE);
  }
  if (newmax < e->attrib.len) {
    /* can't resize below len if no SETLEN access */
    /* note:  SETMAX on, SETLEN off does not make sense! */
    /*        (but we still have to check for it) */
    if ((e->attrib.access & MD_SETLEN) != MD_SETLEN) {
      COND((e->attrib.access & MD_SETLEN) != MD_SETLEN);
      return MD_error(s, MD_ERR_RANGE);
    }
    e->attrib.len = newmax;
  }
  v = e->reallocmem(e->buf, newmax * MD_SIZEOF(e->attrib.type));
  /* note:  NULL can be valid if newmax == 0 */
  if (v == NULL && newmax != 0) {
    COND(v == NULL && newmax != 0);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  else if (newmax == 0) {
    v = NULL;  /* force NULL when freeing memory */
  }
  e->buf = v;
  e->attrib.max = newmax;
  e->attrib.access |= MD_MODIFY;    /* set MODIFY "dirty bit" */
  /* update SHARE access */
  if (newmax == 0 && (e->attrib.access & MD_NOSHARE) == 0
      && (e->attrib.access & (MD_WRITE | MD_RESIZE))
          == (MD_WRITE | MD_RESIZE)) {
    e->attrib.access |= MD_SHARE;   /* set SHARE */
  }
  else {
    e->attrib.access &= ~MD_SHARE;  /* clear SHARE */
  }
  /* set global UPDATE engine flag */
  if (e->attrib.access & MD_NOTIFY) {
    s->flags |= MD_UPDATE;
  }
  /* force setup for any callbacks involving this engine data buffer */
  if (MD_engdata_cbsetup(s, e)) return MD_FAIL;
  return 0;
}


int32 MD_resize(MD_Engine *s, int32 idnum, int32 newlen, int32 newmax)
{
  MD_Engdata *e;
  void *v;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    return MD_error(s, MD_ERR_IDNUM);
  }
  e = data[idnum];
  /* validate RESIZE access (SETLEN + SETMAX) */
  if ((e->attrib.access & MD_RESIZE) != MD_RESIZE) {
    COND((e->attrib.access & MD_RESIZE) != MD_RESIZE);
    return MD_error(s, MD_ERR_ACCESS);
  }
  /* validate range:  0 <= newlen <= newmax */
  if (newlen < 0 || newmax < newlen) {
    COND(newlen < 0);
    COND(newmax < newlen);
    return MD_error(s, MD_ERR_RANGE);
  }
  v = e->reallocmem(e->buf, newmax * MD_SIZEOF(e->attrib.type));
  /* note:  NULL can be valid if newmax == 0 */
  if (v == NULL && newmax != 0) {
    COND(v == NULL && newmax != 0);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  else if (newmax == 0) {
    v = NULL;  /* force NULL when freeing memory */
  }
  e->buf = v;
  e->attrib.len = newlen;
  e->attrib.max = newmax;
  e->attrib.access |= MD_MODIFY;    /* set MODIFY "dirty bit" */
  /* update SHARE access */
  if (newmax == 0 && (e->attrib.access & MD_NOSHARE) == 0
      && (e->attrib.access & (MD_WRITE | MD_RESIZE))
          == (MD_WRITE | MD_RESIZE)) {
    e->attrib.access |= MD_SHARE;   /* set SHARE */
  }
  else {
    e->attrib.access &= ~MD_SHARE;  /* clear SHARE */
  }
  /* set global UPDATE engine flag */
  if (e->attrib.access & MD_NOTIFY) {
    s->flags |= MD_UPDATE;
  }
  /* force setup for any callbacks involving this engine data buffer */
  if (MD_engdata_cbsetup(s, e)) return MD_FAIL;
  return 0;
}


int32 MD_readsub(MD_Engine *s, int32 idnum, void *buf,
    int32 nelems, int32 first)
{
  MD_Engdata *e;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    return MD_error(s, MD_ERR_IDNUM);
  }
  e = data[idnum];
  /* validate READ access */
  if ((e->attrib.access & MD_READ) != MD_READ) {
    COND((e->attrib.access & MD_READ) != MD_READ);
    return MD_error(s, MD_ERR_ACCESS);
  }
  /* validate range:  0 <= first <= len, 0 <= nelems, first + nelems <= len */
  if (first < 0 || nelems < 0 || first + nelems > e->attrib.len) {
    COND(first < 0);
    COND(nelems < 0);
    COND(first + nelems > e->attrib.len);
    return MD_error(s, MD_ERR_RANGE);
  }
  /* copy engine array into user supplied buffer */
  memcpy(buf, ((char *)(e->buf)) + first * MD_SIZEOF(e->attrib.type),
      nelems * MD_SIZEOF(e->attrib.type));
  return 0;
}


int32 MD_writesub(MD_Engine *s, int32 idnum, const void *buf,
    int32 nelems, int32 first)
{
  MD_Engdata *e;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    return MD_error(s, MD_ERR_IDNUM);
  }
  e = data[idnum];
  /* validate WRITE access */
  if ((e->attrib.access & MD_WRITE) != MD_WRITE) {
    COND((e->attrib.access & MD_WRITE) != MD_WRITE);
    return MD_error(s, MD_ERR_ACCESS);
  }
  /* validate range:  0 <= first <= len, 0 <= nelems, first + nelems <= len */
  if (first < 0 || nelems < 0 || first + nelems > e->attrib.len) {
    COND(first < 0);
    COND(nelems < 0);
    COND(first + nelems > e->attrib.len);
    return MD_error(s, MD_ERR_RANGE);
  }
  /* copy user supplied buffer into engine array */
  memcpy(((char *)(e->buf)) + first * MD_SIZEOF(e->attrib.type), buf,
      nelems * MD_SIZEOF(e->attrib.type));
  e->attrib.access |= MD_MODIFY;    /* set MODIFY "dirty bit" */
  /* set global UPDATE engine flag */
  if (e->attrib.access & MD_NOTIFY) {
    s->flags |= MD_UPDATE;
  }
  return 0;
}


int32 MD_share(MD_Engine *s, int32 idnum, void *buf, int32 len, int32 max)
{
  MD_Engdata *e;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= len);
    COND(idnum < 0);
    return MD_error(s, MD_ERR_IDNUM);
  }
  e = data[idnum];
  /* validate SHARE access */
  if ((e->attrib.access & MD_SHARE) != MD_SHARE) {
    COND((e->attrib.access & MD_SHARE) != MD_SHARE);
    return MD_error(s, MD_ERR_ACCESS);
  }
  /* validate range:  0 <= len <= max */
  if (len < 0 || max < len) {
    COND(len < 0);
    COND(max < len);
    return MD_error(s, MD_ERR_RANGE);
  }
  /* access flags should ensure e->buf is NULL */
  ASSERT(e->buf == NULL);
  e->buf = buf;
  e->attrib.len = len;
  e->attrib.max = max;
  /* memory reallocation no longer allowed */
  e->reallocmem = NULL;
  /* rearrange access flags */
  e->attrib.access &= ~(MD_SHARE | MD_SETMAX);  /* clear SHARE and SETMAX */
  e->attrib.access |= MD_UNSHARE | MD_MODIFY;   /* set UNSHARE and MODIFY */
  /* set global UPDATE engine flag */
  if (e->attrib.access & MD_NOTIFY) {
    s->flags |= MD_UPDATE;
  }
  /* force setup for any callbacks involving this engine data buffer */
  if (MD_engdata_cbsetup(s, e)) return MD_FAIL;
  return 0;
}


void *MD_unshare(MD_Engine *s, int32 idnum)
{
  MD_Engdata *e;
  void *buf;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    MD_error(s, MD_ERR_IDNUM);
    return NULL;
  }
  e = data[idnum];
  /* validate UNSHARE access */
  if ((e->attrib.access & MD_UNSHARE) != MD_UNSHARE) {
    COND((e->attrib.access & MD_UNSHARE) != MD_UNSHARE);
    MD_error(s, MD_ERR_ACCESS);
    return NULL;
  }
  buf = e->buf;
  e->buf = NULL;
  e->attrib.len = 0;
  e->attrib.max = 0;
  /* memory reallocation reactivated (we know routine is realloc()) */
  e->reallocmem = realloc;
  /* rearrange access flags */
  e->attrib.access &= ~MD_UNSHARE;                       /* clear UNSHARE */
  e->attrib.access |= MD_SHARE | MD_SETMAX | MD_MODIFY;  /* set */
  /* set global UPDATE engine flag */
  if (e->attrib.access & MD_NOTIFY) {
    s->flags |= MD_UPDATE;
  }
  /* force setup for any callbacks involving this engine data buffer */
  if (MD_engdata_cbsetup(s, e)) return NULL;
  /* return front end provided buffer from MD_share() call */
  return buf;
}


void *MD_direct(MD_Engine *s, int32 idnum)
{
  MD_Engdata *e;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || idnum < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    MD_error(s, MD_ERR_IDNUM);
    return NULL;
  }
  e = data[idnum];
  /* validate DIRECT access */
  if ((e->attrib.access & MD_DIRECT) != MD_DIRECT) {
    COND((e->attrib.access & MD_DIRECT) != MD_DIRECT);
    MD_error(s, MD_ERR_ACCESS);
    return NULL;
  }
  /* return the buffer pointer */
  return e->buf;
}


int32 MD_setmod(MD_Engine *s, int32 idnum)
{
  MD_Engdata *e;
  MD_Engdata **data;
  int32 listlen;

  ASSERT(s != NULL);
  listlen = (int32) adt_getLengthList(&(s->pengdata));
  data = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  /* validate idnum */
  if (idnum >= listlen || listlen < 0) {
    COND(idnum >= listlen);
    COND(idnum < 0);
    return MD_error(s, MD_ERR_IDNUM);
  }
  e = data[idnum];
  /* call is valid only if DIRECT access could have been granted */
  if ((e->attrib.access & MD_DIRECT) != MD_DIRECT) {
    COND((e->attrib.access & MD_DIRECT) != MD_DIRECT);
    return MD_error(s, MD_ERR_ACCESS);
  }
  e->attrib.access |= MD_MODIFY;  /* set MODIFY "dirty bit" */
  /* set global UPDATE engine flag */
  if (e->attrib.access & MD_NOTIFY) {
    s->flags |= MD_UPDATE;
  }
  return 0;
}


int32 MD_update(MD_Engine *s)
{
  /* nothing to do for single threaded implementation */
  return 0;
}


/*
 * engine API
 */

/*** during engine initialization ***/
static MD_Engdata *engdata_alloc(MD_Interface *s, const char *name,
    void *buf, void *(*reallocmem)(void *, size_t), MD_Attrib attrib)
{
  MD_Engdata *engdata;
  TypeInfo *typinfo;
  int32 typinfo_len;
  int32 index;
  int32 idnum, k;

  ASSERT(s != NULL);
  ASSERT(name != NULL);

  /* check attribute len and max with buf allocation */
  if ((buf == NULL && attrib.max != 0) || (buf != NULL && attrib.max == 0)
     || attrib.len < 0 || attrib.max < attrib.len) {
    COND(buf == NULL && attrib.max != 0);
    COND(buf != NULL && attrib.max == 0);
    COND(attrib.len < 0);
    COND(attrib.max < attrib.len);
    MD_error(s, MD_ERR_NEWDATA);
    return NULL;
  }

  /* check type attribute */
  typinfo = (TypeInfo *) adt_getDataList(&(s->typinfo));
  typinfo_len = (int32) adt_getLengthList(&(s->typinfo));
  index = attrib.type >> MD_SHIFT_TYPE;
  if (index >= typinfo_len || index < 0
      || typinfo[index].typnum != attrib.type) {
    COND(index >= typinfo_len);
    COND(index < 0);
    COND(typinfo[index].typnum != attrib.type);
    MD_error(s, MD_ERR_NEWDATA);
    return NULL;
  }

  /* check validity (but not content) of access flags */
  if ((attrib.access & ~MD_ALLFLAGS) != 0) {
    COND((attrib.access & ~MD_ALLFLAGS) != 0);
    MD_error(s, MD_ERR_NEWDATA);
    return NULL;
  }

  /* check attributes SETMAX or ESETMAX against non-NULL reallocmem */
  if (((attrib.access & MD_SETMAX) || (attrib.access & MD_ESETMAX))
      && reallocmem == NULL) {
    COND(((attrib.access & MD_SETMAX) || (attrib.access & MD_ESETMAX))
        && reallocmem == NULL);
    MD_error(s, MD_ERR_NEWDATA);
    return NULL;
  }

  /* set SHARE access if conditions permit */
  if ((attrib.access & MD_ERESIZE) != 0) {
    attrib.access |= MD_NOSHARE;
  }
  else if (buf == NULL && (attrib.access & MD_NOSHARE) == 0
      && (attrib.access & (MD_WRITE | MD_RESIZE)) == (MD_WRITE | MD_RESIZE)) {
    attrib.access |= MD_SHARE;
  }

  /* (F)CBREAD require READ access */
  if ((attrib.access & (MD_CBREAD | MD_FCBREAD)) != 0
      && (attrib.access & MD_READ) == 0) {
    COND((attrib.access & (MD_CBREAD | MD_FCBREAD)) != 0
        && (attrib.access & MD_READ) == 0);
    MD_error(s, MD_ERR_NEWDATA);
    return NULL;
  }

  /* (F)CBWRITE and (F)CBSHARE all require WRITE access */
  if ((attrib.access & (MD_CBWRITE | MD_FCBWRITE | MD_CBSHARE | MD_FCBSHARE))
      != 0 && (attrib.access & MD_WRITE) == 0) {
    COND((attrib.access & (MD_CBWRITE | MD_FCBWRITE | MD_CBSHARE | MD_FCBSHARE))
        != 0 && (attrib.access & MD_WRITE) == 0);
    MD_error(s, MD_ERR_NEWDATA);
    return NULL;
  }

  /* ID number will be next element indexed in engdata array */
  idnum = adt_getLengthList(&(s->pengdata));

  /* first try to place (name,idnum) into idnum table */
  if ((k = adt_insertTable(&(s->idnum), name, idnum)) == ADT_ERROR) {
    COND(k == ADT_ERROR);
    MD_error(s, MD_ERR_MEMALLOC);
    return NULL;
  }
  else if (k != idnum) {
    /* conflicting data array name */
    COND(k != idnum);
    MD_error(s, MD_ERR_NEWDATA);
    return NULL;
  }

  /*
   * append name to data name list
   * since aliasing (i.e. associating different names with single idnum)
   * is allowed, this array might be longer than pengdata array
   */
  if (adt_appendList(&(s->name), &name)) {
    MD_error(s, MD_ERR_MEMALLOC);
    return NULL;
  }

  /* allocate memory for engdata and set fields */
  if ((engdata = (MD_Engdata *) calloc(1, sizeof(MD_Engdata))) == NULL) {
    ERRCOND(engdata == NULL);
    MD_error(s, MD_ERR_MEMALLOC);
    return NULL;
  }
  engdata->name = name;
  engdata->buf = buf;
  engdata->reallocmem = reallocmem;
  engdata->attrib = attrib;
  engdata->idnum = idnum;

  /* append engdata to pengdata array */
  if (adt_appendList(&(s->pengdata), &engdata)) {
    /* have to free engdata, cannot retain it */
    free(engdata);
    MD_error(s, MD_ERR_MEMALLOC);
    return NULL;
  }

  /* init array of callback pointers */
  if (adt_initializeList(&(engdata->cback), sizeof(MD_Callback *), 0, NULL)) {
    MD_error(s, MD_ERR_MEMALLOC);
    return NULL;
  }

  return engdata;
}


MD_Engdata *MD_engdata(MD_Front *s, const char *name,
    int32 type, int32 access)
{
  MD_Attrib attrib;

  ASSERT(s != NULL);
  ASSERT(name != NULL);
  /* typical engdata node:  size 0, use C realloc to manage memory */
  attrib.type = type;
  attrib.len = 0;
  attrib.max = 0;
  attrib.access = access;
  return engdata_alloc(s, name, NULL, realloc, attrib);
}


MD_Engdata *MD_engdata_buffer(MD_Front *s, const char *name,
    MD_Attrib attrib, void *buf)
{
  ASSERT(s != NULL);
  ASSERT(name != NULL);
  /* engine supplies buffer, can't change its size (could be static) */
  attrib.access &= ~(MD_SETMAX | MD_ESETMAX);  /* clear SETMAX and ESETMAX */
  return engdata_alloc(s, name, buf, NULL, attrib);
}


MD_Engdata *MD_engdata_manage(MD_Front *s, const char *name,
    MD_Attrib attrib, void *buf, void *(*reallocmem)(void *ptr, size_t size))
{
  ASSERT(s != NULL);
  ASSERT(name != NULL);
  ASSERT(reallocmem != NULL);
  /* engine supplies memory management routine, front end can't SHARE */
  attrib.access |= MD_NOSHARE;  /* set NOSHARE */
  return engdata_alloc(s, name, buf, reallocmem, attrib);
}


int32 MD_engdata_alias(MD_Front *s, const char *name, MD_Engdata *e)
{
  int32 k;

  ASSERT(s != NULL);
  ASSERT(name != NULL);
  ASSERT(e != NULL);

  /* first try to place (name,idnum) into idnum table */
  if ((k = adt_insertTable(&(s->idnum), name, e->idnum)) == ADT_ERROR) {
    COND(k == ADT_ERROR);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  else if (k != e->idnum) {
    /* conflicting data array name */
    COND(k != e->idnum);
    return MD_error(s, MD_ERR_NEWDATA);
  }

  /* append name to data name list */
  if (adt_appendList(&(s->name), &name)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  return 0;
}


/*** during engine run routine ***/
int32 MD_engdata_setlen(MD_Front *s, MD_Engdata *e, int32 newlen)
{
  void *v;

  ASSERT(s != NULL);
  ASSERT(e != NULL);
  /* validate ESETLEN access */
  if ((e->attrib.access & MD_ESETLEN) != MD_ESETLEN) {
    COND((e->attrib.access & MD_ESETLEN) != MD_ESETLEN);
    return MD_error(s, MD_ERR_ACCESS);
  }
  if (newlen < 0) {
    COND(newlen < 0);
    return MD_error(s, MD_ERR_RANGE);
  }
  if (newlen > e->attrib.max) {
    /* can't resize beyond max if no ESETMAX access */
    if ((e->attrib.access & MD_ESETMAX) != MD_ESETMAX) {
      COND((e->attrib.access & MD_ESETMAX) != MD_ESETMAX);
      return MD_error(s, MD_ERR_RANGE);
    }
    v = e->reallocmem(e->buf, newlen * MD_SIZEOF(e->attrib.type));
    /* note:  newlen > 0, so NULL is an error */
    if (v == NULL) {
      COND(v == NULL);
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    e->buf = v;
    e->attrib.max = newlen;
  }
  e->attrib.len = newlen;
  /* force setup for any callbacks involving this engine data buffer */
  if (MD_engdata_cbsetup(s, e)) return MD_FAIL;
  return 0;
}


int32 MD_engdata_setmax(MD_Front *s, MD_Engdata *e, int32 newmax)
{
  void *v;

  ASSERT(s != NULL);
  ASSERT(e != NULL);
  /* validate ESETMAX access */
  if ((e->attrib.access & MD_ESETMAX) != MD_ESETMAX) {
    COND((e->attrib.access & MD_ESETMAX) != MD_ESETMAX);
    return MD_error(s, MD_ERR_ACCESS);
  }
  if (newmax < 0) {
    COND(newmax < 0);
    return MD_error(s, MD_ERR_RANGE);
  }
  if (newmax < e->attrib.len) {
    /* can't resize below len if no ESETLEN access */
    /* note:  ESETMAX on, ESETLEN off does not make sense! */
    /*        (but we still have to check for it) */
    if ((e->attrib.access & MD_ESETLEN) != MD_ESETLEN) {
      COND((e->attrib.access & MD_ESETLEN) != MD_ESETLEN);
      return MD_error(s, MD_ERR_RANGE);
    }
    e->attrib.len = newmax;
  }
  v = e->reallocmem(e->buf, newmax * MD_SIZEOF(e->attrib.type));
  /* note:  NULL can be valid if newmax == 0 */
  if (v == NULL && newmax != 0) {
    COND(v == NULL && newmax != 0);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  else if (newmax == 0) {
    v = NULL;  /* force NULL when freeing memory */
  }
  e->buf = v;
  e->attrib.max = newmax;
  /* force setup for any callbacks involving this engine data buffer */
  if (MD_engdata_cbsetup(s, e)) return MD_FAIL;
  return 0;
}


int32 MD_engdata_resize(MD_Front *s, MD_Engdata *e, int32 newlen,
    int32 newmax)
{
  void *v;

  ASSERT(s != NULL);
  ASSERT(e != NULL);
  /* validate ERESIZE access (ESETLEN + ESETMAX) */
  if ((e->attrib.access & MD_ERESIZE) != MD_ERESIZE) {
    COND((e->attrib.access & MD_ERESIZE) != MD_ERESIZE);
    return MD_error(s, MD_ERR_ACCESS);
  }
  /* validate range:  0 <= newlen <= newmax */
  if (newlen < 0 || newmax < newlen) {
    COND(newlen < 0);
    COND(newmax < newlen);
    return MD_error(s, MD_ERR_RANGE);
  }
  v = e->reallocmem(e->buf, newmax * MD_SIZEOF(e->attrib.type));
  /* note:  NULL can be valid if newmax == 0 */
  if (v == NULL && newmax != 0) {
    COND(v == NULL && newmax != 0);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  else if (newmax == 0) {
    v = NULL;  /* force NULL when freeing memory */
  }
  e->buf = v;
  e->attrib.len = newlen;
  e->attrib.max = newmax;
  /* force setup for any callbacks involving this engine data buffer */
  if (MD_engdata_cbsetup(s, e)) return MD_FAIL;
  return 0;
}


int32 MD_engdata_ackmod(MD_Front *s, MD_Engdata *e)
{
  ASSERT(s != NULL);
  ASSERT(e != NULL);
  if (e->attrib.access & MD_MODIFY) {
    e->attrib.access &= ~MD_MODIFY;  /* clear modify flag */
  }
  else {
    COND((e->attrib.access & MD_MODIFY) == 0);
    return MD_error(s, MD_ERR_ACCESS);
  }
  return 0;
}


void MD_free_data(MD_Front *s)
{
  MD_Engdata **pe;
  MD_Engdata *e;
  int32 len, idnum;

  ASSERT(s != NULL);
  /* walk through engdata list */
  pe = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  len = (int32) adt_getLengthList(&(s->pengdata));
  for (idnum = 0;  idnum < len;  idnum++) {
    e = pe[idnum];
    /* free buffer using reallocmem routine, then zero buf and attrib */
    if (e->reallocmem != NULL && e->buf != NULL) {
      e->reallocmem(e->buf, 0);
      e->buf = NULL;
      e->attrib.len = e->attrib.max = 0;
    }
  }
}


/*
 * internal methods
 */

int32 MD_init_data(MD_Interface *s)
{
  ASSERT(s != NULL);
  /* init data handling containers */
  if (adt_initializeTable(&(s->idnum), 0)
      || adt_initializeList(&(s->name), sizeof(const char *), 0, NULL)
      || adt_initializeList(&(s->pengdata), sizeof(MD_Engdata *), 0, NULL)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  return 0;
}


void MD_done_data(MD_Interface *s)
{
  MD_Engdata **pe;
  MD_Engdata *e;
  int32 len, idnum;

  ASSERT(s != NULL);
  /* need to free memory for each engdata node */
  pe = (MD_Engdata **) adt_getDataList(&(s->pengdata));
  len = (int32) adt_getLengthList(&(s->pengdata));
  for (idnum = 0;  idnum < len;  idnum++) {
    e = pe[idnum];
    if (e != NULL) {
      /* free buffer using reallocmem routine */
      if (e->reallocmem != NULL && e->buf != NULL) {
        e->reallocmem(e->buf, 0);
      }
      /* free array of callbacks */
      adt_cleanupList(&(e->cback));
      /* zero memory space before freeing node */
      memset(e, 0, sizeof(MD_Engdata));
      free(e);
    }
  }
  /* free data handling containers */
  adt_cleanupTable(&(s->idnum));
  adt_cleanupList(&(s->name));
  adt_cleanupList(&(s->pengdata));
}


/* called internally at end of MD_run() */
void MD_clear_update(MD_Interface *s)
{
  ASSERT(s != NULL);
  if (s->flags & MD_UPDATE) {
    MD_Engdata **pe;
    int32 len, idnum;
    int32 clear = 1;

    /*
     * clear update flag if all modified buffers that engine was
     * notified about have been acknowledged
     */
    pe = (MD_Engdata **) adt_getDataList(&(s->pengdata));
    len = (int32) adt_getLengthList(&(s->pengdata));
    for (idnum = 0;  idnum < len;  idnum++) {
      if ((pe[idnum]->attrib.access & (MD_NOTIFY | MD_MODIFY))
          == (MD_NOTIFY | MD_MODIFY)) {
        clear = 0;
        break;
      }
    }
    if (clear) {
      s->flags &= ~MD_UPDATE;
    }
  }
}
