/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mdcommon.h - type definitions common to front end and engine APIs;
 *   this header should not be included directly by user, instead
 *   indirectly by including either mdfront.h or mdengine.h
 */

#ifndef MDCOMMON_H
#define MDCOMMON_H

#include "mdapi/mdtypes.h"
#include "adt/list.h"
#include "adt/table.h"

#ifdef __cplusplus
extern "C" {
#endif

  /*
   * Naming conventions, units, and types for engine data arrays:
   *
   *   "pos" -- position, A (MD_Dvec or MD_Fvec)
   *   "vel" -- velocity, A/fs (MD_Dvec or MD_Fvec)
   *   "force" -- total force, kcal/mol/A (MD_Dvec or MD_Fvec)
   *
   *   "atom" -- atom "topology" (MD_Atom)
   *   "bond" -- bond topology (MD_Bond)
   *   "angle" -- angle topology (MD_Angle)
   *   "dihed" -- dihedral topology (MD_Tors)
   *   "impr" -- improper topology (MD_Tors)
   *   "excl" -- nonbonded exclusions (MD_Excl)
   *
   *   "atomprm" -- atom parameters (MD_AtomPrm)
   *   "bondprm" -- bond parameters (MD_BondPrm)
   *   "angleprm" -- angle parameters (MD_AnglePrm)
   *   "dihedprm" -- dihedral parameters (MD_TorsPrm)
   *   "imprprm" -- improper parameters (MD_TorsPrm)
   *   "nbfixprm" -- override nonbonded parameters from atomprm (MD_NbfixPrm)
   *
   *   "param" -- simulation parameters (Param)
   *   where "Param" is an engine-defined type with at least members:
   *     "timestep" -- time step (double)
   *     "dielectric" -- dielectric constant (double)
   *     "cutoff" -- cutoff for van der Waals (double)
   *         (perhaps also used for electrostatics)
   *     "switchdist" -- switching distance for van der Waals (double)
   *     "temperature" -- initial temperature for system (double)
   *   others members might be:
   *     "1-4scaling" -- scaling const for modified 1-4 interactions (double)
   *     "exclude" -- specify type of exclusions (MD_String?)
   *     "commotion" -- allow center of mass motion? (MD_String?)
   *
   *   "result" -- computed quantities from simulation (Result)
   *   where "Result" is an engine-defined type with at least members:
   *     "energy" -- total energy (double)
   *     "ke" -- kinetic energy (double)
   *     "pe" -- potential energy (double)
   *     "temp" -- temperature (double)
   *   other members might be:
   *     "bond" -- bond potential (double)
   *     "angle" -- angle potential (double)
   *     "dihed" -- dihedral potential (double)
   *     "impr" -- improper potential (double)
   *     "vdw" -- van der Waals potential (double)
   *     "elec" -- electrostatic potential (double)
   *     "press" -- pressure (double)
   *     "virial" -- virial (double)
   *     "linmo" -- linear momentum (MD_Dvec)
   *     "angmo" -- angular momentum (MD_Dvec)
   */


  /* failure returned from function */
  enum {
    MD_FAIL = -1
  };


  /* error numbers */             /* diagnostic messages */
  enum {
    MD_ERR_NONE = 0,           /* Success                         */
    MD_ERR_VERSION,            /* Inconsistent version number     */
    MD_ERR_MEMALLOC,           /* Memory cannot be allocated      */
    MD_ERR_INIT,               /* Engine initialization failed    */
    MD_ERR_RUN,                /* Engine run simulation failed    */
    MD_ERR_CHECK,              /* Consistency check failed        */
    MD_ERR_NEWDATA,            /* Cannot create new engine data   */
    MD_ERR_NEWTYPE,            /* Cannot create new type          */
    MD_ERR_NAME,               /* Unrecognized name string        */
    MD_ERR_IDNUM,              /* Invalid data ID number          */
    MD_ERR_TYPENUM,            /* Invalid type number             */
    MD_ERR_RANGE,              /* Value is out of range           */
    MD_ERR_ACCESS,             /* Access is denied                */
    MD_ERR_CALLBACK,           /* Callback routine failed         */
    MD_ERR_CBSHARE,            /* Callback shared buffer failed   */
    MD_ERROR_LISTLEN              /* not an error */
  };


  /*
   * Every data array shared between engine and front end has
   * "attributes" indicated by its MD_Attrib structure.
   */
  typedef struct MD_Attrib_tag {
    int32 type;    /* indicates data type of array elements */
    int32 len;     /* used storage length, counting from the 0th element */
    int32 max;     /* maximum length allocated to array (0 <= len <= max) */
    int32 access;  /* bitwise flag indicating access rights granted to */
                   /*   front end by engine, plus internal status */
  } MD_Attrib;


  /*
   * access flags
   *
   * enable access by front end to engine data
   */
  enum {
  /* set by engine */
    MD_READ     = 0x00001, /* read access */
    MD_WRITE    = 0x00002, /* write access */
    MD_CBREAD   = 0x00004, /* read during callback */
    MD_CBWRITE  = 0x00008, /* write during callback */
    MD_CBSHARE  = 0x00010, /* shared data from front end during callback */
    MD_FCBREAD  = 0x00020, /* read during force callback */
    MD_FCBWRITE = 0x00040, /* write during force callback */
    MD_FCBSHARE = 0x00080, /* shared data from front end during force cb */
    MD_SETLEN   = 0x00100, /* front end allowed to set length */
    MD_SETMAX   = 0x00200, /* front end allowed to set maximum allocation */
    MD_RESIZE   = MD_SETLEN | MD_SETMAX,
    MD_ESETLEN  = 0x00400, /* engine allowed to set length */
    MD_ESETMAX  = 0x00800, /* engine allowed to set maximum allocation */
    MD_ERESIZE  = MD_ESETLEN | MD_ESETMAX,
    MD_DIRECT   = 0x01000, /* direct access */
    MD_NOSHARE  = 0x02000, /* do not permit front end shared buffer */
    MD_NOTIFY   = 0x04000, /* notify engine of front end updates */

    MD_ALLFLAGS = MD_READ | MD_WRITE 
        | MD_CBREAD | MD_CBWRITE | MD_CBSHARE
        | MD_FCBREAD | MD_FCBWRITE | MD_FCBSHARE 
        | MD_RESIZE | MD_ERESIZE
        | MD_DIRECT | MD_NOSHARE | MD_NOTIFY,

  /* set by MDAPI, used as internal status */
    MD_SHARE    = 0x10000, /* front end allowed to share its buffer */
    MD_UNSHARE  = 0x20000, /* front end allowed to take back its buffer */
    MD_MODIFY   = 0x40000  /* front end has modified buffer ("dirty bit") */
  };


  /*
   * global flags
   *
   * MD_CBFIRST intended to be set by front end in MD_run() call;
   * engine checks flags against MD_UPDATE to determine if front end
   * modified some engine data that had MD_NOTIFY access set;
   * the other defined flags are for internal use.
   *
   * Flag values up through 0xFFFF (the low-order bits in the word) are
   * reserved to the engine to permit the designation of engine-dependent
   * flags.  These will be ORed with the internally set flags and all will
   * be passed on to the engine run routine.
   */
  enum {
    MD_UPDATE   = 0x1000000, /* array with MD_NOTIFY flag was modified */
    MD_CBFIRST  = 0x2000000, /* run flag invoke callbacks before first step */
    MD_CBRESET  = 0x4000000, /* internal, reset nextstepnum for callbacks */
    MD_CBUPDATE = 0x8000000, /* internal, update one or more nextstepnum */

    MD_ENGINE_FLAG_MASK = 0x0000FFFF  /* low-order bits reserved for engine */
  };


  /*
   * The front end can establish callbacks to itself during a running
   * simulation.  There are three types of callbacks (standard, force,
   * and message) all explained in mdfront.h.  The standard and force
   * callbacks receive an array of MD_Cbdata through which data subarrays
   * are passed.  Setting up a callback also requires an array of MD_Cbdata
   * indicating the data subarrays needed.
   */
  /*** need forward definition ***/
  struct MD_Engdata_tag;

  typedef struct MD_Cbdata_tag {
  /*** these are set by front end for initialization ***/
    int32 idnum;      /* data array identification number */
    int32 access;     /* requested access, choices are: */
                      /*   MD_CBREAD, MD_CBWRITE, MD_CBSHARE, */
                      /*   MD_FCBREAD, MD_FCBWRITE, MD_FCBSHARE */
    int32 nelems;     /* number of elements desired from subarray */
                      /*   (note that -1 is used to indicate "all elements") */
    int32 first;      /* index of first element from data array */
                      /*   to be included in subarray */

  /*** this is provided to engine ***/
    struct MD_Engdata_tag *engdata;

  /*** these are provided to callback routine ***/
    void *data;       /* points to the subarray */
    MD_Attrib attrib; /* attributes for this data array */
  } MD_Cbdata;

  /*
   * For examples below use:  MD_Cbdata c;
   *
   * The access rights requested depend on the type of callback that the
   * front end is attempting to enable.  For a standard callback, choose
   * from MD_CBREAD, MD_CBWRITE, MD_CBSHARE.  For a force callback, choose
   * from MD_FCBREAD, MD_FCBWRITE, MD_FCBSHARE.  Note that the data array
   * must already have the particular access rights enabled that you are
   * requesting for your callback.
   *
   * For example, if the data array allows MD_CBREAD and MD_CBWRITE access,
   * then you can set c.access=MD_CBREAD.
   *
   * Some examples might also help for understanding "nelems" and "first".
   * Suppose that the data array has len=10.  Then setting c.first=3 and
   * c.nelems=5 means that within the callback data[0..4] will correspond
   * to engine array data[3..7].
   *
   * If you want your callback to receive the entire array, but you don't
   * know how long the array is (e.g. engine is allowed to resize it),
   * then set c.nelems=-1 and c.first=0.  If you wanted all but the first
   * element, then you could set c.nelems=-1, c.first=1.  And so on.
   *
   * Access rights MD_CBSHARE and MD_FCBSHARE change the semantics.  In
   * this case, data=NULL and nelems=first=0 is received by the callback,
   * and it is up to the callback to provide the data array to the engine.
   * The subarray semantics still apply, with updates to nelems and first
   * describing what portion of the engine data array is being provided
   * by the callback and possibly resizing the data array, so these values
   * need to be set carefully by the callback.  The front end retains
   * ownership of the memory it sends through the "data" pointer, with
   * copy semantics like MD_write(), which means that the same memory
   * buffer can be reused on subsequent callbacks, and the front end is
   * responsible for freeing the memory.  This mechanism is provided to
   * allow the front end to send arbitrary length arrays to the engine
   * during a simulation.  MD_CBSHARE and MD_FCBSHARE cannot be mixed with
   * the other access types (e.g. you can't have callback access set to
   * both MD_CBREAD and MD_CBSHARE for the given MD_Cbdata element, however
   * the engine could have enabled both MD_CBREAD and MD_CBSHARE access on
   * the same data array).  Data array resize changes are permitted only
   * if front end is granted access to do so, otherwise MD_ERR_CBSHARE is
   * returned from callback processing.  If data array is resized, the
   * MD_MODIFY flag is set on the array.
   */


  /*
   * Derived types are viewed as structs of primary and previously defined
   * derived types.  An array of MD_Member describes the members of the
   * derived type.  For primary types, MD_Member is given with len=1 and
   * name="" (note that this is *not* the name of the type, rather the name
   * of the members of the type).  Also, recall that MD_Fvec and MD_Dvec
   * are considered primary types.
   */
  typedef struct MD_Member_tag {
    int32 type;        /* indicates data type of the member */
    int32 len;         /* the number of elements of this member */
                       /*   (i.e. statically defined array) */
    const char *name;  /* string name of the member */
  } MD_Member;

  /*
   * Examples:
   *
   *   int32 has a 1 element MD_Member array:
   *   { { MD_INT32, 1, "" } }
   *
   *   MD_Dvec has a 1 element MD_Member array (it's a primary type):
   *   { { MD_DVEC, 1, "" } }
   *
   *   MD_BondPrm has a 3 element MD_Member array:
   *   { {MD_DOUBLE, 1, "k"}, {MD_DOUBLE, 1, "r0"}, {MD_NAME, 2, "type"} }
   */

/********************************* internal **********************************/

  typedef struct MD_Error_tag {
    const char *msg;
    int32 isfatal;
  } MD_Error;


  typedef struct MD_Interface_tag {
    /* error handling */
    int32 errnum;        /* stores current error number */
    MD_Error error;      /* stores data on current error state */
    adt_List engerror;   /* list of engine defined errors */

    /* type handling */
    adt_Table typnum;    /* associates each typname key with typnum value */
    adt_List typname;    /* list of all type names */
    adt_List typinfo;    /* list of information for each type */

    /* data array handling */
    adt_Table idnum;     /* associates each data name key with idnum value */
    adt_List name;       /* list of all data array names */
    adt_List pengdata;   /* list of pointers to information for each datum */

    /* callback handling */
    adt_List pcbq;       /* keep pointers to callbacks in priority queue */
    adt_List pfcb;       /* list of pointers to force callbacks */
    adt_List msgcb;      /* list of message callbacks */

    /* run subsystem */
    int32 (*run)(struct MD_Interface_tag *, int32 numsteps, int32 runflags);
    int32 flags;         /* global flags, OR with runflags and pass to run */
    int32 stepnum;       /* step number counter (iterated by engine) */

    /* setup engine */
    void *engine;        /* stores pointer to engine internal data */
    const char *engname; /* nil-terminated string name of engine */
      /* engine constructor and destructor */
    int (*engine_init)(struct MD_Interface_tag *, int32 flags);
    void (*engine_done)(struct MD_Interface_tag *);

  } MD_Interface;


#ifdef __cplusplus
}
#endif

#endif  /* MDCOMMON_H */
