/*
 * Copyright (C) 2003-2004 by David J. Hardy.  All rights reserved.
 *
 * mdcback.c - subsystem for handling callbacks
 */

#include <stdlib.h>
#include <string.h>
#include "mdapi/mddef.h"
#include "debug/debug.h"


/*
 * internal routines
 */

/* internal routines to manage callback list held by MD_Engdata object */
static int32 engdata_append_callback(MD_Engine *, MD_Callback *, MD_Engdata *);
static int32 engdata_remove_callback(MD_Engine *, MD_Callback *, MD_Engdata *);

/* internal routines to manage cbdata */
enum { CB, FCB };  /* cbtype values */
static int32 cbdata_validate(MD_Engine *, MD_Cbdata *, int32 cbtype);

/* internal routines to manage callback processing */
static int32 callback_setup(MD_Engine *, MD_Callback *);
static int32 callback_share(MD_Engine *, MD_Callback *);
static int32 callback_remove(MD_Engine *, MD_Callback *);


/* 
 * append callback pointer to MD_Engdata callback list
 */
int32 engdata_append_callback(MD_Engine *s, MD_Callback *c, MD_Engdata *e)
{
  adt_List *cback;
  MD_Callback **pcb;
  int32 len;

  ASSERT(s != NULL);
  ASSERT(c != NULL);
  ASSERT(e != NULL);
  cback = &(e->cback);
  /* 
   * have to make sure callback is not already listed
   * if already there, it will be last element in array
   */
  len = (int32) adt_getLengthList(cback);
  pcb = (MD_Callback **) adt_getDataList(cback);
  if (len == 0 || pcb[len-1] != c) {
    if (adt_appendList(cback, &c)) {
      return MD_error(s, MD_ERR_MEMALLOC);
    }
  }
  return 0;
}


/*
 * remove callback pointer from MD_Engdata callback list
 */
int32 engdata_remove_callback(MD_Engine *s, MD_Callback *c, MD_Engdata *e)
{
  adt_List *cback;
  MD_Callback **p;
  int32 len, k;

  ASSERT(s != NULL);
  ASSERT(c != NULL);
  ASSERT(e != NULL);
  cback = &(e->cback);
  p = (MD_Callback **) adt_getDataList(cback);
  len = (int32) adt_getLengthList(cback);
  /* there will be at most one instance of c in list */
  for (k = 0;  k < len;  k++) {
    if (p[k] == c) {
      /* fill the hole using last array element (order doesn't matter) */
      p[k] = p[len-1];
      /* reduce length of array by 1 */
      if (adt_deleteList(cback)) {
        return MD_error(s, MD_ERR_MEMALLOC);
      }
      break;
    }
  }
  return 0;
}


int32 cbdata_validate(MD_Engine *s, MD_Cbdata *c, int32 cbtype)
{
  MD_Engdata *e;
  int32 all, share, nelems;
  adt_List *pengdata;
  MD_Engdata **pengdata_data;
  int32 pengdata_len;

  ASSERT(s != NULL);
  ASSERT(c != NULL);
  pengdata = &(s->pengdata);
  pengdata_len = (int32) adt_getLengthList(pengdata);
  pengdata_data = (MD_Engdata **) adt_getDataList(pengdata);

  /* validate idnum */
  if (c->idnum >= pengdata_len || c->idnum < 0) {
    COND(c->idnum >= pengdata_len);
    COND(c->idnum < 0);
    return MD_error(s, MD_ERR_IDNUM);
  }

  /* get pointer to engine data */
  e = pengdata_data[c->idnum];

  /* set pointer in cbdata object back to engine data */
  c->engdata = e;

  /* set which access flags based on "cbtype" */
  switch (cbtype) {
    case CB:
      all = MD_CBREAD | MD_CBWRITE | MD_CBSHARE;
      share = MD_CBSHARE;
      break;
    case FCB:
      all = MD_FCBREAD | MD_FCBWRITE | MD_FCBSHARE;
      share = MD_FCBSHARE;
      break;
    default:
      BUG("invalid callback type");
  };

  /* validate callback access rights */
  if ((c->access & ~all)
      || (c->access & e->attrib.access) != c->access
      || ((c->access & share) && c->access != share)) {
    COND(c->access & ~all);
    COND((c->access & e->attrib.access) != c->access);
    COND((c->access & share) && c->access != share);
    return MD_error(s, MD_ERR_ACCESS);
  }

  /* validate values for nelems and first only if not shared */
  if ((c->access & share) == 0) {
    /* set nelems using wildcard value -1 to mean "all of the elements" */
    nelems = c->nelems;
    if (nelems == -1 && c->first <= e->attrib.len) {
      nelems = e->attrib.len - c->first;
    }

    /* validate subarray range against engine data attributes */
    if (c->first < 0 || nelems < 0
        || c->first + nelems > e->attrib.len) {
      COND(c->first < 0);
      COND(nelems < 0);
      COND(c->first + nelems > e->attrib.len);
      return MD_error(s, MD_ERR_RANGE);
    }
  }

  /* validation successful */
  return 0;
}


int32 callback_setup(MD_Engine *s, MD_Callback *c)
{
  MD_Cbdata *p;
  MD_Engdata *e;
  int32 k;

  ASSERT(s != NULL);
  ASSERT(c != NULL);

  /* loop through all cbdata elements */
  for (k = 0;  k < c->cbarglen;  k++) {
    p = &(c->cbarg[k]);
    e = p->engdata;

    /* copy engine data attributes */
    p->attrib = e->attrib;

    /* setup pointer to buffer */
    if (p->access & (MD_CBSHARE | MD_FCBSHARE)) {
      c->flags |= MD_CALLBACK_SHARE;
      p->nelems = 0;
      p->first = 0;
      p->data = NULL;
    }
    else {
      ASSERT(p->nelems != -1 || p->first <= e->attrib.len);
      ASSERT(p->nelems == -1 || p->first + p->nelems <= e->attrib.len);
      p->data = ((char *)(e->buf)) + p->first * MD_SIZEOF(e->attrib.type);
    }

    /* check if we need to append callback to engine data list */
    if (c->flags & MD_CALLBACK_INIT) {
      if (engdata_append_callback(s, c, e)) return MD_FAIL;
    }
  }

  /* clear INIT and SETUP flags for this callback */
  c->flags &= ~(MD_CALLBACK_INIT | MD_CALLBACK_SETUP);

  /* callback setup successful */
  return 0;
}


int32 callback_remove(MD_Engine *s, MD_Callback *c)
{
  int32 k;

  /* loop over all MD_Cbdata items, remove references to this callback */
  for (k = 0;  k < c->cbarglen;  k++) {
    if (engdata_remove_callback(s, c, c->cbarg[k].engdata)) return MD_FAIL;
  }

  /* clear memory used by callback */
  memset(c, 0, sizeof(MD_Callback));

  return 0;
}


int32 callback_share(MD_Engine *s, MD_Callback *c)
{
  MD_Cbdata *p;
  MD_Engdata *e;
  int32 k, len;

  ASSERT(s != NULL);
  ASSERT(c != NULL);

  /* loop through all cbdata elements */
  for (k = 0;  k < c->cbarglen;  k++) {
    p = &(c->cbarg[k]);

    /* look for shared buffer */
    if ((p->access & (MD_CBSHARE | MD_FCBSHARE)) == 0) continue;

    e = p->engdata;
    /*
     * make sure engine buffer is large enough
     * code below based on MD_setlen()
     */
    len = p->first + p->nelems;
    if (len > e->attrib.len) {
      if ((e->attrib.access & MD_SETLEN) != MD_SETLEN) {
        COND((e->attrib.access & MD_SETLEN) != MD_SETLEN);
        return MD_error(s, MD_ERR_CBSHARE);
      }
      if (len > e->attrib.max) {
        void *v;
        if ((e->attrib.access & MD_SETMAX) != MD_SETMAX) {
          COND((e->attrib.access & MD_SETMAX) != MD_SETMAX);
          return MD_error(s, MD_ERR_CBSHARE);
        }
        v = e->reallocmem(e->buf, len * MD_SIZEOF(e->attrib.type));
        /* note:  newlen > 0, so NULL is an error */
        if (v == NULL) {
          COND(v == NULL);
          return MD_error(s, MD_ERR_MEMALLOC);
        }
        /* update SHARE access */
        e->attrib.access &= ~MD_SHARE;  /* clear SHARE */
        e->buf = v;
        e->attrib.max = len;
      }
      e->attrib.len = len;
      e->attrib.access |= MD_MODIFY;    /* set MODIFY "dirty bit" */
            /* set MODIFY only to tell engine that length changed */

      /* since length attribute changed, must set CALLBACK_SETUP flag(s) */
      if (MD_engdata_cbsetup(s, e)) return MD_FAIL;
    }

    /*
     * copy callback subarray into engine buffer
     * code below based on MD_writesub()
     */
    memcpy(((char *)(e->buf)) + p->first * MD_SIZEOF(e->attrib.type),
        p->data, p->nelems * MD_SIZEOF(e->attrib.type));

    /* reset for next callback: data, nelems, first */
    p->nelems = 0;
    p->first = 0;
    p->data = NULL;
  } /* end loop through cbdata */

  return 0;
}


/*
 * front end API
 */

int32 MD_callback(MD_Engine *s,
    int32 (*cb)(void *, MD_Cbdata *, int32, int32),
    void *info, MD_Cbdata *c, int32 clen, int32 stepincr)
{
  MD_Callback *pcb;
  int32 k;
  adt_List *pcbq;
  int32 pcbq_len;

  ASSERT(s != NULL);
  ASSERT(cb != NULL);
  pcbq = &(s->pcbq);
  pcbq_len = (int32) adt_getLengthList(pcbq);

  /* validate arguments */
  if (stepincr <= 0) {
    COND(stepincr <= 0);
    return MD_error(s, MD_ERR_RANGE);
  }

  /* validate callback data */
  for (k = 0;  k < clen;  k++) {
    if (cbdata_validate(s, &c[k], CB)) return MD_FAIL;
  }

  /* allocate MD_Callback node */
  if ((pcb = (MD_Callback *) calloc(1, sizeof(MD_Callback))) == NULL) {
    ERRCOND(pcb == NULL);
    return MD_error(s, MD_ERR_MEMALLOC);
  }

  /* set MD_Callback node fields */
  pcb->funcptr.cb = cb;
  pcb->info = info;
  pcb->cbarg = c;
  pcb->cbarglen = clen;
  pcb->stepincr = stepincr;
  /* callback exec inits nextstepnum if less than stepnum */
  pcb->nextstepnum = MD_INT32_MIN;
  pcb->cbinfonum = pcbq_len;
  pcb->flags = MD_CALLBACK_INIT;

  /* append node to callback list */
  if (adt_appendList(pcbq, &pcb)) {
    /* free new MD_Callback node allocation since we can't keep it */
    free(pcb);
    return MD_error(s, MD_ERR_MEMALLOC);
  }

  /* init this callback */
  if (callback_setup(s, pcb)) return MD_FAIL;

  /* set CBUPDATE flag */
  s->flags |= MD_CBUPDATE;

  return 0;
}


int32 MD_callback_undo(MD_Engine *s,
    int32 (*cb)(void *, MD_Cbdata *, int32, int32))
{
  adt_List *pcbq;
  MD_Callback **p;
  int32 len, src, dest;

  ASSERT(s != NULL);

  /* get callback array and length */
  pcbq = &(s->pcbq);
  p = (MD_Callback **) adt_getDataList(pcbq);
  len = (int32) adt_getLengthList(pcbq);

  /* see if we need to undo all callbacks */
  if (cb == NULL) {
    /* free all MD_Callback nodes */
    for (src = 0;  src < len;  src++) {
      if (callback_remove(s, p[src])) return MD_FAIL;
      free(p[src]);
      p[src] = NULL;
    }
    /* now resize array to length 0 */
    if (adt_resizeList(pcbq, 0)) {
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    return 0;
  }

  /*
   * throw away array entries calling "cb" routine
   *
   * also need to adjust cbinfonum:
   * the following isn't perfect (it could fail if there are more than
   * INT_MAX+2 deletions) but should work in practice and maintain correct
   * order with cbinfonum initialization in MD_callback() above
   */
  for (src = 0, dest = 0;  src < len;  src++) {
    if (cb == p[src]->funcptr.cb) {
      if (callback_remove(s, p[src])) return MD_FAIL;
      /* free each MD_Callback node as we remove it */
      free(p[src]);
      p[src] = NULL;
      continue;
    }
    else if (dest < src) {
      p[dest++] = p[src];
    }
    else {
      dest++;
    }
    /* subtract "len" from "cbinfonum" */
    p[src]->cbinfonum -= len;
  }
  /* new array length is "dest" */
  for (src = 0;  src < dest;  src++) {
    /* add new length back to "cbinfonum" - maintains order relation */
    p[src]->cbinfonum += dest;
  }

  /* adjust array length */
  if (adt_resizeList(pcbq, dest)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }

  /* set CBUPDATE flag - make sure to re-build heap */
  s->flags |= MD_CBUPDATE;

  return 0;
}


int32 MD_fcallback(MD_Engine *s,
    int32 (*fcb)(void *, MD_Cbdata *, int32, int32, double),
    void *info, MD_Cbdata *c, int32 clen)
{
  MD_Callback *pcb;
  int32 k;
  adt_List *pfcb;
  int32 pfcb_len;

  ASSERT(s != NULL);
  ASSERT(fcb != NULL);
  pfcb = &(s->pfcb);
  pfcb_len = (int32) adt_getLengthList(pfcb);

  /* validate callback data */
  for (k = 0;  k < clen;  k++) {
    if (cbdata_validate(s, &c[k], FCB)) return MD_FAIL;
  }

  /* alocate MD_Fcbdata element and set fields */
  if ((pcb = (MD_Callback *) calloc(1, sizeof(MD_Callback))) == NULL) {
    ERRCOND(pcb == NULL);
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  pcb->funcptr.fcb = fcb;
  pcb->info = info;
  pcb->cbarg = c;
  pcb->cbarglen = clen;
  /* (the standard callback fields are not used) */
  pcb->flags = MD_CALLBACK_INIT;

  /* append to force callback list */
  if (adt_appendList(pfcb, &pcb)) {
    /* free new MD_Callback node allocation since we can't keep it */
    free(pcb);
    return MD_error(s, MD_ERR_MEMALLOC);
  }

  /* init this callback */
  if (callback_setup(s, pcb)) return MD_FAIL;
  return 0;
}


int32 MD_fcallback_undo(MD_Engine *s,
    int32 (*fcb)(void *, MD_Cbdata *, int32, int32, double))
{
  adt_List *pfcb;
  MD_Callback **p;
  int32 len, src, dest;

  ASSERT(s != NULL);

  /* get force callback array and length */
  pfcb = &(s->pfcb);
  p = (MD_Callback **) adt_getDataList(pfcb);
  len = (int32) adt_getLengthList(pfcb);

  /* see if we need to undo all force callbacks */
  if (fcb == NULL) {
    /* free all MD_Fcbdata nodes */
    for (src = 0;  src < len;  src++) {
      if (callback_remove(s, p[src])) return MD_FAIL;
      free(p[src]);
      p[src] = NULL;
    }
    /* now resize array to length 0 */ 
    if (adt_resizeList(pfcb, 0)) {
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    return 0;
  }

  /* throw away array entries calling "fcb" routine */
  for (src = 0, dest = 0;  src < len;  src++) {
    if (fcb == p[src]->funcptr.fcb) {
      if (callback_remove(s, p[src])) return MD_FAIL;
      /* free each MD_Fcbdata node as we remove it */
      free(p[src]);
      p[src] = NULL;
      continue;
    }
    else if (dest < src) {
      p[dest++] = p[src];
    }
    else {
      dest++;
    }
  }

  /* new array length is "dest" */
  if (adt_resizeList(pfcb, dest)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  return 0;
}


int32 MD_msgcallback(MD_Engine *s,
    int32 (*msgcb)(void *info, const char *msg, int32 stepnum),
    void *info)
{
  MsgcbInfo m;

  ASSERT(s != NULL);
  ASSERT(msgcb != NULL);

  m.msgcb = msgcb;
  m.info = info;
  /* append to list */
  if (adt_appendList(&(s->msgcb), &m)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  return 0;
}


int32 MD_msgcallback_undo(MD_Engine *s,
    int32 (*msgcb)(void *, const char *, int32))
{
  adt_List *mlist;
  MsgcbInfo *msg;
  int32 msglen, src, dest;

  ASSERT(s != NULL);
  mlist = &(s->msgcb);

  /* see if we need to undo all message callbacks */
  if (msgcb == NULL) {
    if (adt_resizeList(mlist, 0)) {
      return MD_error(s, MD_ERR_MEMALLOC);
    }
    return 0;
  }

  /* get message callback array and length */
  msg = (MsgcbInfo *) adt_getDataList(mlist);
  msglen = (int32) adt_getLengthList(mlist);

  /* throw away array entries calling "msgcb" routine */
  for (src = 0, dest = 0;  src < msglen;  src++) {
    if (msgcb == msg[src].msgcb) {
      continue;
    }
    else if (dest < src) {
      msg[dest++] = msg[src];
    }
    else {
      dest++;
    }
  }

  /* new array length is "dest" */
  if (adt_resizeList(mlist, dest)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  return 0;
}


/*
 * engine API
 */

int32 MD_engdata_cbsetup(MD_Front *s, MD_Engdata *e)
{
  adt_List *cback;
  MD_Callback **cb;
  int32 len, k;

  ASSERT(s != NULL);
  ASSERT(e != NULL);
  cback = &(e->cback);
  cb = (MD_Callback **) adt_getDataList(cback);
  len = (int32) adt_getLengthList(cback);
  for (k = 0;  k < len;  k++) {
    ASSERT(cb != NULL);
    cb[k]->flags |= MD_CALLBACK_SETUP;
  }
  return 0;
}


const MD_Callback **MD_callback_list(MD_Front *s, int32 *listlen)
{
  adt_List *pcbq;
  int32 pcbq_len;
  const MD_Callback **pcbq_data;

  ASSERT(s != NULL);
  ASSERT(listlen != NULL);
  pcbq = &(s->pcbq);
  pcbq_len = (int32) adt_getLengthList(pcbq);
  pcbq_data = (const MD_Callback **) adt_getDataList(pcbq);
  *listlen = pcbq_len;
  return (pcbq_len > 0 ? pcbq_data : NULL);
}


const MD_Callback **MD_fcallback_list(MD_Front *s, int32 *listlen)
{
  adt_List *pfcb;
  int32 pfcb_len;
  const MD_Callback **pfcb_data;

  ASSERT(s != NULL);
  ASSERT(listlen != NULL);
  pfcb = &(s->pfcb);
  pfcb_len = (int32) adt_getLengthList(pfcb);
  pfcb_data = (const MD_Callback **) adt_getDataList(pfcb);
  *listlen = pfcb_len;
  return (pfcb_len > 0 ? pfcb_data : NULL);
}


int32 MD_ready_callback(MD_Front *s)
{
  adt_List *pcbq;
  int32 pcbq_len;
  const MD_Callback **pcbq_data;

  ASSERT(s != NULL);
  pcbq = &(s->pcbq);
  pcbq_len = (int32) adt_getLengthList(pcbq);
  pcbq_data = (const MD_Callback **) adt_getDataList(pcbq);
  /* are we ready to process a callback right now? */
  return pcbq_len > 0
    && ( pcbq_data[0]->nextstepnum <= s->stepnum
       || (s->flags & (MD_CBRESET | MD_CBUPDATE)) );
}


/******************************************************************************
 *
 * define internal routines for MD_exec_callback()
 *
 *****************************************************************************/

/* prototypes */
static void update_nextstepnum(MD_Front *);
#ifdef DEBUG_SUPPORT
static void heap_check(MD_Front *);
#endif
static void heap_build(MD_Front *);
static void heap_update(MD_Front *);


/* initialize "nextstepnum" field for standard callbacks */
void update_nextstepnum(MD_Front *s)
{
  adt_List *pcbq;
  MD_Callback **c;
  int32 clen, k;

  ASSERT(s != NULL);
  pcbq = &(s->pcbq);
  c = (MD_Callback **) adt_getDataList(pcbq);
  clen = (int32) adt_getLengthList(pcbq);

  if ((s->flags & (MD_CBRESET | MD_CBFIRST)) == (MD_CBRESET | MD_CBFIRST)) {
    /* reset all callbacks to be processed now */
    for (k = 0;  k < clen;  k++) {
      c[k]->nextstepnum = s->stepnum;
    }
  }
  else if (s->flags & MD_CBRESET) {
    /* reset all callbacks to be processed after "stepincr" steps */
    for (k = 0;  k < clen;  k++) {
      c[k]->nextstepnum = s->stepnum + c[k]->stepincr;
    }
  }
  else if ((s->flags & (MD_CBUPDATE | MD_CBFIRST))
      == (MD_CBUPDATE | MD_CBFIRST)) {
    /* set only the new callbacks to be processed now */
    for (k = 0;  k < clen;  k++) {
      if (c[k]->nextstepnum < s->stepnum) {
        c[k]->nextstepnum = s->stepnum;
      }
    }
  }
  else {  /* only CBUPDATE is set */
    /* set only the new callbacks to be processed after "stepincr" steps */
    for (k = 0;  k < clen;  k++) {
      if (c[k]->nextstepnum < s->stepnum) {
        c[k]->nextstepnum = s->stepnum + c[k]->stepincr;
      }
    }
  }
}


#ifdef DEBUG_SUPPORT
void heap_check(MD_Front *s)
{
  adt_List *pcbq;
  MD_Callback **c;
  int32 clen, k, child;

  ASSERT(s != NULL);
  pcbq = &(s->pcbq);
  c = (MD_Callback **) adt_getDataList(pcbq);
  clen = (int32) adt_getLengthList(pcbq);

#ifdef DEBUG_WATCH
  /* print out heap */
  printf("\n*** callback heap ***\n");
  for (k = 0;  k < clen;  k++) {
    printf("*** c[%d]:  nextstepnum=%d  cbinfonum=%d  stepincr=%d\n",
        k, c[k]->nextstepnum, c[k]->cbinfonum, c[k]->stepincr);
  }
  printf("\n");
#endif
  /* for debugging purposes, verify heap order property */
  for (k = 0;  k < clen;  k++) {
    child = 2*k + 1;
    if (child < clen) {
      ASSERT(c[k]->nextstepnum < c[child]->nextstepnum
          || (c[k]->nextstepnum == c[child]->nextstepnum
            && c[k]->cbinfonum < c[child]->cbinfonum));
    }
    child++;
    if (child < clen) {
      ASSERT(c[k]->nextstepnum < c[child]->nextstepnum
          || (c[k]->nextstepnum == c[child]->nextstepnum
            && c[k]->cbinfonum < c[child]->cbinfonum));
    }
  }
}
#endif


/*
 * setup priority queue (build binary heap)
 *
 * this is an O(n) algorithm (n callbacks in heap):
 * loop backwards through heap and do percolate
 * down to head node of each subheap to arrange the
 * subheap into proper order
 */
void heap_build(MD_Front *s)
{
  adt_List *pcbq;
  MD_Callback **c;
  MD_Callback *tmp;
  int32 clen, j, k, child, sibling;

  ASSERT(s != NULL);
  pcbq = &(s->pcbq);
  c = (MD_Callback **) adt_getDataList(pcbq);
  clen = (int32) adt_getLengthList(pcbq);

  for (j = (clen >> 1) - 1;  j >= 0;  j--) {  /* (clen/2 - 1) */
    /* percolate node j down */
    tmp = c[j];
    /* create a "hole" in heap where k is */
    k = j;
    child = (k << 1) + 1;  /* (2*k + 1) */
    /* while more children in binary heap */
    while (child < clen) {
      /* find smallest between two children */
      sibling = child + 1;
      if (sibling < clen
          && (c[sibling]->nextstepnum < c[child]->nextstepnum
            || (c[sibling]->nextstepnum == c[child]->nextstepnum
              && c[sibling]->cbinfonum < c[child]->cbinfonum))) {
        child = sibling;
      }
      /* break out of loop if tmp is smaller than smallest child */
      if (tmp->nextstepnum < c[child]->nextstepnum
          || (tmp->nextstepnum == c[child]->nextstepnum
            && tmp->cbinfonum < c[child]->cbinfonum)) {
        break;
      }
      /* otherwise promote smallest child */
      c[k] = c[child];
      /* the hole is now where smallest child had been */
      k = child;
      child = (k << 1) + 1;  /* (2*k + 1) */
    }
    /* place tmp node to maintain heap order */
    c[k] = tmp;
  }
#ifdef DEBUG_SUPPORT
  heap_check(s);
#endif
}


/*
 * update priority queue (percolate down head node)
 *
 * this is a O(log n) algorithm (n callbacks in heap):
 * create hole at head node, while this tmp node is
 * bigger than both children, promote smallest child
 * and slide hole down
 */
void heap_update(MD_Front *s)
{
  adt_List *pcbq;
  MD_Callback **c;
  MD_Callback *tmp;
  int32 clen, k, child, sibling;

  ASSERT(s != NULL);
  pcbq = &(s->pcbq);
  c = (MD_Callback **) adt_getDataList(pcbq);
  clen = (int32) adt_getLengthList(pcbq);

  tmp = c[0];
  /* create a "hole" in heap where k is */
  k = 0;
  child = (k << 1) + 1;  /* (2*k + 1) */
  /* while more children in binary heap */
  while (child < clen) {
    /* find smallest between two children */
    sibling = child + 1;
    if (sibling < clen
        && (c[sibling]->nextstepnum < c[child]->nextstepnum
          || (c[sibling]->nextstepnum == c[child]->nextstepnum
            && c[sibling]->cbinfonum < c[child]->cbinfonum))) {
      child = sibling;
    }
    /* break out of loop if tmp is smaller than smallest child */
    if (tmp->nextstepnum < c[child]->nextstepnum
        || (tmp->nextstepnum == c[child]->nextstepnum
          && tmp->cbinfonum < c[child]->cbinfonum)) {
      break;
    }
    /* otherwise promote smallest child */
    c[k] = c[child];
    /* the hole is now where smallest child had been */
    k = child;
    child = (k << 1) + 1;  /* (2*k + 1) */
  }
  /* place tmp node to maintain heap order */
  c[k] = tmp;
#ifdef DEBUG_SUPPORT
  heap_check(s);
#endif
}

/******************************************************************************
 *
 * end of internal routines for MD_exec_callback()
 *
 *****************************************************************************/


/*
 * standard callbacks
 * engine must increment stepnum and process callbacks when requested
 */
int32 MD_exec_callback(MD_Front *s)
{
  adt_List *pcbq;
  MD_Callback **c;
  int32 clen, retval;

  ASSERT(s != NULL);
  pcbq = &(s->pcbq);
  c = (MD_Callback **) adt_getDataList(pcbq);
  clen = (int32) adt_getLengthList(pcbq);

  /* make sure there really are registered callbacks */
  if (clen == 0) {
    return 0;
  }
  ASSERT(clen > 0);

  /* see if any initialization needs to be done */
  if (s->flags & (MD_CBRESET | MD_CBUPDATE)) {

    /* init/update next step number for callbacks */
    update_nextstepnum(s);

    /* setup correct priority queue ordering */
    heap_build(s);

    /* clear initialization flags */
    s->flags &= ~(MD_CBRESET | MD_CBUPDATE | MD_CBFIRST);
  }

  /* validate callback processing */
  if (c[0]->nextstepnum < s->stepnum) {
    /*
     * this indicates an error by the engine
     * MD_exec_callback() was not called when it should have been
     */
    COND(c[0]->nextstepnum < s->stepnum);
    return MD_error(s, MD_ERR_CHECK);
  }

  /* process callbacks */
  while (c[0]->nextstepnum == s->stepnum) {

    /* setup callback */
    if (c[0]->flags & MD_CALLBACK_SETUP) {
      if (callback_setup(s, c[0])) return MD_FAIL;
    }

    /* invoke the callback */
    ASSERT(c[0]->funcptr.cb != NULL);
    retval = c[0]->funcptr.cb(c[0]->info, c[0]->cbarg, c[0]->cbarglen,
        s->stepnum);
    if (retval != 0) {
      COND(retval != 0);
      /* error returned by callback, terminate MD_run() */
      return MD_error(s, MD_ERR_CALLBACK);
    }

    /* handle any shared buffers */
    if (c[0]->flags & MD_CALLBACK_SHARE) {
      if (callback_share(s, c[0])) return MD_FAIL;
    }

    /* increment "nextstepnum" */
    c[0]->nextstepnum += c[0]->stepincr;

    /* update priority queue (percolate down head node) */
    heap_update(s);

  } /* end loop over all callbacks */
  return 0;
}


int32 MD_ready_fcallback(MD_Front *s)
{
  ASSERT(s != NULL);
  return adt_getLengthList(&(s->pfcb)) > 0;
}


/*
 * execute force callbacks
 * should be called by engine as part of every force evaluation
 */
int32 MD_exec_fcallback(MD_Front *s, double timestepfrac)
{
  adt_List *pfcb;
  MD_Callback **f;
  int32 flen, k, retval;

  ASSERT(s != NULL);
  pfcb = &(s->pfcb);
  f = (MD_Callback **) adt_getDataList(pfcb);
  flen = (int32) adt_getLengthList(pfcb);

  /* loop over all force callbacks */
  for (k = 0;  k < flen;  k++) {

    /* setup callback */
    if (f[k]->flags & MD_CALLBACK_SETUP) {
      if (callback_setup(s, f[k])) return MD_FAIL;
    }

    /* invoke the force callback */
    ASSERT(f[k]->funcptr.fcb != NULL);
    retval = f[k]->funcptr.fcb(f[k]->info, f[k]->cbarg, f[k]->cbarglen,
        s->stepnum, timestepfrac);
    if (retval != 0) {
      COND(retval != 0);
      /* error returned by callback, terminate MD_run() */
      return MD_error(s, MD_ERR_CALLBACK);
    }

    /* handle any shared buffers */
    if (f[k]->flags & MD_CALLBACK_SHARE) {
      if (callback_share(s, f[k])) return MD_FAIL;
    }

  } /* end loop over all force callbacks */
  return 0;
}


int32 MD_ready_msgcallback(MD_Front *s)
{
  ASSERT(s != NULL);
  return adt_getLengthList(&(s->msgcb)) > 0;
}


int32 MD_exec_msgcallback(MD_Front *s, const char *msg)
{
  adt_List *mlist;
  MsgcbInfo *m;
  int32 mlen, k, retval;

  ASSERT(s != NULL);
  mlist = &(s->msgcb);
  m = (MsgcbInfo *) adt_getDataList(mlist);
  mlen = (int32) adt_getLengthList(mlist);
  for (k = 0;  k < mlen;  k++) {
    /* invoke the message callback */
    retval = m[k].msgcb(m[k].info, msg, s->stepnum);
    if (retval != 0) {
      COND(retval != 0);
      /* error returned by callback, terminate MD_run() */
      return MD_error(s, MD_ERR_CALLBACK);
    }
  }
  return 0;
}


/*
 * internal methods
 */

int32 MD_init_callback(MD_Front *s)
{
  ASSERT(s != NULL);
  /* init callback handling containers */
  if (adt_initializeList(&(s->pcbq), sizeof(MD_Callback *), 0, NULL)
      || adt_initializeList(&(s->pfcb), sizeof(MD_Callback *), 0, NULL)
      || adt_initializeList(&(s->msgcb), sizeof(MsgcbInfo), 0, NULL)) {
    return MD_error(s, MD_ERR_MEMALLOC);
  }
  /* init run flags with CBRESET to init callbacks on first pass */
  s->flags |= MD_CBRESET;
  return 0;
}


void MD_done_callback(MD_Front *s)
{
  adt_List *list;
  MD_Callback **p;
  int32 k, len;

  ASSERT(s != NULL);
  /* need to free memory for each pcbq node */
  list = &(s->pcbq);
  p = (MD_Callback **) adt_getDataList(list);
  len = (int32) adt_getLengthList(list);
  for (k = 0;  k < len;  k++) {
    free(p[k]);
  }
  /* need to free memory for each pfcb node */
  list = &(s->pfcb);
  p = (MD_Callback **) adt_getDataList(list);
  len = (int32) adt_getLengthList(list);
  for (k = 0;  k < len;  k++) {
    free(p[k]);
  }
  /* free callback handling containers */
  adt_cleanupList(&(s->pcbq));
  adt_cleanupList(&(s->pfcb));
  adt_cleanupList(&(s->msgcb));
}
