/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    force.h
 * @brief   Force evaluation library.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c Force_tag class evaluates the basic force field for biomolecules.
 * Boundary conditions may be nonperiodic or orthogonal periodic
 * (or semi-periodic) cells.  Nonbonded interactions are treated using
 * a cutoff distance.  Helper class @c ForceParam_tag is used for
 * initialization.  Two helper classes @c ForceEnergy_tag and
 * @c ForceResult_tag provide computed energies and forces, respectively.
 *
 * The force evaluation can compute subsets from the following:
 * @li bonds
 * @li angles
 * @li dihedrals
 * @li impropers
 * @li electrostatics - uses a cutoff distance, either shifted or
 *   truncated
 * @li van der Waals - uses cutoff and switching distances with the
 *   same switching function as used by NAMD 2.5
 * @li boundary restraints - spherical and cylindrical restraints as
 *   computed by NAMD, intended for nonperiodic or semi-periodic systems
 * @li nonbonded exclusions - subtracts out excluded interactions
 *   (instead of accumulating the nonbonded pairs to be included)
 *   for either or both electrostatics and van der Waals
 * @li full direct electrostatics - this is the @f$ O(N^2) @f$ approach;
 *   for periodic systems it uses nearest image convention
 *
 * The @c Force_tag class is used with three helper classes:
 * @li @c ForceParam_tag is configured at the beginning of the simulation
 *   to describe which aspects of the force field are to be evaluated,
 *   along with force field parameter and molecule topology information
 * @li @c ForceEnergy_tag returns the total potential energy along with
 *   parts of the potential energy (e.g. bond, angle, van der Waals)
 * @li @c ForceResult_tag returns the total forces for each atom of the
 *   system and optionally the separated forces for each atom
 *   (e.g. bond, angle, van der Waals)
 *
 * Periodic boundaries are computed using
 * an array of the true atomic positions with an array of the wrapped
 * offset optionally returned, so that the atom position that lies
 * within the cell (see @c force_compute() below) is always:
 * @verbatim
   pos[i] + wrap[i]
   @endverbatim
 *
 * The units are as outlined by the MDAPI documentation:
 * @li distance in Angstroms
 * @li time in femtoseconds
 * @li energy in kcal/mol
 * @li force in kcal/mol/Angstrom
 *
 * The internals of the force field parameter data types and topology
 * data types are discussed in detail by the MDAPI documentation.
 *
 * Caveats:
 * @li Full electrostatics are not computed (at least not with a fast method).
 *   To use with an external electrostatics library:  turn @e off the
 *   electrostatics capability.  Instead you might want to turn @e on
 *   nonbonded exclusions for electrostatics.
 * @li Periodic cells must be at least twice the length of the cutoff
 *   in each dimension.  (This guideline seems preferable in practice.)
 * @li Periodic cells must be orthogonal.  Cell resizing requires
 *   reinitializing the @c Force_tag class, so this does not provide good
 *   support for constant pressure simulations.
 */
#ifndef FORCE_H
#define FORCE_H

#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**@brief Used to configure @c Force_tag class.
   *
   * User sets the internal fields of this class in order to configure
   * the evaluation performed by the @c Force_tag class.  The
   * @c ForceParam_tag class is intended to be setup once and used to
   * initialize the @c Force_tag class.
   *
   * The user provides and retains ownership of memory buffer space for all
   * arrays.  The force field parameter and topology arrays all follow the
   * MDAPI guidelines as defined by @c mdtypes.h file.  In particular,
   * it is expected that the proper cross-indexing between topology and
   * force field parameter arrays have been established.  The best way to
   * do this is to use the MDIO library for reading the relevant force
   * field parameter and topology files.
   *
   * The parameter @c atom_len indicates the number of atoms in the system,
   * providing the expected array length of the @c ForceResult_tag arrays
   * and the @c pos and @c wrap arrays passed to @c force_compute().
   *
   * The boundary restraint and nonbonded parameters follow the NAMD
   * conventions, as described in its documentation.
   *
   * The @c flags indicates what parts of the force field are to be
   * evaluated.  It is assigned a bitwise ORed collection of
   * @c ForceFlags_tag constants.
   */
  typedef struct ForceParam_tag {
    /* Force field parameters as defined by the MDAPI mdtypes.h file. */
    MD_AtomPrm *atomprm;   /**< Array of atom parameters. */
    MD_BondPrm *bondprm;   /**< Array of bond parameters. */
    MD_AnglePrm *angleprm; /**< Array of angle parameters. */
    MD_TorsPrm *dihedprm;  /**< Array of dihedral parameters. */
    MD_TorsPrm *imprprm;   /**< Array of improper parameters. */
    MD_NbfixPrm *nbfixprm; /**< Array of nonbonded parameters to override
                             the combined pairs of atom parameters. */
    int32 atomprm_len;     /**< Length of atom parameters array. */
    int32 bondprm_len;     /**< Length of bond parameters array. */
    int32 angleprm_len;    /**< Length of angle parameters array. */
    int32 dihedprm_len;    /**< Length of dihedral parameters array. */
    int32 imprprm_len;     /**< Length of improper parameters array. */
    int32 nbfixprm_len;    /**< Length of nonbonded parameters array. */

    /* Topology parameters as defined by the MDAPI mdtypes.h file. */
    MD_Atom *atom;         /**< Array of atoms (contains mass and charge). */
    MD_Bond *bond;         /**< Array of bonds. */
    MD_Angle *angle;       /**< Array of angles. */
    MD_Tors *dihed;        /**< Array of dihedrals. */
    MD_Tors *impr;         /**< Array of impropers. */
    MD_Excl *excl;         /**< Array of nonbonded exclusions. */
    int32 atom_len;        /**< Length of atom array
                             (number of atoms in system). */
    int32 bond_len;        /**< Length of bond array. */
    int32 angle_len;       /**< Length of angle array. */
    int32 dihed_len;       /**< Length of dihedral array. */
    int32 impr_len;        /**< Length of improper array. */
    int32 excl_len;        /**< Length of nonbonded exclusion array. */

    /* (semi-)periodic orthogonal cell size */
    double xlen;           /**< Size of periodic cell in x direction. */
    double ylen;           /**< Size of periodic cell in y direction. */
    double zlen;           /**< Size of periodic cell in z direction. */
    MD_Dvec center;        /**< Center of cell. */

    /* boundary conditions defined around center */
    double radius1;        /**< Distance at which first potential boundary
                             takes effect, measured as radius from center. */
    double radius2;        /**< Distance at which second potential boundary
                             takes effect, measured as radius from center. */
    double length1;        /**< For cylindrical restraints: distance at
                             which first potential takes effect along
                             cylinder axis. */
    double length2;        /**< For cylindrical restraints: distance at
                             which second potential takes effect along
                             cylinder axis. */
    double konst1;         /**< Constant for first harmonic potential. */
    double konst2;         /**< Constant for second harmonic potential. */
    int32 exp1;            /**< Exponent for first harmonic potential.
                             Should be an even positive integer. */
    int32 exp2;            /**< Exponent for second harmonic potential.
                             Should be an even positive integer. */

    /* nonbonded parameters */
    double cutoff;         /**< Cutoff distance used to define both
                             electrostatics and van der Waals interactions. */
    double elec_cutoff;    /**< Cutoff distance for electrostatics. */
    double vdw_cutoff;     /**< Cutoff distance for van der Waals. */
    double switchdist;     /**< Switching distance for van der Waals
                             (needs to be strictly less than the cutoff). */
    double elec_const;     /**< Constant for electrostatic interactions,
                             typically should be defined using @c MD_COULOMB
                             from MDAPI @c mdtypes.h file. */
    double dielectric;     /**< Dielectric constant, typically should be
                             set to 1.0. */
    double scaling14;      /**< Extra multiplicative constant for
                             electrostatic interactions between 1-4 pairs,
                             applied only if @c FORCE_EXCL_SCAL14 flag is
                             set. */

    /* flags */
    int32 flags;           /**< Indicates what parts of force field are to be
                             evaluated.  See @c ForceFlags_tag for details. */
  } ForceParam;


  /**@brief Flags to indicate which parts of the force field are to be
   * evaluated.
   *
   * Flags should be bitwise ORed together, used to initialize
   * @c ForceParam_tag::flags field.
   *
   * There are some values that collect commonly used flags:
   * @li @c FORCE_MD_VACUUM is good for nonperiodic systems without
   *   boundary restraints (you can OR in desired restraints flag).
   * @li @c FORCE_MD_CELL is good for fully periodic systems.
   *
   * These both perform continuous cutoff nonbonded evaluations
   * using scaled 1-4 exclusion policy.
   *
   * If you want to use an external electrostatic library, you can use
   * the @c Force_tag class  to subtract out electrostatic exclusions by
   * defining the following flags:
   * @verbatim
     FORCE_ALL | FORCE_EXCL_SCAL14 | FORCE_ELEC_EXCL | FORCE_SWITCH
     @endverbatim
   * to compute all bonded interactions and smoothly switched van der Waals
   * interactions using the scaled 1-4 exclusion policy;
   * you can also include with this flags to define periodic
   * boundary conditions or nonperiodic boundary restraints.
   */
  enum ForceFlags_tag {
    /* force types */
    FORCE_BOND        = 0x000001,
    /**< Evaluate bond interactions. */
    FORCE_ANGLE       = 0x000002,
    /**< Evaluate angle interactions. */
    FORCE_DIHED       = 0x000004,
    /**< Evaluate dihedral interactions. */
    FORCE_IMPR        = 0x000008,
    /**< Evaluate improper interactions. */
    FORCE_ELEC        = 0x000010,
    /**< Evaluate electrostatic interactions. */
    FORCE_VDW         = 0x000020,
    /**< Evaluate van der Waals interactions. */
    FORCE_BONDED      = FORCE_BOND | FORCE_ANGLE | FORCE_DIHED | FORCE_IMPR,
    /**< Collection of flags: evaluate all types of bonded interactions. */
    FORCE_NONBONDED   = FORCE_ELEC | FORCE_VDW,
    /**< Collection of flags: evaluate all types of nonbonded interactions. */
    FORCE_ALL         = FORCE_BONDED | FORCE_NONBONDED,
    /**< Collection of flags: evaluate all types of bonded and nonbonded
     * interactions. */
    FORCE_ELEC_DIRECT = 0x000040,
    /**< Perform direct evaluation of electrostatic interactions. */
    FORCE_VDW_DIRECT  = 0x000080,
    /**< Perform direct evaluation of van der Waals interactions. */
    FORCE_DIRECT      = FORCE_ELEC_DIRECT | FORCE_VDW_DIRECT,
    /**< Collection of flags: perform direct evaluation of all nonbonded
     * interactions. */
    FORCE_MASK_TYPE   = 0x0000FF,
    /**< (internal use only) */

    /* periodicity */
    FORCE_NONPERIODIC = 0x000000,
    /**< Indicates nonperiodic system. */
    FORCE_X_PERIODIC  = 0x000100,
    /**< Indicates periodicity in x-direction. */
    FORCE_Y_PERIODIC  = 0x000200,
    /**< Indicates periodicity in y-direction. */
    FORCE_Z_PERIODIC  = 0x000400,
    /**< Indicates periodicity in z-direction. */
    FORCE_PERIODIC    = FORCE_X_PERIODIC | FORCE_Y_PERIODIC | FORCE_Z_PERIODIC,
    /**< Collection of flags: indicates fully periodic system. */
    FORCE_MASK_PERIOD = 0x000F00,
    /**< (internal use only) */

    /* boundary restraints (for nonperiodic systems) */
    FORCE_SPHERE      = 0x001000,
    /**< Use spherical boundary restraints. */
    FORCE_X_CYLINDER  = 0x002000,
    /**< Use cylindrical boundary restraints with axis aligned in
     * x-direction. */
    FORCE_Y_CYLINDER  = 0x004000,
    /**< Use cylindrical boundary restraints with axis aligned in
     * y-direction. */
    FORCE_Z_CYLINDER  = 0x008000,
    /**< Use cylindrical boundary restraints with axis aligned in
     * z-direction. */
    FORCE_MASK_BC     = 0x00F000,
    /**< (internal use only) */

    /* nonbonded exclusion policy */
    FORCE_EXCL_NONE   = 0x000000,
    /**< Do not exclude any nonbonded pairs except for those indicated
     * explicitly within the @c ForceParam_tag::excl array. */
    FORCE_EXCL_12     = 0x010000,
    /**< Exclude 1-2 interactions (atoms joined by spring bond). */
    FORCE_EXCL_13     = 0x020000,
    /**< Exclude 1-3 interactions (atoms joined by pair of spring bonds),
     * which includes all angles. */
    FORCE_EXCL_14     = 0x030000,
    /**< Exclude 1-4 interactions (atoms joined by trio of spring bonds),
     * which includes all dihedrals and impropers. */
    FORCE_EXCL_SCAL14 = 0x040000,
    /**< Exclude 1-3 interactions but redefine 1-4 interactions using
     * alternate van der Waals parameters and electrostatic 1-4 scaling
     * constant.  (This is the commonly used standard for modern systems.) */
    FORCE_MASK_EXCL   = 0x0F0000,
    /**< (internal use only) */

    /* nonbonded cutoff continuity and exclusions */
    FORCE_SMOOTH      = 0x100000,
    /**< Apply shifting to electrostatics potential so that cutoff
     * interactions are smooth. */
    FORCE_SWITCH      = 0x200000,
    /**< Apply switching function, piecewise defined from cutoff to
     * switching distance, to van der Waals potential so that cutoff
     * interactions are smooth. */
    FORCE_CONTINUOUS  = FORCE_SMOOTH | FORCE_SWITCH,
    /**< Collection of flags: apply electrostatic smoothing and
     * van der Waals switching. */
    FORCE_ELEC_EXCL   = 0x400000,
    /**< Do not evaluate electrostatic interactions.
     * Instead subtract out nonbonded exclusions for electrostatics. */
    FORCE_VDW_EXCL    = 0x800000,
    /**< Do not evaluate van der Waals interactions.
     * Instead subtract out nonbonded exclusions for van der Waals. */
    FORCE_EXCL        = FORCE_ELEC_EXCL | FORCE_VDW_EXCL,
    /**< Collection of flags: subtract out both electrostatics and
     * van der Waals exclusions. */
    FORCE_MASK_CUTOFF = 0xF00000,
    /**< (internal use only) */

    FORCE_MD_VACUUM   = FORCE_ALL | FORCE_EXCL_SCAL14 | FORCE_CONTINUOUS,
    /**< Collection of flags:  good default for cutoff MD in a vacuum,
     * scaled 1-4 exclusion policy, no boundary restraints. */

    FORCE_MD_CELL     = FORCE_MD_VACUUM | FORCE_PERIODIC
    /**< Collection of flags:  good default for cutoff MD with periodic
     * boundaries, scaled 1-4 exclusion policy. */
  };


  enum {
    FORCE_FAIL = -1  /**< Return value from failed function call. */
  };


  /**@brief Container class returning potential energies.
   *
   * User-supplied container class that returns potential energies
   * from @c force_compute() routine.  It is convenient to separate
   * the energy data from the force results.  Each respective value
   * is meaningful only if the @c ForceParam_tag::flags has indicated
   * that potential be evaluated.
   */
  typedef struct ForceEnergy_tag {
    double pe;     /**< Total potential energy, sum of indicated potentials. */
    double bond;   /**< Potential energy from spring bonds. */
    double angle;  /**< Potential energy from angle bonds. */
    double dihed;  /**< Potential energy from dihedrals. */
    double impr;   /**< Potential energy from impropers. */
    double elec;   /**< Potential energy from electrostatics. */
    double vdw;    /**< Potential energy from van der Waals. */
    double bound;  /**< Potential energy from boundary restraints. */
  } ForceEnergy;


  /**@brief Container class returning atomic forces.
   *
   * User-supplied container class that returns atomic forces from
   * @c force_compute() routine.  It is convenient to separate the
   * force results from the energy data.  The arrays (if defined)
   * must have enough buffer space for @c ForceParam_tag::atom_len
   * atoms.  The caller must perform memory management for these
   * array buffers.
   *
   * The @c ForceResult_tag::f array must be supplied.  All of the
   * others are optional.  If a pointer is not @c NULL and the
   * @c ForceParam_tag::flags has indicated the evaluation of that
   * corresponding force type, then that array is expected to have
   * space for @c ForceParam_tag::atom_len atoms and will receive those
   * partial force contributions during the call to @c force_compute().
   */
  typedef struct ForceResult_tag {
    MD_Dvec *f;        /**< Total force, sum of indicated forces. */
    MD_Dvec *f_bond;   /**< Force from spring bonds. */
    MD_Dvec *f_angle;  /**< Force from angle bonds. */
    MD_Dvec *f_dihed;  /**< Force from dihedrals. */
    MD_Dvec *f_impr;   /**< Force from impropers. */
    MD_Dvec *f_elec;   /**< Force from electrostatics. */
    MD_Dvec *f_vdw;    /**< Force from van der Waals. */
    MD_Dvec *f_bound;  /**< Force from boundary restraints. */
  } ForceResult;


  /**@brief (For internal use only.)
   */
  typedef struct ForceCell_tag {
    /* cells for geometric hashing, implement as cursor link list */
    int32 head;          /* index of first atom in this cell */
    int32 cnt;           /* count number of atoms in this cell */
    int32 nnbrs;         /* length of neighbor cell list */
    int32 nbr[14];       /* up to half-shell of neighbors, including self */
    MD_Dvec offset[14];  /* offset for neighbors that are periodic images */
  } ForceCell;


  /**@brief Force evaluation class.
   *
   * Members should be treated as private.
   */
  typedef struct Force_tag {
    ForceParam *param;
    ForceEnergy *energy;
    ForceResult *result;

    double elec_const;
    double inv_elec_cutoff2;
    double switchdist2;
    double inv_denom_switch;
    double elec_cutoff2;
    double vdw_cutoff2;

    double *vdwtable;

    MD_Dvec lo;
    double inv_cutoff;
    double inv_xcellsize;
    double inv_ycellsize;
    double inv_zcellsize;

    MD_Dvec *wrap;

    int32 *next;
    ForceCell *cell;
    int32 ncells;
    int32 nxcells;
    int32 nycells;
    int32 nzcells;
    int32 is_resize;
    int32 is_xresize;
    int32 is_yresize;
    int32 is_zresize;

    /* nonbonded exclusion lists */
    int32 **excl_list;
    int32 **scaled14_list;

    /* helper arrays for building exclusion lists */
    int32 **exclx;     /* explicit exclusions (from MD_Excl) */
    int32 **excl12;    /* lists for 1-2 exclusions (from MD_Bond) */
    int32 **excl13;    /* lists for 1-3 exclusions */
    int32 **excl14;    /* lists for 1-4 exclusions */
    int32 **scaled14;  /* lists for scaled 1-4 exclusions */
    int32 *lenx;       /* counters for exclx lists */
    int32 *len12;      /* counters for excl12 lists */
    int32 *len13;      /* counters for excl13 lists */
    int32 *accum;      /* for merging lists */
    int32 *dest;       /* for merging lists */

    /* for boundary conditions */
    int32 is_bcterm1;
    int32 is_bcterm2;
    double sq_minradius;
    double minlength;

  } Force;


  /* force interface routines */

  /**@brief Constructor.
   *
   * Initializes a @c Force_tag object.
   * Must call this routine first.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_init(Force *);

  /**@brief Setup the @c Force_tag object.
   *
   * @param[in] fp  The @c ForceParam_tag object used for configuration.
   * @param[in] fe  The @c ForceEnergy_tag container into which energies
   *   will be returned.
   * @param[in] fr  The @c ForceResult_tag container into which atomic
   *   forces will be returned.
   *
   * Must call this routine once before @c force_compute().
   * The user should not change these objects, particularly not the
   * @c ForceParam_tag object, after calling this routine.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_setup(Force *, ForceParam *fp, ForceEnergy *fe, ForceResult *fr);

  /**@brief Perform force evaluation.
   *
   * @param[in] pos       The atomic positions supplied by user.
   * @param[in,out] wrap  Updated offset data needed to wrap the actual
   *   atomic position back to its "periodic image"  into the periodic cell.
   *
   * Evaluate the force.  This routine is to be called for each time step
   * of a molecular dynamics simulation.
   *
   * The convention here is that @c pos[i] changes "smoothly" during
   * simulation, perhaps wandering outside the periodic cell, with the
   * atomic coordinate @c pos[i]+wrap[i] always mapping back into the cell.
   * The @c wrap array is optional, but if supplied it is expected to be
   * of length @c ForceParam_tag::atom_len, and it should either be
   * initialized to all zeros or the result of some previous call to
   * @c force_compute().  The @c wrap array is unused for nonperiodic
   * systems.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute(Force *, const MD_Dvec *pos, MD_Dvec *wrap);

  /**@brief Destructor.
   *
   * Frees memory allocations made internally during @c force_setup().
   * Call when finished using @c Force_tag object.
   */
  void force_done(Force *);


  /* provide access to exclusion lists */

  /**@brief Return exclusion lists.
   *
   * This returns the exclusion lists created during @c force_setup().
   * The format is a jagged array of arrays.  The first array is indexed
   * for each atom, and the second array is an ordered list of atom numbers
   * indicating which pairs are to be excluded.  Each of these second
   * arrays ends with a sentinel value greater than or equal to the number
   * of atoms.  The atom numbering is C-style beginning with 0.
   */
  int32 **force_excl_list(const Force *);

  /**@brief Return scaled 1-4 exclusion lists.
   *
   * The same format as returned by @c force_excl_list(), except that this
   * represents only the atomic pairs participating in scaled 1-4
   * interactions.  This list is not meaningful unless the
   * @c ForceParam_tag::flags has indicated @c FORCE_EXCL_SCAL14.
   */
  int32 **force_scaled14_list(const Force *);


  /* setup subroutines */
  int force_setup_bonded(Force *);
  int force_setup_nonbonded(Force *);
  int force_setup_nonbonded_cells(Force *);
  int force_setup_boundary(Force *);

  /* compute subroutines */
  int force_compute_bonded(Force *, const MD_Dvec *pos);
  int force_compute_nonbonded(Force *, const MD_Dvec *pos, MD_Dvec *wrap);
  int force_compute_boundary(Force *, const MD_Dvec *pos, const MD_Dvec *wrap);


#ifdef __cplusplus
}
#endif

#endif /* FORCE_H */
