/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * force.c
 *
 * setup and cleanup force library data structures
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "force/force.h"
#include "debug/debug.h"


int force_init(Force *f)
{
  memset(f, 0, sizeof(Force));
  return 0;
}


int32 **force_excl_list(const Force *f)
{
  return f->excl_list;
}


int32 **force_scaled14_list(const Force *f)
{
  return f->scaled14_list;
}


int force_setup(Force *f, ForceParam *p, ForceEnergy *e, ForceResult *r)
{
  if (p == NULL || e == NULL || r == NULL) return FORCE_FAIL;

  /* check validity of flags */
  if ((p->flags & FORCE_MASK_TYPE & ~(FORCE_ALL | FORCE_DIRECT))
      || (p->flags & FORCE_MASK_PERIOD & ~FORCE_PERIODIC)
      || (p->flags & FORCE_MASK_EXCL) > FORCE_EXCL_SCAL14
      || (p->flags & ~(FORCE_MASK_TYPE | FORCE_MASK_PERIOD | FORCE_MASK_BC
          | FORCE_MASK_EXCL | FORCE_MASK_CUTOFF))
      || ((p->flags & FORCE_SMOOTH) && (p->flags & FORCE_ELEC) == 0)
      || ((p->flags & FORCE_SWITCH) && (p->flags & FORCE_VDW) == 0)
      || ((p->flags & FORCE_ELEC_EXCL) && (p->flags & FORCE_ELEC) == 0)
      || ((p->flags & FORCE_VDW_EXCL) && (p->flags & FORCE_VDW) == 0)
      || ((p->flags & FORCE_ELEC_DIRECT) && (p->flags & FORCE_ELEC) == 0)
      || ((p->flags & FORCE_VDW_DIRECT) && (p->flags & FORCE_VDW) == 0)
      || ((p->flags & FORCE_ELEC_DIRECT) && (p->flags & FORCE_ELEC_EXCL))
      || ((p->flags & FORCE_VDW_DIRECT) && (p->flags & FORCE_VDW_EXCL))
      || ((p->flags & FORCE_ELEC_DIRECT) && (p->flags & FORCE_SMOOTH))
      || ((p->flags & FORCE_VDW_DIRECT) && (p->flags & FORCE_SWITCH))
      || ((p->flags & FORCE_MASK_BC)
        && (p->flags & FORCE_MASK_BC) != FORCE_SPHERE
        && (p->flags & FORCE_MASK_BC) != FORCE_X_CYLINDER
        && (p->flags & FORCE_MASK_BC) != FORCE_Y_CYLINDER
        && (p->flags & FORCE_MASK_BC) != FORCE_Z_CYLINDER)) {
    return FORCE_FAIL;
  }

  /* check for positive number of atoms */
  if (p->atom_len <= 0) return FORCE_FAIL;

  /* check that total force array is non-NULL */
  if (r->f == NULL) return FORCE_FAIL;

  /* check validity of nonbonded params */
  if (p->cutoff < 0.0 || p->elec_cutoff < 0.0 || p->vdw_cutoff < 0.0
      || p->switchdist < 0.0 || p->elec_const < 0.0 || p->dielectric < 0.0) {
    return FORCE_FAIL;
  }

  if ((p->flags & FORCE_ELEC) && p->elec_cutoff == 0.0) {
    if ((p->flags & FORCE_ELEC_DIRECT) == 0
        && (p->flags & FORCE_ELEC_EXCL) == 0) {
      if (p->cutoff == 0.0) {
        return FORCE_FAIL;
      }
      p->elec_cutoff = p->cutoff;
    }
  }
  else if (((p->flags & FORCE_ELEC_DIRECT) || (p->flags & FORCE_ELEC_EXCL))
      && p->elec_cutoff != 0.0) {
    return FORCE_FAIL;
  }

  if ((p->flags & FORCE_VDW) && p->vdw_cutoff == 0.0) {
    if ((p->flags & FORCE_VDW_DIRECT) == 0
        && (p->flags & FORCE_VDW_EXCL) == 0) {
      if (p->cutoff == 0.0) {
        return FORCE_FAIL;
      }
      p->vdw_cutoff = p->cutoff;
    }
  }
  else if (((p->flags & FORCE_VDW_DIRECT) || (p->flags & FORCE_VDW_EXCL))
      && p->vdw_cutoff != 0.0) {
    return FORCE_FAIL;
  }

  if ((p->flags & FORCE_VDW) && p->atomprm_len <= 0) {
    return FORCE_FAIL;
  }

  if (p->flags & FORCE_NONBONDED) {
    if (p->elec_cutoff > p->vdw_cutoff) p->cutoff = p->elec_cutoff;
    else p->cutoff = p->vdw_cutoff;
    ASSERT(p->cutoff > 0.0 
        || (p->flags & FORCE_DIRECT) == FORCE_DIRECT
        || (p->flags & FORCE_EXCL) == FORCE_EXCL
        || ((p->flags & FORCE_ELEC_DIRECT) && (p->flags & FORCE_VDW_EXCL))
        || ((p->flags & FORCE_VDW_DIRECT) && (p->flags & FORCE_ELEC_EXCL))
        || ((p->flags & FORCE_ELEC_DIRECT) && (p->flags & FORCE_VDW) == 0)
        || ((p->flags & FORCE_ELEC_EXCL) && (p->flags & FORCE_VDW) == 0)
        || ((p->flags & FORCE_VDW_DIRECT) && (p->flags & FORCE_ELEC) == 0)
        || ((p->flags & FORCE_VDW_EXCL) && (p->flags & FORCE_ELEC) == 0));
  }

  if ((p->flags & FORCE_ELEC) && 
      (p->elec_const <= 0.0 || p->dielectric < 1.0)) {
    return FORCE_FAIL;
  }

  if ((p->flags & (FORCE_VDW | FORCE_SWITCH)) == (FORCE_VDW | FORCE_SWITCH)
      && (p->switchdist == 0.0 || p->switchdist >= p->vdw_cutoff)) {
    return FORCE_FAIL;
  }

  /* check validity of boundary constraint constants */
  if (p->radius1 < 0.0 || p->radius2 < 0.0
      || p->length1 < 0.0 || p->length2 < 0.0
      || p->exp1 < 0 || p->exp2 < 0) {
    return FORCE_FAIL;
  }

  /* check validity of cell size and boundary conditions */
  if (p->xlen < 0.0 || p->ylen < 0.0 || p->zlen < 0.0) {
    return FORCE_FAIL;
  }
  if ((p->flags & FORCE_NONBONDED)
      && (((p->flags & FORCE_X_PERIODIC) && p->xlen <= p->cutoff)
        || ((p->flags & FORCE_Y_PERIODIC) && p->ylen <= p->cutoff)
        || ((p->flags & FORCE_Z_PERIODIC) && p->zlen <= p->cutoff))) {
    return FORCE_FAIL;
  }

  /* save pointers */
  f->param = p;
  f->energy = e;
  f->result = r;

  /* setup bonded data structures */
  if ((p->flags & FORCE_BONDED) && force_setup_bonded(f)) {
    return FORCE_FAIL;
  }

  /* setup nonbonded data structures */
  if ((p->flags & FORCE_NONBONDED) && force_setup_nonbonded(f)) {
    return FORCE_FAIL;
  }

  /* setup boundary constraints */
  if ((p->flags & FORCE_MASK_BC) && force_setup_boundary(f)) {
    return FORCE_FAIL;
  }
  return 0;
}


int force_compute(Force *f, const MD_Dvec *pos, MD_Dvec *wrap)
{
  ForceResult *r = f->result;
  ForceEnergy *e = f->energy;
  const int32 flags = f->param->flags;
  const int32 natoms = f->param->atom_len;
  size_t sz = natoms * sizeof(MD_Dvec);

  ASSERT(r->f != NULL);
  memset(r->f, 0, sz);
  if (r->f_bond) memset(r->f_bond, 0, sz);
  if (r->f_angle) memset(r->f_angle, 0, sz);
  if (r->f_dihed) memset(r->f_dihed, 0, sz);
  if (r->f_impr) memset(r->f_impr, 0, sz);
  if (r->f_elec) memset(r->f_elec, 0, sz);
  if (r->f_vdw) memset(r->f_vdw, 0, sz);
  if (r->f_bound) memset(r->f_bound, 0, sz);

  /* zero energy */
  memset(e, 0, sizeof(ForceEnergy));

  /* setup wrap array (or zero it) */
  if (wrap == NULL) {
    if (f->wrap == NULL) {
      f->wrap = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
      if (f->wrap == NULL) return FORCE_FAIL;
    }
    wrap = f->wrap;
  }
  else {
    memset(wrap, 0, natoms * sizeof(MD_Dvec));
  }

  /* compute bonded forces */
  if ((flags & FORCE_BONDED) && force_compute_bonded(f, pos)) {
    return FORCE_FAIL;
  }

  /* compute nonbonded forces */
  if ((flags & FORCE_NONBONDED) && force_compute_nonbonded(f, pos, wrap)) {
    return FORCE_FAIL;
  }

  /* compute boundary constraint forces */
  if ((flags & FORCE_MASK_BC) && force_compute_boundary(f, pos, wrap)) {
    return FORCE_FAIL;
  }

  /* sum potentials */
  e->pe = e->bond + e->angle + e->dihed + e->impr
    + e->elec + e->vdw + e->bound;

  return 0;
}


void force_done(Force *f)
{
  int32 natoms = 0;
  int32 k;

  free(f->vdwtable);
  free(f->wrap);
  free(f->next);
  free(f->cell);
  if (f->param != NULL) {
    natoms = f->param->atom_len;
  }
  if (f->excl_list != NULL) {
    for (k = 0;  k < natoms;  k++)  free(f->excl_list[k]);
    free(f->excl_list);
  }
  if (f->scaled14_list != NULL) {
    for (k = 0;  k < natoms;  k++)  free(f->scaled14_list[k]);
    free(f->scaled14_list);
  }
  if (f->exclx != NULL) {
    for (k = 0;  k < natoms;  k++) {
      free(f->exclx[k]);
    }
    free(f->exclx);
  }
  if (f->excl12 != NULL) {
    for (k = 0;  k < natoms;  k++)  free(f->excl12[k]);
    free(f->excl12);
  }
  if (f->excl13 != NULL) {
    for (k = 0;  k < natoms;  k++)  free(f->excl13[k]);
    free(f->excl13);
  }
  if (f->excl14 != NULL) {
    for (k = 0;  k < natoms;  k++)  free(f->excl14[k]);
    free(f->excl14);
  }
  if (f->scaled14 != NULL) {
    for (k = 0;  k < natoms;  k++)  free(f->scaled14[k]);
    free(f->scaled14);
  }
  free(f->lenx);
  free(f->len12);
  free(f->len13);
  free(f->accum);
  free(f->dest);
}
